package com.icetech.park.service.down.pnc.impl;

import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.cloudcenter.api.lcd.LcdService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.domain.entity.lcd.LcdBrightness;
import com.icetech.park.domain.entity.lcd.LcdConfig;
import com.icetech.park.domain.entity.lcd.LcdShow;
import com.icetech.park.domain.entity.lcd.LcdSound;
import com.icetech.park.domain.entity.lcd.LcdSoundcode;
import com.icetech.park.domain.entity.lcd.LcdTips;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.cloudcenter.domain.request.pnc.LcdConfigRequest;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.NumberUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;
import static com.icetech.cloudcenter.domain.request.pnc.LcdConfigRequest.BrightnessConfig;
import static com.icetech.cloudcenter.domain.request.pnc.LcdConfigRequest.DisplayConfig;
import static com.icetech.cloudcenter.domain.request.pnc.LcdConfigRequest.SoundCodeConfig;
import static com.icetech.cloudcenter.domain.request.pnc.LcdConfigRequest.SoundConfig;
import static com.icetech.cloudcenter.domain.request.pnc.LcdConfigRequest.TipsConfig;

/**
 * LCD自定义语音屏显下发
 */
@Slf4j
@Service
public class LcdsoundConfigServiceImpl extends AbstractService implements DownSendService {
    @Resource
    private PncDownHandle downHandle;
    @Resource
    private LcdService lcdService;
    @Resource
    private ParkService parkService;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private ParkRegionDao parkRegionDao;

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                String.valueOf(sendRequest.getParkId()), String.class);
        if(version == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getDefaultMessage(TextConstant.TWO,"找不到车场版本"));
        }
        if(PncVersionEnum.getIndex(version) < PncVersionEnum.版本5.getIndex()) {
            log.info("获取到车场[{}]版本: [{}] -> [{}]", sendRequest.getParkId(), version,
                    PncVersionEnum.getIndex(version));
            return ObjectResponse.failed(CodeConstants.ERROR_407, TextConstant.getVersionMessage(version, PncVersionEnum.版本5.getVersion()));
        }

        Long parkId = sendRequest.getParkId(), channelId = sendRequest.getServiceId();
        try {
            ObjectResponse<ParkInoutdevice> channelResp = parkService.getInoutDeviceById(channelId);
            ObjectResponse.notError(channelResp);
            ParkInoutdevice channel = channelResp.getData();
            LcdConfigRequest request = buildRequest(channel);
            List<DisplayConfig> displayConfigs = buildDisplayConfig(channel);
            request.setLcdShow(displayConfigs);
            List<TipsConfig> tipsConfigs = buildTipsConfig(channelId);
            request.setLcdTips(tipsConfigs);
            List<SoundCodeConfig> soundCodeConfigs = buildSoundCodeConfig();
            request.setLcdSoundcode(soundCodeConfigs);
            List<SoundConfig> soundConfigs = buildSoundConfig(channelId, version);
            request.setLcdSound(soundConfigs);

            String messageId = downHandle.signAndSend(parkId, DownServiceEnum.LCD语音屏显.getServiceName(), request, channelId);
            if (messageId == null) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
            }
            return ObjectResponse.success();
        } catch (ResponseBodyException e) {
            log.warn("下发LCD屏显语音失败: {}, {}", e.getErrCode(), e.getMessage(), e);
            return ObjectResponse.failed(ERROR_410,TextConstant.getDefaultMessage(TWO,"下发LCD屏显语音失败"));
        } catch (Exception e) {
            log.error("下发LCD屏显语音失败", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
        }
    }

    private LcdConfigRequest buildRequest(ParkInoutdevice channel) {
        ObjectResponse<LcdConfig> lcdConfigResp = lcdService.getLcdConfigByChannel(channel.getId());
        LcdConfig lcdConfig = lcdConfigResp.getData();
        if (lcdConfig == null) lcdConfig = lcdService.getLcdConfigByChannel(null).getData();

        LcdConfigRequest request = new LcdConfigRequest();
        request.setInandoutCode(channel.getInandoutCode());
        request.setLineNum(lcdConfig.getLineNum());
        request.setRemainDaysMc(lcdConfig.getRemainDaysMc());
        request.setExpireDaysMc(lcdConfig.getExpireDaysMc());
        request.setExpireDaysMonth(lcdConfig.getExpireDaysMonth());
        request.setBlackCar(lcdConfig.getBlackCar());
        request.setEntranceFreeLayout(lcdConfig.getEntranceFreeLayout());
        request.setEnterLayout(lcdConfig.getEnterLayout());
        request.setExitFreeLayout(lcdConfig.getExitFreeLayout());
        request.setExitLayout(lcdConfig.getExitLayout());
        request.setRestoreDefaultTime(lcdConfig.getRestoreDefaultTime());
        request.setVolumeValue(lcdConfig.getVolumeValue());
        request.setQuietHoursSwitch(lcdConfig.getQuietHoursSwitch());
        request.setQuietStartTime(lcdConfig.formatQuietStartTime());
        request.setQuietEndTime(lcdConfig.formatQuietEndTime());
        request.setQuietVolumeValue(lcdConfig.getQuietVolumeValue());
        request.setLimitType(lcdConfig.getLimitType());
        request.setLimitDriveNum1(lcdConfig.getLimitDriveNum1());
        request.setLimitDriveNum2(lcdConfig.getLimitDriveNum2());
        request.setLimitDriveNum3(lcdConfig.getLimitDriveNum3());
        request.setLimitDriveNum4(lcdConfig.getLimitDriveNum4());
        request.setLimitDriveNum5(lcdConfig.getLimitDriveNum5());
        request.setLcdBrightness(buildBrightnessConfig(lcdConfig.getLcdBrightnessList()));
        if (lcdConfig.getRegionFreeSpace() == null
                || Integer.valueOf(0).equals(lcdConfig.getRegionFreeSpace())) {
            request.setRegionFreeSpace(String.valueOf(0));
        } else {
            ParkRegion parkRegion = parkRegionDao.selectById(lcdConfig.getRegionFreeSpace());
            if (parkRegion != null) {
                request.setRegionFreeSpace(parkRegion.getRegionCode());
            } else {
                request.setRegionFreeSpace(String.valueOf(0));
            }
        }
        return request;
    }

    private List<BrightnessConfig> buildBrightnessConfig(List<LcdBrightness> brightnesses) {
        if (CollectionUtils.isEmpty(brightnesses)) {
            return null;
        }
        return brightnesses.stream().map(brightness -> {
            BrightnessConfig config = new BrightnessConfig();
            config.setStartTimePoint(brightness.getStartTimePoint());
            config.setEndTimePoint(brightness.getEndTimePoint());
            config.setBrightnessVal(brightness.getBrightnessVal());
            config.setSort(brightness.getSort());
            return config;
        }).collect(Collectors.toList());
    }

    private List<DisplayConfig> buildDisplayConfig(ParkInoutdevice channel) {
        Map<String, List<LcdShow>> lcdShowMap = lcdService.getLcdShowMapByChannel(channel.getId());
        if (lcdShowMap.isEmpty()) return null;
        int channelType = NumberUtils.toPrimitive(channel.getInandoutType());
        int otherChannelType = -1;
        String[] otherDisplayTypes = null;
        if (channelType == 1) {
            otherChannelType = 2;
            otherDisplayTypes = new String[]{String.valueOf(LcdShow.DisplayTypeEnum.出口空闲显示.type), String.valueOf(LcdShow.DisplayTypeEnum.出场显示.type)};
        } else {
            otherChannelType = 1;
            otherDisplayTypes = new String[]{String.valueOf(LcdShow.DisplayTypeEnum.入口空闲显示.type), String.valueOf(LcdShow.DisplayTypeEnum.入场显示.type)};
        }
        ObjectResponse<List<ParkInoutdevice>> otherChannels = parkService.getChannelByType(channel.getParkId(), otherChannelType);
        Long otherChannelId = Optional.ofNullable(otherChannels)
                .map(ObjectResponse::getData).filter(CollectionUtils::isNotEmpty)
                .map(list -> list.get(0)).map(ParkInoutdevice::getId).orElse(null);
        Map<String, List<LcdShow>> otherLcdMap = lcdService.getLcdShowMapByChannel(otherChannelId);
        for (int i = 0; i < otherDisplayTypes.length; i++) {
            String otherDisplayType = otherDisplayTypes[i];
            lcdShowMap.put(otherDisplayType, otherLcdMap.get(otherDisplayType));
        }

        return lcdShowMap.values().stream()
                .filter(Objects::nonNull).flatMap(Collection::stream)
                .filter(Objects::nonNull).map(show -> {
                    DisplayConfig config = new DisplayConfig();
                    config.setDisplayType(show.getDisplayType());
                    config.setRowNum(show.getRowNum());
                    config.setRowColor(show.getRowColor());
                    config.setDynamicContent(show.getDynamicContent());
                    config.setCustomContent(show.getCustomContent());
                    return config;
                }).collect(Collectors.toList());
    }

    private List<TipsConfig> buildTipsConfig(long channelId) {
        ObjectResponse<LcdTips> lcdTipsResp = lcdService.getLcdTipsByChannel(channelId);
        LcdTips lcdTips = lcdTipsResp.getData();
        if (lcdTips == null) lcdTips = lcdService.getLcdTipsByChannel(null).getData();

        TipsConfig config = new TipsConfig();
        config.setEnterTmpSpecial(lcdTips.getEnterTmpSpecial());
        config.setEnterMonthVip(lcdTips.getEnterMonthVip());
        config.setEnterNotAllowTmpRun(lcdTips.getEnterNotAllowTmpRun());
        config.setEnterMultiParkCar(lcdTips.getEnterMultiParkCar());
        config.setEnterBlackCar(lcdTips.getEnterBlackCar());
        config.setEnterMonthExpire(lcdTips.getEnterMonthExpire());
        config.setEnterNoParkingSpace(lcdTips.getEnterNoParkingSpace());
        config.setEnterNotAllowNocarRun(lcdTips.getEnterNotAllowNocarRun());
        config.setExitPay(lcdTips.getExitPay());
        config.setExitOpenGate(lcdTips.getExitOpenGate());
        config.setExitNoEnterinfo(lcdTips.getExitNoEnterinfo());
        config.setExitNoCarplate(lcdTips.getExitNoCarplate());
        config.setExitNotAllowRun(lcdTips.getExitNotAllowRun());
        return Collections.singletonList(config);
    }

    private List<SoundCodeConfig> buildSoundCodeConfig() {
        ObjectResponse<List<LcdSoundcode>> soundCodesResp = lcdService.getAllLcdSoundCodes();
        ObjectResponse.notError(soundCodesResp);
        return soundCodesResp.getData().stream().map(code -> {
            SoundCodeConfig config = new SoundCodeConfig();
            config.setSoundCode(code.getSoundCode());
            config.setContent(code.getContent());
            config.setType(code.getType());
            config.setIsCustom(code.getIsCustom());
            config.setCustomSoundPath(code.getCustomSoundPath());
            return config;
        }).collect(Collectors.toList());
    }

    private List<SoundConfig> buildSoundConfig(long channelId, String version) {
        Map<String, List<LcdSound>> lcdSoundMap = lcdService.getLcdSoundMapByChannel(channelId);
        if (MapUtils.isEmpty(lcdSoundMap)) {
            lcdSoundMap = lcdService.getLcdSoundMapByChannel(null);
        }

        return lcdSoundMap.values().stream()
                .filter(Objects::nonNull).flatMap(Collection::stream)
                .filter(lcdSound -> PncVersionEnum.getIndex(version) >= PncVersionEnum.版本2500.getIndex()
                        || (LcdSound.SoundTypeEnum.入口欠费补缴.type != lcdSound.getSoundType()
                        && LcdSound.SoundTypeEnum.出口欠费补缴.type != lcdSound.getSoundType()))
                .filter(Objects::nonNull).map(sound -> {
                    SoundConfig config = new SoundConfig();
                    config.setSoundType(sound.getSoundType());
                    config.setContentPattern(sound.getContentPattern());
                    return config;
                }).collect(Collectors.toList());
    }
}
