package com.icetech.park.service.down.pnc.impl;

import com.icetech.cloudcenter.api.DownSendService;
import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.common.utils.NumberUtils;
import com.icetech.fee.dao.monthcar.MonthProductDao;
import com.icetech.fee.dao.monthcar.MonthRecordDao;
import com.icetech.fee.dao.monthcar.MonthRelSpaceDao;
import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.fee.domain.entity.monthcar.MonthProduct;
import com.icetech.fee.domain.entity.monthcar.MonthRecord;
import com.icetech.fee.domain.entity.monthcar.MonthRelSpace;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.request.pnc.BatchDelBizRequest;
import com.icetech.cloudcenter.domain.request.pnc.CardStatusRequest;
import com.icetech.cloudcenter.domain.request.pnc.IssuedCardRequest;
import com.icetech.cloudcenter.domain.request.pnc.MonthPlot;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;

/**
 * 月卡信息下发接口
 *
 * @author fangct
 */
@Service
public class IssuedCardServiceImpl implements DownSendService {
    @Autowired
    private PncDownHandle downHandle;
    @Autowired
    private MonthRecordDao monthRecordDao;
    @Autowired
    private MonthProductDao monthProductDao;
    @Autowired
    private MonthRelSpaceDao monthRelSpaceDao;
    @Autowired
    private CardPauseRecoverServiceImpl cardPauseRecoverService;
    @Autowired
    private ParkRegionDao parkRegionDao;
    @Autowired
    private RedisUtils redisUtils;

    /**
     * 分时段卡
     */
    private static final int CARD_TYPE_2 = 2;
    private static final int PAUSE = 7;
    private static final int RECOVER = 8;

    @Override
    public ObjectResponse send(SendRequest sendRequest) {
        Long serviceId = sendRequest.getServiceId();

        MonthRecord monthRecord = monthRecordDao.selectById(serviceId);
        if (monthRecord == null){
            return ObjectResponse.failed(ERROR_410,TextConstant.getDefaultMessage(TextConstant.TWO,"月卡操作记录不存在"));
        }
        String version = redisUtils.hGet(RedisConstants.PNC_VERSION,
                String.valueOf(sendRequest.getParkId()), String.class);
        boolean monthPlot = "spaceTime".equals(sendRequest.getInclude());
        String messageId;
        if (version == null || version.startsWith("N")){
            if (PAUSE == monthRecord.getCardOpertype() || RECOVER == monthRecord.getCardOpertype()){
                return cardPauseRecoverService.send(serviceId, monthRecord);
            }
            IssuedCardRequest issuedCardRequest = buildRequest(monthRecord, monthPlot);
            String regionId = monthRecord.getRegionId();
            if (StringUtils.isNotEmpty(regionId) && !regionId.equals("0")) {
                issuedCardRequest.setMcRange(2);
                Map<String, String> parkRegion = getParkRegion(regionId);
                issuedCardRequest.setRegionCode(parkRegion.get("regionCodes"));
                issuedCardRequest.setRegionName(parkRegion.get("regionNames"));
            }
            messageId = downHandle.signAndSend(sendRequest.getParkId(), DownServiceEnum.月卡.getServiceName(),
                    issuedCardRequest, serviceId);
        }else{
            CardStatusRequest cardStatusRequest = buildNewRequest(monthRecord, monthPlot);
            messageId = downHandle.signAndSend(sendRequest.getParkId(), DownServiceEnum.月卡.getServiceName(),
                    cardStatusRequest, serviceId);
        }
        if (StringUtils.isEmpty(messageId)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getUnEstablishConnection());
        }
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode){
        List<Long> ids = sendRequestList.stream().map(SendRequest::getServiceId).collect(Collectors.toList());
        List<MonthRecord> list = monthRecordDao.selectBatchIds(ids);
        if (CollectionUtils.isEmpty(list)) {
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }
        List<String> monthIdList = list.stream().map(MonthRecord :: getMonthId)
                .map(String :: valueOf).collect(Collectors.toList());
        BatchDelBizRequest batchDelBizRequest = new BatchDelBizRequest();
        batchDelBizRequest.setServiceType(2);
        batchDelBizRequest.setIdList(monthIdList);
        batchDelBizRequest.setServiceIds(ids);
        String messageId = downHandle.signAndSend(parkId, DownServiceEnum.批量删除业务数据.getServiceName(), batchDelBizRequest);
        if (StringUtils.isEmpty(messageId)) {
            throw new ResponseBodyException(CodeConstants.ERROR, "下发失败");
        }
        return ObjectResponse.success();
    }
    /**
     * 获取区域编号
     * @param regionIds
     * @return
     */
    private Map<String, String> getParkRegion(String regionIds){
        Map<String, String> retMap = new HashMap<>();
        String regionCodes = "";
        String regionNames = "";
        String[] split = regionIds.split(",");
        for (int i = 0; i < split.length; i++) {
            ParkRegion parkRegion = parkRegionDao.selectById(Long.valueOf(split[i]));
            if(i == split.length-1){
                regionCodes = regionCodes + parkRegion.getRegionCode();
                regionNames = regionNames + parkRegion.getRegionName();
            }else{
                regionCodes = regionCodes + parkRegion.getRegionCode() + ",";
                regionNames = regionNames + parkRegion.getRegionName() + ",";
            }
        }
        retMap.put("regionCodes", regionCodes);
        retMap.put("regionNames", regionNames);
        return retMap;
    }

    /**
     * 构建请求参数
     *
     * @param monthRecord
     * @return
     */
    private IssuedCardRequest buildRequest(MonthRecord monthRecord, boolean monthPlot) {
        IssuedCardRequest issuedCardRequest = new IssuedCardRequest();
        issuedCardRequest.setCardId(String.valueOf(monthRecord.getMonthId()));
        if (monthRecord.getPlateNum() != null){
            issuedCardRequest.setPlateNum(monthRecord.getPlateNum().replaceAll(",", "/"));
        }
        issuedCardRequest.setPhone(monthRecord.getPhone());
        issuedCardRequest.setCardOwner(StringUtils.isBlank(monthRecord.getCardOwner()) ? "default" : monthRecord.getCardOwner());
        issuedCardRequest.setCardProperty(monthRecord.getCardProperty());
        if (monthRecord.getStartTime() == null || monthRecord.getEndTime() == null) {
            throw new ResponseBodyException(ERROR_410, TextConstant.getDefaultMessage(TWO, "日期不合法"));
        }
        issuedCardRequest.setStartDate(DateFormatUtils.format(monthRecord.getStartTime(), "yyyy-MM-dd"));
        issuedCardRequest.setEndDate(DateFormatUtils.format(monthRecord.getEndTime(), "yyyy-MM-dd"));
        issuedCardRequest.setCardOperType(monthRecord.getCardOpertype());
        issuedCardRequest.setPlotCount(monthRecord.getPlotCount());
        if (monthRecord.getSpaceNum() != null){
            issuedCardRequest.setPlotNum(monthRecord.getSpaceNum().replaceAll(",", "/"));
        }
        //Todo 退款金额
        issuedCardRequest.setPayMoney(monthRecord.getPayMoney() == null ? "0.0" : monthRecord.getPayMoney().toString());
        issuedCardRequest.setOperAccount(monthRecord.getOperAccount());
        issuedCardRequest.setOperTime(monthRecord.getCreateTime().getTime() / 1000);

        MonthProduct monthProduct = monthProductDao.selectById(monthRecord.getProductId());
        issuedCardRequest.setProdName(monthProduct.getName());
        issuedCardRequest.setProdDuration(monthProduct.getDuration());
        int cardType = monthProduct.getCardType();
        issuedCardRequest.setCardType(cardType);
        if (cardType == CARD_TYPE_2){
            issuedCardRequest.setStartTime(DateFormatUtils.format(monthProduct.getStartTime(), "HH:mm:ss"));
            issuedCardRequest.setEndTime(DateFormatUtils.format(monthProduct.getEndTime(), "HH:mm:ss"));
        }

        if (monthPlot)
            issuedCardRequest.setPlotValidPeriod(buildMonthPlot(monthRecord.getMonthId()));
        issuedCardRequest.setNotWorkdayCharge(NumberUtils.toPrimitive(monthProduct.getNotWorkdayCharge(), 2));
        return issuedCardRequest;
    }

    /**
     * 构建请求参数
     *
     * @param monthRecord
     * @return
     */
    private CardStatusRequest buildNewRequest(MonthRecord monthRecord, boolean monthPlot) {
        CardStatusRequest cardStatusRequest = new CardStatusRequest();
        Integer cardOpertype = monthRecord.getCardOpertype();
        if (cardOpertype == 1 || cardOpertype == 2 || cardOpertype == 3 || cardOpertype == 6 || cardOpertype == 8){
            cardStatusRequest.setCardStatus(1);
        }else{
            cardStatusRequest.setCardStatus(2);
        }
        cardStatusRequest.setCardId(String.valueOf(monthRecord.getMonthId()));
        if (monthRecord.getPlateNum() != null){
            cardStatusRequest.setPlateNum(monthRecord.getPlateNum().replaceAll(",", "/"));
        }
        cardStatusRequest.setPhone(monthRecord.getPhone());
        cardStatusRequest.setCardOwner(StringUtils.isBlank(monthRecord.getCardOwner()) ? "default" : monthRecord.getCardOwner());
        cardStatusRequest.setCardProperty(monthRecord.getCardProperty());
        cardStatusRequest.setStartDate(DateFormatUtils.format(monthRecord.getStartTime(), "yyyy-MM-dd"));
        cardStatusRequest.setEndDate(DateFormatUtils.format(monthRecord.getEndTime(), "yyyy-MM-dd"));
        cardStatusRequest.setPlotCount(monthRecord.getPlotCount());
        cardStatusRequest.setPlotNum(monthRecord.getSpaceNum());
        cardStatusRequest.setPayMoney(monthRecord.getPayMoney().toString());
        cardStatusRequest.setOperAccount(monthRecord.getOperAccount());
        cardStatusRequest.setOperTime(monthRecord.getCreateTime().getTime() / 1000);

        MonthProduct monthProduct = monthProductDao.selectById(monthRecord.getProductId());
        int cardType = monthProduct.getCardType();
        cardStatusRequest.setCardType(cardType);
        if (cardType == CARD_TYPE_2){
            cardStatusRequest.setStartTime(DateFormatUtils.format(monthProduct.getStartTime(), "HH:mm:ss"));
            cardStatusRequest.setEndTime(DateFormatUtils.format(monthProduct.getEndTime(), "HH:mm:ss"));
        }
        String regionId = monthRecord.getRegionId();
        if(StringUtils.isNotEmpty(regionId) && !regionId.equals("0")){
            cardStatusRequest.setMcRange(2);
            Map<String, String> parkRegion = getParkRegion(regionId);
            cardStatusRequest.setRegionCode(parkRegion.get("regionCodes"));
            cardStatusRequest.setRegionName(parkRegion.get("regionNames"));
        }

        if (monthPlot)
            cardStatusRequest.setPlotValidPeriod(buildMonthPlot(monthRecord.getMonthId()));
        return cardStatusRequest;
    }

    @Nullable
    private List<MonthPlot> buildMonthPlot(long monthId) {
        List<MonthRelSpace> monthRelSpaces = monthRelSpaceDao.getByMonthId(monthId);
        return CollectionUtils.isEmpty(monthRelSpaces) ? null :
                monthRelSpaces.stream().map(monthRelSpace ->
                                new MonthPlot()
                                        .setPlotNum(monthRelSpace.getSpaceNum())
                                        .setStartDate(monthRelSpace.getStartTime() == null ? null : monthRelSpace.getStartTime().format(DateUtils.DEFAULT_DATE_FORMAT))
                                        .setEndDate(monthRelSpace.getEndTime() == null ? null : monthRelSpace.getEndTime().format(DateUtils.DEFAULT_DATE_FORMAT)))
                        .collect(Collectors.toList());
    }
}
