package com.icetech.park.service.down.pnc.impl;

import com.icetech.basics.constants.TextConstant;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.fee.dao.monthcar.MonthRecordDao;
import com.icetech.basics.domain.SendMessage;
import com.icetech.third.utils.JsonUtils;
import com.icetech.common.utils.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.icetech.fee.domain.entity.monthcar.MonthRecord;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.request.pnc.CardPauseRecoverRequest;
import com.icetech.park.service.handle.PncDownHandle;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;

import lombok.extern.slf4j.Slf4j;

import java.util.Collections;

import static com.icetech.basics.constants.TextConstant.*;

/**
 * 月卡暂停/恢复下发接口
 *
 * @author fangct
 */
@Service
@Slf4j
public class CardPauseRecoverServiceImpl {

    @Autowired
    private PncDownHandle downHandle;
    @Autowired
    private MonthRecordDao monthRecordDao;
    /**
     * 月卡有效期开始前多久下发，5分钟
     */
    private static final int BEFORE_TIME = 5 * 60;
    private static final int RECOVER = 8;

    public ObjectResponse send(Long serviceId, MonthRecord monthRecord) {
        /**
         * 如果当前时间未到月卡有效期，则先不下发
         */
        SendMessage sendMessage = SendMessage.builder()
                .type(TWO)
                .build();
        if (monthRecord.getCardstopStart() == null) {
            throw new ResponseBodyException(ERROR_410, TextConstant.getDefaultMessage(ONE, "月卡暂停实际开始时间不能为空"));
        }
        long pauseStartTime = monthRecord.getCardstopStart().getTime() / 1000;
        if (pauseStartTime > DateTools.unixTimestamp()){
            log.info("<端网云-月卡暂停恢复下发> 未到暂停开始时间，延迟下发，参数:{}", monthRecord);
            Integer fireTime = (int)pauseStartTime - BEFORE_TIME;
            sendMessage.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                    .failType(DEFAULT)
                    .fixedDisplay("月卡未到暂停开始时间")
                    .build()));
            return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(sendMessage), fireTime);
        }
        CardPauseRecoverRequest cardPauseRecoverRequest = buildRequest(monthRecord);
        String messageId = downHandle.signAndSend(monthRecord.getParkId(), DownServiceEnum.月卡暂停恢复.getServiceName(),
                cardPauseRecoverRequest, serviceId);
        sendMessage.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                .failType(UN_ESTABLISH_CONNECTION)
                .build()));
        if (StringUtils.isEmpty(messageId)) {
            return ObjectResponse.failed(ERROR_410, JsonUtils.toJson(sendMessage));
        }
        return ObjectResponse.success();
    }
    public ObjectResponse send(SendRequest sendRequest) {
        Long serviceId = sendRequest.getServiceId();
        MonthRecord monthRecord = monthRecordDao.selectById(serviceId);
        return send(serviceId, monthRecord);
    }

    /**
     * 构建请求参数
     *
     * @param monthRecord
     * @return
     */
    private CardPauseRecoverRequest buildRequest(MonthRecord monthRecord) {
        CardPauseRecoverRequest cardPauseRecoverRequest = new CardPauseRecoverRequest();
        cardPauseRecoverRequest.setCardId(String.valueOf(monthRecord.getMonthId()));
        cardPauseRecoverRequest.setCardOperType(monthRecord.getCardOpertype());
        cardPauseRecoverRequest.setStartDate(DateFormatUtils.format(monthRecord.getCardstopStart(), "yyyy-MM-dd"));
        cardPauseRecoverRequest.setEndDate(DateFormatUtils.format(monthRecord.getCardstopEnd(), "yyyy-MM-dd"));
        if (monthRecord.getCardOpertype() == RECOVER){
            cardPauseRecoverRequest.setMonthEndDate(DateFormatUtils.format(monthRecord.getEndTime(), "yyyy-MM-dd"));
            cardPauseRecoverRequest.setEndDate(DateFormatUtils.format(monthRecord.getCardstopRealyend(), "yyyy-MM-dd"));
        }
        cardPauseRecoverRequest.setOperAccount(monthRecord.getOperAccount());
        cardPauseRecoverRequest.setOperTime(monthRecord.getCreateTime().getTime() / 1000);

        return cardPauseRecoverRequest;
    }
}
