package com.icetech.park.service.down.p2c.impl;

import cn.hutool.core.thread.ThreadUtil;
import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.park.ParkVisitService;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.p2c.ClearListRequest;
import com.icetech.cloudcenter.domain.request.p2c.QueryListRequest;
import com.icetech.cloudcenter.domain.request.p2c.VisitCardRequest;
import com.icetech.cloudcenter.domain.response.ParkVisitDto;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.park.dao.park.ParkVisitDao;
import com.icetech.park.domain.entity.park.ParkVisit;
import com.icetech.park.service.AbstractConfigDownService;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.TWO;

@Service("p2cVisitCardServiceImpl")
@Slf4j
public class VisitCardServiceImpl extends AbstractConfigDownService implements DownService<List<VisitCardRequest>, Void>, ResponseService<String> {

    @Resource
    private P2cDownHandle p2cDownHandle;
    @Resource
    private SendMsgServiceImpl sendMsgService;
    @Resource
    private ParkVisitService parkVisitService;
    @Resource
    private ParkService parkService;
    @Resource
    private ParkVisitDao parkVisitDao;

    @Override
    public ObjectResponse<Void> send(SendRequest sendRequest, String parkCode) {
        if (StringUtils.isNotEmpty(sendRequest.getInclude()) && TWO.equals(sendRequest.getInclude())) {
            return batchSend(Collections.singletonList(sendRequest), null, parkCode);
        }
        try {
            ObjectResponse<ParkVisitDto> objectResponse = parkVisitService.selectVisitByid(sendRequest.getServiceId());
            if(!ObjectResponse.isSuccess(objectResponse)) {
                return ObjectResponse.failed(objectResponse.getCode(), objectResponse.getMsg());
            }
            ParkVisitDto parkVisitDto = objectResponse.getData();
            String[] split = parkVisitDto.getPlateNums().split(",");
            ObjectResponse<ParkConfig> parkConfig = parkService.getParkConfig(parkVisitDto.getParkCode());
            //是否收费
            Integer visitIscharge = 0;
            if (ObjectResponse.isSuccess(parkConfig)) {
                visitIscharge = parkConfig.getData().getVisitIscharge();
            }
            for (String s : split) {
                VisitCardRequest visitCardRequest = new VisitCardRequest();
                visitCardRequest.setVisitId(parkVisitDto.getId());
                visitCardRequest.setIsCharge(visitIscharge);
                visitCardRequest.setIsInoutMore(parkVisitDto.getInoutMore());
                visitCardRequest.setPlateNum(s);
                visitCardRequest.setStartTime(parkVisitDto.getStartTime());
                visitCardRequest.setEndTime(parkVisitDto.getEndTime());
                List<VisitCardRequest> list = new ArrayList<>();
                list.add(visitCardRequest);
                return sendMsgService.send2Park(sendRequest, parkCode, list);
            }

        }catch (ResponseBodyException e){
            log.warn(String.valueOf(e.getMessage()), e);
            return ObjectResponse.failed(e.getErrCode(), TextConstant.getDefaultMessage(TextConstant.ONE, e.getMessage() == null ? "访客车辆下发失败" : e.getMessage()));
        }catch (Exception e) {
            log.warn(String.valueOf(e.getMessage()), e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "访客车辆下发失败"));
        }
        return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "访客车辆下发失败"));

    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.访客车辆.getCmdType());
    }

    @Override
    public ObjectResponse<Void> batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode) {
        List<Long> ids = sendRequestList.stream()
                .map(SendRequest::getServiceId)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        List<ParkVisit> lists = parkVisitDao.findAllByIds(ids,Boolean.TRUE);
        if (CollectionUtils.isEmpty(lists)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "删除访客车辆信息不存在"));
        }
        Map<Long, ParkVisit> map = lists.stream()
                .collect(Collectors.toMap(ParkVisit::getId, Function.identity(), (key1, key2) -> key1));
        for (SendRequest sendRequest : sendRequestList) {
            ParkVisit parkVisit = map.get(sendRequest.getServiceId());
            if (parkVisit == null) {
                continue;
            }
            ClearListRequest request = ClearListRequest.builder()
                    .id(parkVisit.getId().intValue())
                    .dataType(QueryListRequest.DataType.访客车辆.getVal())
                    .build();
            sendRequest.setReqServiceType(P2cDownCmdEnum.名单数据清空.getCmdType());
            sendMsgService.send2Park(sendRequest, parkCode, request);
            ThreadUtil.sleep(200);
        }
        return ObjectResponse.success();
    }
}
