package com.icetech.park.service.down.p2c.impl;

import cn.hutool.core.util.StrUtil;
import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.entity.park.ParkChargeconfig;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.api.fee.ChargeService;
import com.icetech.cloudcenter.api.month.VipCarService;
import com.icetech.cloudcenter.api.month.VipRegionService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.p2c.VipTypeRequest;
import com.icetech.cloudcenter.domain.response.VipTypeDto;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.ToolsUtil;
import com.icetech.fee.domain.entity.vip.VipRegion;
import com.icetech.park.service.AbstractConfigDownService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.ONE;

@Service("p2cVipTypeServiceImpl")
@Slf4j
public class VipTypeServiceImpl extends AbstractConfigDownService implements DownService<List<VipTypeRequest>, Void>, ResponseService<String> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private SendMsgServiceImpl sendMsgService;
    @Resource
    private VipCarService vipCarService;
    @Resource
    private VipRegionService vipRegionService;
    @Resource
    private ChargeService chargeService;
    @Resource
    private ParkService parkService;

    @Override
    public ObjectResponse<Void> send(SendRequest sendRequest, String parkCode) {
        Long vipTypeId = sendRequest.getServiceId();
        int type = Integer.parseInt(sendRequest.getInclude());
        VipTypeRequest request = new VipTypeRequest();
        //删除操作
        if (type == 3) {
            request.setVipTypeId(vipTypeId.intValue());
        }else{
            ObjectResponse<VipTypeDto> vipTypeResp = vipCarService.getVipTypeById(vipTypeId.intValue());
            if (!ObjectResponse.isSuccess(vipTypeResp)) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(ONE, "vip类未找到"));
            }
            ObjectResponse.notError(vipTypeResp);
            request = buildRequest(vipTypeResp.getData());
        }
        request.setOperType(type);
        List<VipTypeRequest> list = new ArrayList<>();
        list.add(request);
        return sendMsgService.send2Park(sendRequest, parkCode, list);

    }

    private VipTypeRequest buildRequest(VipTypeDto vipType) {
        VipTypeRequest request = new VipTypeRequest();
        request.setVipTypeId(vipType.getId());
        request.setVipTypeName(vipType.getName());
        request.setDiscountType(vipType.getType());
        Integer discountType = vipType.getType();
        if (discountType != null && discountType == 3) {
            request.setDiscountNumber((int) (Float.parseFloat(vipType.getAmount()) * 100));
        } else if (discountType != null && discountType == 4) {
            request.setDiscountNumber((int) (Float.parseFloat(vipType.getAmount()) * 10));
        } else if (discountType != null && discountType == 2) {
            request.setDiscountNumber(ToolsUtil.parseInt(vipType.getAmount()));
        } else if (discountType != null && discountType == 1) {
            request.setDiscountNumber(-1);
        }
        request.setBilltypecode(vipType.getBilltypecode() == null ? "" : vipType.getBilltypecode());
        List<VipRegion> vipRegionList = vipRegionService.getVipRegionsListByVipTypeId(vipType.getId());
        if (CollectionUtils.isNotEmpty(vipRegionList)) {
            VipRegion vipRegion = vipRegionList.stream()
                    .filter(v -> v.getRegionId() == 0)
                    .findFirst().orElse(null);

            // 不是全车场
            if (vipRegion == null) {
                List<VipTypeRequest.BillInfo> billInfos = vipRegionList.stream()
                        .filter(Objects::nonNull)
                        .map(v -> {
                            ObjectResponse<ParkRegion> regionObjectResponse = parkService.getParkRegionById(v.getRegionId());
                            if (!ObjectResponse.isSuccess(regionObjectResponse)) {
                                return null;
                            }
                            return VipTypeRequest.BillInfo.builder()
                                .regionId(v.getRegionId())
                                .billtypecode(getParkChargeConfigCode(v.getBillId()))
                                .build();
                        }).collect(Collectors.toList());
                request.setBillInfo(billInfos);
            } else {
                List<VipTypeRequest.BillInfo> billInfos = new ArrayList<>();
                VipTypeRequest.BillInfo billInfo = new VipTypeRequest.BillInfo();
                billInfo.setRegionId(vipRegion.getRegionId());
                billInfo.setBilltypecode(vipType.getBilltypecode() == null ? "" : vipType.getBilltypecode());
                billInfos.add(billInfo);
                request.setBillInfo(billInfos);
            }
        }
        return request;
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.VIP车辆类型.getCmdType());
    }

    public boolean send(Long parkId, String parkCode, String serialNumber, String version, String filter) {
        List<VipTypeRequest> list = new ArrayList<>();
        if (StrUtil.isNotBlank(filter)){
            ObjectResponse<VipTypeDto> objectResponse = vipCarService.getVipTypeById(Integer.parseInt(filter));
            ObjectResponse.notError(objectResponse);
            VipTypeRequest request = buildRequest(objectResponse.getData());
            request.setOperType(1);
            list.add(request);
        }else{
            ObjectResponse<List<VipTypeDto>> listObjectResponse = vipCarService.getVipTypeByParkId(parkId);
            ObjectResponse.notError(listObjectResponse);
            listObjectResponse.getData().forEach(vipTypeDto ->{
                VipTypeRequest request = buildRequest(vipTypeDto);
                request.setOperType(1);
                list.add(request);
            });
        }
        Message<List<VipTypeRequest>> message = new Message<>(parkId, P2cDownCmdEnum.VIP车辆类型.getCmdType(), list);
        String messageId = p2cDownHandle.send(parkCode, serialNumber, message);
        if (messageId == null){
            log.info("[端云-VIP车辆类型下发] 下发失败，参数[{}]", message);
            return false;
        }else{
            return true;
        }
    }

    public String getParkChargeConfigCode(Integer id) {
        ParkChargeconfig chargeConfig = chargeService.getParkChargeConfigById(id);
        if (chargeConfig == null || StringUtils.isEmpty(chargeConfig.getBilltypecode())) {
            return "";
        }
        return chargeConfig.getBilltypecode();
    }
}
