package com.icetech.park.service.down.p2c.impl;

import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.StrUtil;
import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.cloudcenter.api.month.VipCarService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.request.p2c.ClearListRequest;
import com.icetech.cloudcenter.domain.request.p2c.MonthCardRequest;
import com.icetech.cloudcenter.domain.request.p2c.QueryListRequest;
import com.icetech.cloudcenter.domain.request.p2c.VipInfoRequest;
import com.icetech.cloudcenter.domain.response.VipInfoDto;
import com.icetech.cloudcenter.domain.response.VipTypeDto;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.BatchSendRepeatVO;
import com.icetech.cloudcenter.domain.vo.BatchSendVO;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.StringUtils;
import com.icetech.common.utils.ToolsUtil;
import com.icetech.fee.dao.vip.VipRecordDao;
import com.icetech.fee.domain.entity.monthcar.MonthRecord;
import com.icetech.fee.domain.entity.vip.VipRecord;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.BatchDownBizHandle;
import com.icetech.park.service.handle.P2cDownHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.ONE;

@Service("p2cVipInfoServiceImpl")
@Slf4j
public class VipInfoServiceImpl extends AbstractService implements DownService<List<VipInfoRequest>, Void>, ResponseService<String> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private VipCarService vipCarService;
    @Autowired
    private VipRecordDao vipRecordDao;
    @Autowired
    private SendMsgServiceImpl sendMsgService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private BatchDownBizHandle<VipInfoRequest> batchDownBizHandle;
    //VIP数据头部
    private static final String[] HEADERS = {"id", "vipId", "vipTypeName", "plateNum", "phone", "startDate", "endDate", "discountType", "discountNumber", "billtypecode", "vipTypeId"};

    @Override
    public ObjectResponse<Void> send(SendRequest sendRequest, String parkCode) {
        Long serviceId = sendRequest.getServiceId();
        ObjectResponse<VipRecord> recordObjectResponse = vipCarService.getOperatorRecordById(serviceId);
        if (!ObjectResponse.isSuccess(recordObjectResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(ONE, "vip操作记录不存在"));
        }
        VipRecord vipRecord = recordObjectResponse.getData();
        ObjectResponse<VipTypeDto> vipTypeObjectResponse = vipCarService.getVipTypeById(vipRecord.getTypeId());
        if (!ObjectResponse.isSuccess(vipTypeObjectResponse)) {
            return ObjectResponse.failed(ERROR_410, "未找到类型信息");
        }
        VipTypeDto vipType = vipTypeObjectResponse.getData();
        Integer operType = vipRecord.getOperType();
        if (operType == 3) {//非删除操作
            ClearListRequest clearListRequest = new ClearListRequest();
            clearListRequest.setDataType(QueryListRequest.DataType.VIP.getVal());
            clearListRequest.setId(vipRecord.getVipId());
            sendRequest.setReqServiceType(P2cDownCmdEnum.名单数据清空.getCmdType());
            return sendMsgService.send2Park(sendRequest, parkCode, clearListRequest);
        }
        if (vipRecord.getStartTime() == null || vipRecord.getEndTime() == null) {
            throw new ResponseBodyException(ERROR_410, TextConstant.getDefaultMessage(ONE, "日期不合法"));
        }
        List<VipInfoRequest> list = new ArrayList<>();
        VipInfoRequest vipInfoRequest = new VipInfoRequest();
        vipInfoRequest.setPlateNum(vipRecord.getPlateNum().replaceAll("、", ","));
        vipInfoRequest.setVipId((long) vipRecord.getVipId());
        vipInfoRequest.setVipTypeId((long) vipRecord.getTypeId());
        vipInfoRequest.setPhone(StringUtils.isBlank(vipRecord.getPhone()) ? "123" : vipRecord.getPhone());
        vipInfoRequest.setStartDate(DateTools.getFormat(vipRecord.getStartTime()));
        vipInfoRequest.setEndDate(DateTools.getFormat(DateTools.DF_, vipRecord.getEndTime()) + " 23:59:59");
        vipInfoRequest.setDiscountType(vipType.getType());
        Integer discountType = vipType.getType();
        if (discountType != null && discountType == 3) {
            vipInfoRequest.setDiscountNumber((int) (Float.parseFloat(vipType.getAmount()) * 100));
        } else if (discountType != null && discountType == 4) {
            vipInfoRequest.setDiscountNumber((int) (Float.parseFloat(vipType.getAmount()) * 10));
        } else if (discountType != null && discountType == 2) {
            vipInfoRequest.setDiscountNumber(ToolsUtil.parseInt(vipType.getAmount()));
        } else if (discountType != null && discountType == 1) {
            vipInfoRequest.setDiscountNumber(-1);
        }
        vipInfoRequest.setVipTypeName(vipType.getName());
        vipInfoRequest.setBilltypecode(vipType.getBilltypecode());
        list.add(vipInfoRequest);
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(sendRequest.getParkId());
        Park park = parkObjectResponse.getData();

        //场中场车场，只下发到入口，避免大场出口相机直接抬杆
        if (park.getIsInterior() != null && park.getIsInterior() == 1) {
            ObjectResponse<List<ParkDevice>> listObjectResponse = parkDeviceService.getDeviceListByType(sendRequest.getParkId(), 1);
            ObjectResponse.notError(listObjectResponse, "车场ID[" + sendRequest.getParkId() + "]下未添加设备");
            List<ParkDevice> parkDevices = listObjectResponse.getData().stream()
                    .filter(pd -> P2cVersionEnum.版本17.getIndex() > P2cVersionEnum.getIndex(pd.getProtocolVer()))
                    .collect(Collectors.toList());
            //旧版本处理
            if (!CollectionUtils.isEmpty(parkDevices)) {
                ObjectResponse<List<ParkRegion>> objectResponse = parkService.selectRegionById(sendRequest.getParkId());
                if (ObjectResponse.isSuccess(objectResponse)) {
                    List<ParkRegion> parkRegions = objectResponse.getData();
                    for (ParkRegion parkRegion : parkRegions) {
                        ObjectResponse<List<ParkDevice>> deviceListObjectResponse = parkDeviceService.getDeviceListByParkRegionId(sendRequest.getParkId(), parkRegion.getId().intValue(), 1);
                        if (ObjectResponse.isSuccess(deviceListObjectResponse)) {
                            List<ParkDevice> parkDeviceList = deviceListObjectResponse.getData();
                            for (ParkDevice parkDevice : parkDeviceList) {
                                Integer channelId = parkDevice.getChannelId();
                                if (channelId != null) {
                                    /*
                                     * 不给大场出口下发，小场要收费，防止相机直接开闸
                                     */
                                    ObjectResponse<ParkInoutdevice> inoutdeviceObjectResponse = parkService.getInoutDeviceById(parkDevice.getChannelId().longValue());
                                    if (ObjectResponse.isSuccess(inoutdeviceObjectResponse)) {
                                        ParkInoutdevice parkInoutdevice = inoutdeviceObjectResponse.getData();
                                        //大场出口
                                        if (parkInoutdevice.getInandoutType() == 2 && parkInoutdevice.getIsMaster() == 1) {
                                            continue;
                                        }
                                    }
                                }
                                String serialNumber = parkDevice.getSerialNumber();
                                Message<List<VipInfoRequest>> message = new Message<>(sendRequest.getTaskId(), sendRequest.getParkId(), sendRequest.getServiceType(), sendRequest.getServiceId(), list);
                                p2cDownHandle.send(parkCode, serialNumber, message);
                            }
                        }
                    }
                    return ObjectResponse.success();
                }
            }
        }
        return sendMsgService.send2Park(sendRequest, parkCode, list);
    }

    @Override
    public ObjectResponse<Void> batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode) {
        List<Long> ids = sendRequestList.stream().map(SendRequest::getServiceId).collect(Collectors.toList());
        List<VipRecord> list = vipRecordDao.getRecordListByIds(ids);
        if (CollectionUtils.isEmpty(list)) {
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }
        Map<Integer, Integer> recordIdVipIdMap = list.stream().collect(
                Collectors.toMap(VipRecord::getId, VipRecord::getVipId, (a, b) -> a));
        sendRequestList.forEach(sendRequest -> {
            ClearListRequest clearListRequest = new ClearListRequest();
            Integer vipId = recordIdVipIdMap.get(sendRequest.getServiceId().intValue());
            if (vipId != null) {
                clearListRequest.setId(vipId);
                clearListRequest.setDataType(QueryListRequest.DataType.VIP.getVal());
                sendRequest.setReqServiceType(P2cDownCmdEnum.名单数据清空.getCmdType());
                sendMsgService.send2Park(sendRequest, parkCode, clearListRequest);
                ThreadUtil.sleep(200);
            }
        });
        return ObjectResponse.success();
    }

    private MonthCardRequest buildRequest(VipRecord vipRecord) {
        Integer typeId = vipRecord.getTypeId();
        String plateNums = vipRecord.getPlateNum();
        MonthCardRequest monthCardRequest = new MonthCardRequest();
        Integer operType = vipRecord.getOperType();
        if (operType == 1) {
            monthCardRequest.setCardOperType(MonthRecord.CardOpertype.新增.getType());
        } else if (operType == 2) {
            monthCardRequest.setCardOperType(MonthRecord.CardOpertype.修改.getType());
        } else {
            monthCardRequest.setCardOperType(MonthRecord.CardOpertype.退卡.getType());
        }

        List<MonthCardRequest.Details> list = new ArrayList<>();
        String[] plates = plateNums.split(",");
        for (String plateNum : plates) {
            MonthCardRequest.Details details = new MonthCardRequest.Details();
            details.setPlateNum(plateNum);
            details.setCardId(vipRecord.getVipId());
            details.setCardOwner("VIP车区或内部车区" + typeId);
            details.setStartDate(DateTools.getFormat(DateTools.DF, vipRecord.getStartTime()));
            details.setEndDate(DateTools.getFormat(DateTools.DF, vipRecord.getEndTime()));
            details.setPlotCount(1);
            details.setCardType(1);
            list.add(details);
        }

        monthCardRequest.setDetails(list);
        return monthCardRequest;
    }

    @Override
    public ObjectResponse<Void> batchSend(BatchSendVO vo) {
        Long parkId = vo.getParkId();
        String parkCode = vo.getParkCode();
        ObjectResponse<List<VipInfoDto>> vipInfoDtoRet = vipCarService.getValidByParkId(parkId);
        if (vipInfoDtoRet.getCode().equals(CodeConstants.ERROR_404)) {
            log.info("<批量VIP下发> 没有符合批量下发条件的VIP车，忽略下发，车场编号：{}", parkCode);
            return ObjectResponse.success();
        } else {
            if (ObjectResponse.isSuccess(vipInfoDtoRet)) {

                List<VipInfoDto> vipInfoDtos = vipInfoDtoRet.getData();

                List<VipInfoRequest> list = new ArrayList<>();
                for (VipInfoDto vipInfoDto : vipInfoDtos) {
                    VipInfoRequest vipInfoRequest = new VipInfoRequest();
                    BeanUtils.copyProperties(vipInfoDto, vipInfoRequest);
                    vipInfoRequest.setPhone(StringUtils.isBlank(vipInfoDto.getPhone()) ? "123" : vipInfoDto.getPhone());
                    Integer discountType = vipInfoDto.getDiscountType();
                    if (discountType != null && discountType == 3) {
                        vipInfoRequest.setDiscountNumber((int) (Float.parseFloat(vipInfoDto.getDiscountNumber()) * 100));
                    } else if (discountType != null && discountType == 4) {
                        vipInfoRequest.setDiscountNumber((int) (Float.parseFloat(vipInfoDto.getDiscountNumber()) * 10));
                    } else if (discountType != null && discountType == 2) {
                        vipInfoRequest.setDiscountNumber(ToolsUtil.parseInt(vipInfoDto.getDiscountNumber()));
                    }
                    vipInfoRequest.setStartDate(vipInfoDto.getStartTime() + " 00:00:00");
                    vipInfoRequest.setEndDate(vipInfoDto.getEndTime() + " 23:59:59");
                    list.add(vipInfoRequest);
                }
                List<Object> collect = list.stream().map(VipInfoRequest::getVipId).collect(Collectors.toList());
                batchDownBizHandle.batchDown(vo, list, collect, HEADERS, P2cDownCmdEnum.VIP车辆.getCmdType());
                return ObjectResponse.success();

            } else {
                return ObjectResponse.instance(vipInfoDtoRet.getCode(), vipInfoDtoRet.getMsg(), null);
            }
        }
    }

    @Override
    public ObjectResponse<Void> batchSendRepeat(BatchSendRepeatVO vo) {
        String parkCode = vo.getParkCode();
        ObjectResponse<List<VipInfoDto>> vipInfoDtoRet;
        if (StrUtil.isNotBlank(vo.getFailIds())){
            vipInfoDtoRet = vipCarService.getValidByIds(Arrays.asList(vo.getFailIds().split(",")));
        } else {
            vipInfoDtoRet = vipCarService.getValidByParkId(vo.getParkId());
        }
        if (vipInfoDtoRet.getCode().equals(CodeConstants.ERROR_404)) {
            log.info("<批量VIP下发> 没有符合批量下发条件的VIP车，忽略下发，车场编号：{}", parkCode);
            //todo
            return ObjectResponse.success();
        } else {
            if (ObjectResponse.isSuccess(vipInfoDtoRet)) {

                List<VipInfoDto> vipInfoDtos = vipInfoDtoRet.getData();

                List<VipInfoRequest> list = new ArrayList<VipInfoRequest>();
                for (VipInfoDto vipInfoDto : vipInfoDtos) {
                    VipInfoRequest vipInfoRequest = new VipInfoRequest();
                    BeanUtils.copyProperties(vipInfoDto, vipInfoRequest);
                    Integer discountType = vipInfoDto.getDiscountType();
                    if (discountType != null && discountType == 3) {
                        vipInfoRequest.setDiscountNumber((int) (Float.parseFloat(vipInfoDto.getDiscountNumber()) * 100));
                    } else if (discountType != null && discountType == 4) {
                        vipInfoRequest.setDiscountNumber((int) (Float.parseFloat(vipInfoDto.getDiscountNumber()) * 10));
                    } else if (discountType != null && discountType == 2) {
                        vipInfoRequest.setDiscountNumber(ToolsUtil.parseInt(vipInfoDto.getDiscountNumber()));
                    }
                    vipInfoRequest.setStartDate(vipInfoDto.getStartTime() + " 00:00:00");
                    vipInfoRequest.setEndDate(vipInfoDto.getEndTime() + " 23:59:59");
                    list.add(vipInfoRequest);
                }
                List<Object> collect = list.stream().map(VipInfoRequest::getVipId).collect(Collectors.toList());
                batchDownBizHandle.repeatBatch(vo, list, collect, HEADERS, P2cDownCmdEnum.VIP车辆.getCmdType());
                return ObjectResponse.success();

            } else {
                return ObjectResponse.instance(vipInfoDtoRet.getCode(), vipInfoDtoRet.getMsg(), null);
            }
        }
    }

    private ObjectResponse<Void> send(SendRequest sendRequest, String parkCode, List<VipInfoRequest> list) {
        return sendMsgService.send2Park(sendRequest, parkCode, list);
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.VIP车辆.getCmdType());
    }
}
