package com.icetech.park.service.down.p2c.impl;

import com.icetech.cloudcenter.api.NotifyService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.park.dao.FirmwareUpgradeDao;
import com.icetech.park.domain.entity.FirmwareUpgrade;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.p2c.UpgradeRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.UUIDTools;
import com.icetech.common.validator.Validator;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

@Service("p2cUpgradeServiceImpl")
@Slf4j
public class UpgradeServiceImpl extends AbstractService implements ResponseService<String>, NotifyService<String> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private ParkService parkService;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private FirmwareUpgradeDao firmwareUpgradeDao;

    public ObjectResponse<Void> execute(UpgradeRequest upgradeRequest, String sn, Long parkId) {

        if (!Validator.validate(upgradeRequest)){
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(parkId);
        if (!ObjectResponse.isSuccess(parkObjectResponse)){
            return ObjectResponse.failed(CodeConstants.ERROR_404, "车场不存在");
        }
        Park park = parkObjectResponse.getData();
        Message<UpgradeRequest> message = new Message<>(parkId, P2cDownCmdEnum.升级指令下发.getCmdType(), upgradeRequest);
        String messageId;
        if (Objects.nonNull(upgradeRequest.getUpgradeChannel()) && upgradeRequest.getUpgradeChannel() == 2) {
            Integer reqServiceType = message.getReqServiceType();
            UpgradeRequest payload = message.getPayload();
            P2cBaseRequest<UpgradeRequest> p2cBaseRequest = new P2cBaseRequest<>();
            p2cBaseRequest.setCmd(P2cDownCmdEnum.getCmd(reqServiceType));
            messageId = UUIDTools.getUuid();
            p2cBaseRequest.setMessageId(messageId);
            p2cBaseRequest.setTimestamp(DateTools.unixTimestamp());
            p2cBaseRequest.setBizContent(payload);
            p2cDownHandle.pushAll(sn, p2cBaseRequest, "ops");
        } else {
            messageId = p2cDownHandle.send(park.getParkCode(), sn, message);
        }
        if (messageId == null){
            return ObjectResponse.failed(CodeConstants.ERROR_3003,"相机未连接");
        }else{
            Map<String, Object> hashMap = new HashMap<>();
            hashMap.put("sn", sn);
            hashMap.put("newFirmwareVer", upgradeRequest.getNewFirmwareVer());
            hashMap.put("sendTime", upgradeRequest.getSendTime());
            //1：下发固件地址，3：固件升级
            hashMap.put("step", 1);
            hashMap.put("stepStatus", 1);
            String key = RedisConstants.UPGRADE_TASK_HASH + upgradeRequest.getTaskId();
            redisUtils.hPutAll(key, hashMap);
            redisUtils.expire(key, 600);

            //设置对应关系
            redisUtils.set(RedisConstants.UPGRADE_TASK_MESSAGE + messageId, upgradeRequest.getTaskId(), 60L);

            FirmwareUpgrade firmwareUpgrade = new FirmwareUpgrade();
            firmwareUpgrade.setActionTime(new Date(upgradeRequest.getSendTime() * 1000));
            firmwareUpgrade.setTaskId(upgradeRequest.getTaskId());
            firmwareUpgrade.setStatus(FirmwareUpgrade.StatusEnum.升级中.status);
            firmwareUpgradeDao.updateByTaskId(firmwareUpgrade);
            return ObjectResponse.success();
        }

    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String sn) {
        String taskId = redisUtils.get(RedisConstants.UPGRADE_TASK_MESSAGE + p2cBaseResponse.getMessageId(), String.class);
        if (taskId != null){
            Integer successCode = 200;
            String key = RedisConstants.UPGRADE_TASK_HASH + taskId;

            FirmwareUpgrade firmwareUpgrade = firmwareUpgradeDao.selectByTaskId(taskId);
            if (firmwareUpgrade == null){
                log.warn("[设备升级指令下发响应] 未查询到任务ID[{}]", taskId);
                return;
            }
            if (successCode.equals(p2cBaseResponse.getCode())){
                firmwareUpgrade.setStatus(FirmwareUpgrade.StatusEnum.升级中.status);
                redisUtils.hPut(key, "step", 3);
                redisUtils.hPut(key, "updateTime", DateTools.Date());
            }else{
                firmwareUpgrade.setStatus(FirmwareUpgrade.StatusEnum.升级失败.status);
                firmwareUpgrade.setReason(p2cBaseResponse.getMsg());
                redisUtils.hPut(key, "stepStatus", 3);
                redisUtils.hPut(key, "reason", p2cBaseResponse.getMsg());
                redisUtils.hPut(key, "updateTime", DateTools.Date());
            }
            firmwareUpgrade.setUpdateTime(new Date());
            firmwareUpgradeDao.updateById(firmwareUpgrade);

        }else{
            log.warn("[设备升级指令下发响应] 未查询到messageId[{}]对应的任务ID", p2cBaseResponse.getMessageId());
        }
    }
}
