package com.icetech.park.service.down.p2c.impl;

import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.api.store.StoreCardService;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.p2c.StoredCardRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.fee.domain.entity.storecard.StoreCard;
import com.icetech.fee.domain.entity.storecard.StoreCardRecord;
import com.icetech.park.service.AbstractConfigDownService;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.ONE;

/**
 * 储值卡车辆信息下发
 */
@Slf4j
@Service("p2cStoredCardServiceImpl")
public class StoredCardServiceImpl extends AbstractConfigDownService
        implements DownService<List<StoredCardRequest>, Void>, ResponseService<String> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Resource
    private StoreCardService storeCardService;
    @Autowired
    private SendMsgServiceImpl sendMsgService;

    @Override
    public ObjectResponse<Void> send(SendRequest sendRequest, String parkCode) {

        Long serviceId = sendRequest.getServiceId();
        try {
            StoredCardRequest request = null;
            // include等于"use"时代表使用储值卡, 取储值卡使用记录表数据
            if("use".equals(sendRequest.getInclude())) {
                return ObjectResponse.failed(CodeConstants.ERROR_406);
            } else {
                ObjectResponse<StoreCardRecord> recordResp = storeCardService.getStoreCardRecordById(serviceId);
                if (!ObjectResponse.isSuccess(recordResp)) {
                    return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "储值卡记录不存在"));
                }
                StoreCardRecord record = recordResp.getData();
                int operaType = convertOperaType(record.getCardOperType());

                ObjectResponse<StoreCard> storeCardResp = storeCardService.getStoreCardDetail(record.getCardId().intValue());
                if (!ObjectResponse.isSuccess(storeCardResp)) {
                    return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "储值卡信息不存在"));
                }
                StoreCard storeCard = storeCardResp.getData();
                request = buildRequest(storeCard);
                request.setOperType(operaType);
                request.setCardId(record.getCardId());
                request.setPlateNums(record.getPlateNum() == null ? "" : record.getPlateNum().replace(",", "/"));
            }

            List<StoredCardRequest> list = new ArrayList<>();
            list.add(request);
            return sendMsgService.send2Park(sendRequest, parkCode, list);

        } catch (ResponseBodyException e) {
            log.warn("[端云储值卡下发] 下发失败, code[{}], msg[{}]", e.getErrCode(), e.getMessage(), e);
            return ObjectResponse.failed(e.getErrCode(), TextConstant.getDefaultMessage(TextConstant.ONE, e.getMessage() == null ? "端云储值卡下发失败" : e.getMessage()));
        } catch (Exception e) {
            log.error("[端云储值卡下发] 下发储值卡操作记录失败", e);
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "端云储值卡下发失败"));
        }
    }

    private StoredCardRequest buildRequest(StoreCard storeCard) {
        StoredCardRequest request = new StoredCardRequest();
        request.setCardOwner(storeCard.getCardOwner());
        if (storeCard.getStartTime() == null || storeCard.getEndTime() == null) {
            throw new ResponseBodyException(ERROR_410, TextConstant.getDefaultMessage(ONE, "日期不合法"));
        }
        request.setStartTime(DateFormatUtils.format(storeCard.getStartTime(), StoredCardRequest.DATE_PATTERN) + " 00:00:00");
        request.setEndTime(DateFormatUtils.format(storeCard.getEndTime(), StoredCardRequest.DATE_PATTERN) + " 23:59:59");
        return request;
    }

    private int convertOperaType(int operaType) {
        // 充值及编辑统一按照编辑下发
        if (operaType == 3) {
            return 2;
        }
        if (operaType == 4) {
            return 3;
        }

        return operaType;
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String sn) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.储值卡车辆信息.getCmdType());
    }
}
