package com.icetech.park.service.down.p2c.impl;


import com.icetech.cloudcenter.api.order.CarOrderEnterService;
import com.icetech.cloudcenter.api.order.CarOrderExitService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.enumeration.DataCollectionEnum;
import com.icetech.cloudcenter.domain.enumeration.EnexTypeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.request.DataEnterRequest;
import com.icetech.cloudcenter.domain.request.DataExitRequest;
import com.icetech.cloudcenter.domain.request.p2c.SoftTriggerRequest;
import com.icetech.cloudcenter.domain.response.EnexCarInfoBaseResponse;
import com.icetech.cloudcenter.domain.response.EnterCarInfoResponse;
import com.icetech.cloudcenter.domain.response.ExitCarInfoResponse;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.cloudcenter.domain.response.p2c.CarEnterResult;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderCarInfoConstant;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.common.utils.UUIDTools;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.oss.OssService;
import com.icetech.park.component.AsyncNotifyClient;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.CaptureService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import com.icetech.third.domain.entity.third.SendInfoRecord;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Map;

@Service("p2cSoftTriggerServiceImpl")
@Slf4j
public class SoftTriggerServiceImpl extends AbstractService implements CaptureService, ResponseService<String> {

    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private OrderService orderService;
    @Autowired
    private OssService ossService;
    @Autowired
    private CarOrderEnterService carOrderEnterService;
    @Autowired
    private CarOrderExitService carOrderExitService;
    @Resource
    private AsyncNotifyClient asyncNotifyClient;
    public static final String LOCK_KEY = "softTrigger:";

    @Override
    public ObjectResponse<Void> execute(Long parkId, String parkCode, String channelId, String topic, SoftTriggerRequest extraInfo){
        SoftTriggerRequest softTriggerRequest = new SoftTriggerRequest();
        String triggerNo = UUIDTools.getUuid();
        softTriggerRequest.setTriggerNo(triggerNo);
        String deviceNo = cacheHandle.getSerialNumber(parkCode, channelId);
        Message<SoftTriggerRequest> message = new Message<>(parkId, P2cDownCmdEnum.软触发.getCmdType(), softTriggerRequest);
        if (redisUtils.tryLock(LOCK_KEY + deviceNo, deviceNo, 3000)) {
            String messageId = p2cDownHandle.send(parkCode, deviceNo, message, topic, JsonUtils.toString(extraInfo));
            if (messageId == null){
                redisUtils.releaseLock(LOCK_KEY + deviceNo);
                return ObjectResponse.failed(CodeConstants.ERROR_3003,"相机未连接");
            }
            redisUtils.set(RedisConstants.MESSAGE_ID_TRIGGER_NO + triggerNo, messageId, 10L);
            return ObjectResponse.failed(CodeConstants.ERROR_12002, triggerNo);
        } else {
            return ObjectResponse.failed(CodeConstants.ERROR_405, "操作太快，请稍后");
        }
    }

    @Override
    public ObjectResponse<String> executeAsync(Long parkId, String parkCode, String channelId){

        SoftTriggerRequest softTriggerRequest = new SoftTriggerRequest();
        String triggerNo = UUIDTools.getUuid();
        softTriggerRequest.setTriggerNo(triggerNo);
        String deviceNo = cacheHandle.getSerialNumber(parkCode, channelId);
        Message<SoftTriggerRequest> message = new Message<>(parkId, P2cDownCmdEnum.软触发.getCmdType(), softTriggerRequest);
        String messageId = p2cDownHandle.send(parkCode, deviceNo, message);
        if (messageId == null){
            return ObjectResponse.failed(CodeConstants.ERROR_3003,"相机未连接");
        }else{
            return ObjectResponse.success(triggerNo);
        }
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.软触发.getCmdType());
    }

    public void notify(String triggerNo, Object object, SendInfoRecord<SoftTriggerRequest> sendInfoRecord) {
        log.debug("triggerNo:{}, object:{}, sendInfoRecord: {}", triggerNo, object, sendInfoRecord);
        Map<String, String> extraInfo = JsonUtils.parseStringMap(sendInfoRecord.getExtraInfo());
        String biz = extraInfo.get("biz");
        if (SoftTriggerRequest.ExtraInfoEnum.ALLOW_ENTER.val.equals(biz)) {
            CarEnterRequest entrance = (CarEnterRequest) object;
            entrance.setPlateNum(extraInfo.get("plateNum"));
            DataEnterRequest enterRequest = JsonUtils.parseObject(extraInfo.get("requestVO"), DataEnterRequest.class);
            entrance.setType(enterRequest.getType());
            entrance.setCarType(enterRequest.getCarType());
            entrance.setCarDesc(enterRequest.getSpecialCar());
            entrance.setDebug(enterRequest.isDebug());
            entrance.setEnterTime(enterRequest.getEnterTime() == null ? DateTools.unixTimestamp() : enterRequest.getEnterTime());
            entrance.setEnterWay(entrance.getPlateNum().startsWith("临")
                    ? OrderCarInfoConstant.IN_OUT_WAY_QR_CODE : OrderCarInfoConstant.IN_OUT_WAY_SOFTWARE_MANUAL);
            entrance.setOperaUser(enterRequest.getOperaUser());
            entrance.setRemark(extraInfo.get("remark"));
            entrance.setEnterTerminal(enterRequest.getEnterTerminal());
            ObjectResponse<CarEnterResult> objectResponse = carOrderEnterService.enter(entrance);
            if (ObjectResponse.isSuccess(objectResponse)) {
                //清除缓存中的上次异常记录
                cacheHandle.removeEntrace(sendInfoRecord.getParkCode(), entrance.getInandoutCode());
            }
            log.debug("topic: {}, triggerNo: {}", sendInfoRecord.getTopic(), triggerNo);
            asyncNotifyClient.callBack(triggerNo, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), objectResponse);
        } else if (SoftTriggerRequest.ExtraInfoEnum.GET_CAR_INFO.val.equals(biz)) {
            EnexCarInfoBaseResponse enexCarInfoBaseResponse;
            Integer type = Integer.parseInt(extraInfo.get("type"));
            if (EnexTypeEnum.入场.getType().equals(type)) {
                //入口通道
                try {
                    CarEnterRequest entrance = (CarEnterRequest) object;
                    enexCarInfoBaseResponse = getEntranceCarInfoResponse(entrance);
                } catch (ClassCastException e) {
                    CarExitRequest exit = (CarExitRequest) object;
                    CarEnterRequest entrance = new CarEnterRequest();
                    BeanUtils.copyProperties(exit, entrance);
                    entrance.setEnterTime(exit.getExitTime());
                    enexCarInfoBaseResponse = getEntranceCarInfoResponse(entrance);
                } catch (Exception e) {
                    log.error("软触发处理异常, e=", e);
                    return;
                }
            } else {
                //出口通道
                try {
                    CarExitRequest exit = (CarExitRequest) object;
                    if (exit.getInandoutCode() != null) {
                        redisUtils.set(RedisConstants.SOFT_TRIGGER_IMAGE_KEY + exit.getInandoutCode(), exit.getMaxImage(), 600L);
                    }
                    enexCarInfoBaseResponse = getExitCarInfoResponse(exit.getParkId(), exit.getParkCode(), exit);
                } catch (ClassCastException e) {
                    CarEnterRequest entrance = (CarEnterRequest) object;
                    CarExitRequest exit = new CarExitRequest();
                    BeanUtils.copyProperties(entrance, exit);
                    exit.setExitTime(entrance.getEnterTime());
                    enexCarInfoBaseResponse = getExitCarInfoResponse(exit.getParkId(), exit.getParkCode(), exit);
                } catch (Exception e) {
                    log.error("软触发处理异常, e=", e);
                    return;
                }
            }
            enexCarInfoBaseResponse.setDataCollection(DataCollectionEnum.端云.getType());
            log.debug("enexCarInfoBaseResponse:{}", enexCarInfoBaseResponse);
            asyncNotifyClient.callBack(triggerNo, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), ObjectResponse.success(enexCarInfoBaseResponse));
        } else if (SoftTriggerRequest.ExtraInfoEnum.ALLOW_EXIT.val.equals(biz)) {
            CarExitRequest exit = (CarExitRequest) object;
            String cacheOrderNum = exit.getOrderNum();
            DataExitRequest exitRequest = JsonUtils.parseObject(extraInfo.get("requestVO"), DataExitRequest.class);
            if (cacheOrderNum != null && !exitRequest.getOrderNum().equals(cacheOrderNum)) {
                log.warn("[平台离场] 订单与缓存中的异常订单不匹配，缓存数据为:{},请求参数为:{}", exit, exitRequest);
            }
            String orderNum = exitRequest.getOrderNum();
            ObjectResponse<OrderInfo> orderInfoObjectResponse = orderService.findByOrderNum(orderNum);
            if (ObjectResponse.isSuccess(orderInfoObjectResponse)){
                OrderInfo orderInfo = orderInfoObjectResponse.getData();
                Long enterTime = orderInfo.getEnterTime();
                //如果入场时间大于参数中的离场时间，则使用当前时间作为离场时间
                if (enterTime > exitRequest.getExitTime().getTime() / 1000){
                    exitRequest.setExitTime(new Date());
                }
            }
            exit.setType(exitRequest.getType());
            exit.setInandoutCode(exitRequest.getAisleCode());
            exit.setCarType(exitRequest.getCarType() != null ? exitRequest.getCarType() : exit.getCarType());
            exit.setOrderNum(orderNum == null ? cacheOrderNum : orderNum);
            exit.setPlateNum(exitRequest.getPlateNum());
            exit.setExitTime(exit.getExitTime() != null ? exit.getExitTime() : (exitRequest.getExitTime().getTime() / 1000));
            if (exitRequest.getOperAccount() != null) {
                exit.setOperAccount(exitRequest.getOperAccount());
            }
            if (exitRequest.getExitWay() != null) {
                exit.setExitWay(exitRequest.getExitWay());
            }
            if (exitRequest.getExitTerminal() != null) {
                exit.setExitTerminal(exitRequest.getExitTerminal());
            }
            ObjectResponse<Map<String, Object>> objectResponse = carOrderExitService.exit(exit);
            if (ObjectResponse.isSuccess(objectResponse)) {
                //清除缓存中的上次异常记录
                cacheHandle.removeExit(exitRequest.getParkCode(), exitRequest.getAisleCode());
                cacheHandle.removeChannelFee(exitRequest.getParkCode(), exitRequest.getAisleCode());
            }
            log.debug("topic: {}, triggerNo: {}", sendInfoRecord.getTopic(), triggerNo);
            asyncNotifyClient.callBack(triggerNo, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), objectResponse);
        } else if (SoftTriggerRequest.ExtraInfoEnum.EXCEPTION_EXIT.val.equals(biz)) {
            CarExitRequest exit = (CarExitRequest) object;
            exit.setPlateNum(extraInfo.get("plateNum"));
            exit.setOrderNum(extraInfo.get("orderNum"));
            exit.setOperAccount(extraInfo.get("operAccount"));
            QueryOrderFeeResponse channelFee = cacheHandle.getChannelFee(exit.getParkCode(), exit.getInandoutCode());
            if (channelFee != null){
                exit.setTotalAmount(channelFee.getTotalAmount());
                exit.setPaidAmount(channelFee.getPaidAmount());
                //String discountPrice = NumberUtils.decimalAdd(channelFee.getDiscountPrice(), channelFee.getDiscountAmount()).toString();
                exit.setDiscountAmount(channelFee.getDiscountAmount());
            }
            exit.setExitWay(OrderCarInfoConstant.IN_OUT_WAY_SOFTWARE_MANUAL);
            if (extraInfo.get("operAccount") != null) {
                exit.setOperAccount(extraInfo.get("operAccount"));
            }
            if (extraInfo.get("exitTerminal") != null) {
                exit.setOperAccount(extraInfo.get("exitTerminal"));
            }
            carOrderExitService.exceptionExit(exit, extraInfo.get("reasonType") == null
                    ? null : Integer.parseInt(extraInfo.get("reasonType")));
        } else if (SoftTriggerRequest.ExtraInfoEnum.CALL_VOICE_REPORT.val.equals(biz)) {
            int type = Integer.parseInt(extraInfo.get("type"));
            if (type == 2) {
                CarExitRequest exit = (CarExitRequest) object;
                exit.setPlateNum(extraInfo.get("plateNum"));
                exit.setOrderNum(extraInfo.get("orderNum"));
                exit.setOperAccount(extraInfo.get("operAccount"));
                cacheHandle.setExit(exit.getParkCode(), exit.getInandoutCode(), exit);
            } else {
                CarEnterRequest enterRequest = (CarEnterRequest) object;
                enterRequest.setPlateNum(extraInfo.get("plateNum"));
                enterRequest.setOrderNum(extraInfo.get("orderNum"));
                enterRequest.setOperaUser(extraInfo.get("operAccount"));
                cacheHandle.setEntrance(enterRequest.getParkCode(), enterRequest.getInandoutCode(), enterRequest);
            }

            asyncNotifyClient.callBack(triggerNo, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), ObjectResponse.failed(extraInfo.get("code"), extraInfo.get("msg")));
        }
    }

    protected EnterCarInfoResponse getEntranceCarInfoResponse(CarEnterRequest entrace){
        EnterCarInfoResponse enterCarInfoResponse = new EnterCarInfoResponse();
        enterCarInfoResponse.setEnterNum(entrace.getPlateNum());
        enterCarInfoResponse.setCarType(entrace.getCarType());
        enterCarInfoResponse.setType(entrace.getType());
        enterCarInfoResponse.setOrderNum(entrace.getOrderNum());
        String maxImage = entrace.getMaxImage();
        if (maxImage != null) {
            enterCarInfoResponse.setImgUrl(ossService.getImageUrl(maxImage));
            enterCarInfoResponse.setImgPath(maxImage);
        }
        return enterCarInfoResponse;
    }
    /**
     * 获取离场通道信息
     * @param parkId 车场ID
     * @param parkCode 车场编号
     * @param carExitRequest 离场请求参数
     * @return 响应
     */
    private ExitCarInfoResponse getExitCarInfoResponse(Long parkId, String parkCode, CarExitRequest carExitRequest) {
        ExitCarInfoResponse exitCarInfoResponse = new ExitCarInfoResponse();
        String orderNum = carExitRequest.getOrderNum();
        exitCarInfoResponse.setExitNum(carExitRequest.getPlateNum());
        exitCarInfoResponse.setCarType(carExitRequest.getCarType());
        exitCarInfoResponse.setType(carExitRequest.getType());
        if (StringUtils.isNotBlank(orderNum)){
            OrderInfo orderInfo = new OrderInfo();
            orderInfo.setOrderNum(orderNum);
            orderInfo.setParkId(parkId);
            ObjectResponse<OrderInfo> byOrderInfo = orderService.findByOrderInfo(orderInfo);
            if (byOrderInfo.getCode().equals(CodeConstants.SUCCESS)){
                OrderInfo data = byOrderInfo.getData();
                exitCarInfoResponse.setEnterTime(new Date(data.getEnterTime() * 1000L));
                exitCarInfoResponse.setEnterNum(data.getPlateNum());
                exitCarInfoResponse.setLocalOrderNum(data.getLocalOrderNum());
                exitCarInfoResponse.setNoneEnterFlag(data.getNoneEnterFlag());
            }
            exitCarInfoResponse.setOrderNum(carExitRequest.getOrderNum());
        }else{
            String plateNum = carExitRequest.getPlateNum();
            ObjectResponse<OrderInfo> byOrderInfo = orderService.findInPark(plateNum, parkCode);
            if (byOrderInfo.getCode().equals(CodeConstants.SUCCESS)){
                OrderInfo data = byOrderInfo.getData();
                exitCarInfoResponse.setEnterTime(new Date(data.getEnterTime() * 1000L));
                exitCarInfoResponse.setEnterNum(data.getPlateNum());
                exitCarInfoResponse.setOrderNum(data.getOrderNum());
                exitCarInfoResponse.setLocalOrderNum(data.getLocalOrderNum());
                exitCarInfoResponse.setNoneEnterFlag(data.getNoneEnterFlag());
            }
        }
        Long exitTime = carExitRequest.getExitTime();
        if (exitTime != null){
            exitCarInfoResponse.setExitTime(new Date(exitTime * 1000L));
        }else{
            //离场时间默认为当前时间
            exitCarInfoResponse.setExitTime(new Date());
        }
        String maxImage = carExitRequest.getMaxImage();
        if (maxImage != null){
            exitCarInfoResponse.setImgUrl(ossService.getImageUrl(maxImage));
            exitCarInfoResponse.setImgPath(maxImage);
        }
        return exitCarInfoResponse;
    }
}
