package com.icetech.park.service.down.p2c.impl;

import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.api.order.OrderPayService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.charge.dto.OrderSumFeeDto;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.request.NoplateExitRequest;
import com.icetech.cloudcenter.domain.request.p2c.PrepaySyncRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.MoneyTool;
import com.icetech.common.utils.StringUtils;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

import static com.icetech.basics.constants.TextConstant.ERROR_410;

/**
 * 缴费通知下发接口
 *
 * @author fangct
 */
@Slf4j
@Service("p2cPrepaySyncServiceImpl")
public class PrepaySyncServiceImpl extends AbstractService implements ResponseService<String>, DownService<PrepaySyncRequest, Void> {
    @Autowired
    private OrderPayService orderPayService;
    @Autowired
    private OrderService orderService;
    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private SendMsgServiceImpl sendMsgService;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private P2cNoplateExitServiceImpl p2cNoplateExitService;

    @Override
    public ObjectResponse<Void> send(SendRequest sendRequest, String parkCode) {
        Long serviceId = sendRequest.getServiceId();

        OrderPay orderPay = new OrderPay();
        orderPay.setId(serviceId);
        ObjectResponse<OrderPay> objectResponse = orderPayService.findOne(orderPay);
        if (!ObjectResponse.isSuccess(objectResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "支付单记录不存在"));
        }
        ObjectResponse.notError(objectResponse);
        OrderPay orderPayRes = objectResponse.getData();

        PrepaySyncRequest prepaySyncRequest = buildRequest(orderPayRes, parkCode);
        /*
         * 判断车辆是否场内码缴费，如果是，则下发开闸和屏显
         */
        CarExitRequest cacheExitOrder = cacheHandle.getExitByOrderNum(parkCode, orderPayRes.getOrderNum());
        if (StringUtils.isBlank(orderPayRes.getChannelId()) && cacheExitOrder != null){
            cacheHandle.removeExitOrder(parkCode, orderPayRes.getOrderNum());
            String inandoutCode = cacheExitOrder.getInandoutCode();
            /*
             * 判断通道缓存的订单是否和当前缴费的订单一致
             */
            CarExitRequest cacheExit = cacheHandle.getExit(parkCode, inandoutCode);
            if (cacheExit != null && cacheExit.getOrderNum() != null && cacheExit.getOrderNum().equals(orderPayRes.getOrderNum())){
                Long exitTime = cacheExitOrder.getExitTime();
                log.info("<端云-预缴费下发> 缓存中的离场时间与当前时间相差{}s", System.currentTimeMillis()/1000 - exitTime);
                NoplateExitRequest noplateExitRequest = new NoplateExitRequest();
                noplateExitRequest.setParkCode(parkCode);
                noplateExitRequest.setChannelId(cacheExitOrder.getInandoutCode());
                noplateExitRequest.setExitTime(exitTime);
                noplateExitRequest.setPlateNum(cacheExitOrder.getPlateNum());
                noplateExitRequest.setOrderNum(orderPayRes.getOrderNum());
                p2cNoplateExitService.noplateExit(noplateExitRequest);
            }

        }
        /*
         * 端云场中场，小场出口支付时，不下发预缴费，防止大场出口相机直接开闸
         */
        String channelId = orderPayRes.getChannelId();
        if (channelId != null){
            ObjectResponse<ParkInoutdevice> inoutdeviceObjectResponse = parkService.getInOutDeviceByCode(orderPayRes.getParkId(), channelId);
            ObjectResponse.notError(inoutdeviceObjectResponse);
            ParkInoutdevice parkInoutdevice = inoutdeviceObjectResponse.getData();
            if (parkInoutdevice != null && parkInoutdevice.getIsMaster() != null && parkInoutdevice.getIsMaster() == 0) {//子通道
                String devices = getSonExit(sendRequest.getParkId());
                if (StringUtils.isNotBlank(devices)){
                    return sendMsgService.send2Devices(devices, sendRequest, parkCode, prepaySyncRequest);
                }else{
                    return ObjectResponse.failed(CodeConstants.ERROR_406, "没有满足下发条件的设备");
                }
            }
        }
        return sendMsgService.send2Exit(sendRequest, parkCode, prepaySyncRequest);

    }

    /**
     * 构建请求参数
     *
     * @param orderPay 支付记录
     * @param parkCode 车场编号
     * @return 响应
     */
    private PrepaySyncRequest buildRequest(OrderPay orderPay, String parkCode) {
        PrepaySyncRequest prepaySyncRequest = new PrepaySyncRequest();
        prepaySyncRequest.setOrderNum(orderPay.getOrderNum());
        ObjectResponse<OrderInfo> objectResponse = orderService.findByOrderNum(orderPay.getOrderNum());
        if (CodeConstants.SUCCESS.equals(objectResponse.getCode())) {
            OrderInfo orderInfo = objectResponse.getData();
            prepaySyncRequest.setPlateNum(orderInfo.getPlateNum());
        } else {
            log.info("Dubbo根据订单号查询未找到记录，订单号：{}，返回结果：{}", orderPay.getOrderNum(), objectResponse);
            throw new ResponseBodyException(objectResponse.getCode(), "订单号不存在");
        }
        prepaySyncRequest.setTradeNo(orderPay.getTradeNo());
        prepaySyncRequest.setTotalPrice(orderPay.getTotalPrice());
        prepaySyncRequest.setPrepay(orderPay.getPaidPrice());
        prepaySyncRequest.setDiscountPrice(orderPay.getDiscountPrice());
        prepaySyncRequest.setPayTime(orderPay.getPayTime());
        ObjectResponse<OrderSumFeeDto> sumFeeDtoObjectResponse = orderPayService.getSumFee(parkCode, orderPay.getOrderNum());
        ObjectResponse.notError(sumFeeDtoObjectResponse);
        OrderSumFeeDto sumFeeDto = sumFeeDtoObjectResponse.getData();
        prepaySyncRequest.setPrepayFen(Integer.parseInt(MoneyTool.fromYuanToFen(String.valueOf(sumFeeDto.getPaidPrice()))));
        prepaySyncRequest.setDiscountPriceFen(Integer.parseInt(MoneyTool.fromYuanToFen(String.valueOf(sumFeeDto.getDiscountPrice()))));
        return prepaySyncRequest;
    }
    @Autowired
    private ParkService parkService;
    @Autowired
    private ParkDeviceService parkDeviceService;

    /**
     * 获取子出口相机序列号
     * @param parkId 车场ID
     * @return 响应
     */
    private String getSonExit(Long parkId) {
        //查询所有出口通道
        ObjectResponse<List<ParkInoutdevice>> objectResponse = parkService.getChannelByType(parkId, 2);
        ObjectResponse.notError(objectResponse);
        StringBuilder sendDevices = new StringBuilder();
        List<ParkInoutdevice> inoutdevices = objectResponse.getData();
        for (ParkInoutdevice inoutdevice : inoutdevices) {
            //只下发子通道出口
            if (inoutdevice.getIsMaster() != null && inoutdevice.getIsMaster() == 0) {
                ObjectResponse<List<ParkDevice>> deviceListResponse = parkDeviceService.getDeviceListByParkChannelId(parkId, inoutdevice.getId().intValue(), 1);
                if (ObjectResponse.isSuccess(deviceListResponse)) {
                    List<ParkDevice> parkDevices = deviceListResponse.getData();
                    for (ParkDevice parkDevice : parkDevices) {
                        String serialNumber = parkDevice.getSerialNumber();
                        sendDevices.append(serialNumber).append(";");
                    }
                } else {
                    log.info("通道{}下尚未添加设备", inoutdevice.getInandoutCode());
                }
            }
        }
        return sendDevices.toString();
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.预缴费.getCmdType());
    }

}
