package com.icetech.park.service.down.p2c.impl;

import com.icetech.basics.constants.TextConstant;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.p2c.MonthCardRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.dao.more.MoreMonthInfoMapper;
import com.icetech.park.domain.entity.more.MoreMonthInfo;
import com.icetech.park.domain.entity.more.MoreMonthPark;
import com.icetech.park.domain.entity.more.MoreMonthPlate;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import com.icetech.park.service.more.MoreMonthParkService;
import com.icetech.park.service.more.MoreMonthPlateService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.ONE;

/**
 * 集团月卡下发
 * @author wangzhiwei
 * @date 2024-08-20
 */
@Service("p2cMoreMonthCardServiceImpl")
@RefreshScope
@Slf4j
public class MoreMonthCardServiceImpl extends AbstractService implements ResponseService<String>, DownService<MonthCardRequest, Long> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private MoreMonthInfoMapper moreMonthInfoMapper;
    @Autowired
    private MoreMonthPlateService moreMonthPlateService;
    @Autowired
    private MoreMonthParkService moreMonthParkService;
    @Autowired
    private SendMsgServiceImpl sendMsgService;
    @Autowired
    private ParkService parkService;



    @Override
    public ObjectResponse<Long> send(SendRequest sendRequest, String parkCode) {
        Long serviceId = sendRequest.getServiceId();
        MoreMonthInfo moreMonthInfo = moreMonthInfoMapper.selectById(serviceId);
        if (moreMonthInfo == null){
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(ONE, "集团月卡不存在"));
        }
        // 2.获取多车场月卡关联车场信息
        List<MoreMonthPark> monthParks = moreMonthParkService.getMonthParkByMoreMonthId(serviceId);
        if (CollectionUtils.isEmpty(monthParks)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.TWO, "多车场月卡没有关联车场"));
        }
        monthParks.forEach(monthPark -> {
            Park park = parkService.findByParkId(monthPark.getParkId()).getData();
            MonthCardRequest monthCardRequest = buildRequest(sendRequest,moreMonthInfo, 1);
            sendRequest.setServiceType(P2cDownCmdEnum.月卡.getCmdType());
            sendRequest.setParkId(monthPark.getParkId());
            sendMsgService.send2Park(sendRequest, park.getParkCode(), monthCardRequest);
        });
        return ObjectResponse.success();

    }

    /**
     * 构建请求参数
     * @return 响应
     */
    private MonthCardRequest buildRequest(SendRequest sendRequest,MoreMonthInfo moreMonthInfo, int cardType) {
        Long serviceId = sendRequest.getServiceId();
        MonthCardRequest monthCardRequest = new MonthCardRequest();
        monthCardRequest.setCardOperType(1);
        // 5.下发删除逻辑处理
        if (StringUtils.isNotEmpty(sendRequest.getInclude())) {
            monthCardRequest.setCardOperType(4);
        }
        // 3.获取多车场月卡关联车牌信息
        List<MoreMonthPlate> moreMonthPlates = moreMonthPlateService.getMoreMonthPlateByMoreMonthId(serviceId);
        String plateNums = moreMonthPlates.stream()
                .map(MoreMonthPlate::getPlateNum)
                .filter(Objects::nonNull)
                .collect(Collectors.joining(","));
        List<MonthCardRequest.Details> list = new ArrayList<>();
        MonthCardRequest.Details details = new MonthCardRequest.Details();
        details.setPlateNum(plateNums);
        details.setCardId(moreMonthInfo.getId().intValue());
        details.setCardOwner(StringUtils.isBlank(moreMonthInfo.getCardOwner()) ? "default" : moreMonthInfo.getCardOwner());
        details.setStartDate(DateFormatUtils.format(moreMonthInfo.getStartTime(), "yyyy-MM-dd 00:00:00"));
        details.setEndDate(DateFormatUtils.format(moreMonthInfo.getEndTime(), "yyyy-MM-dd 23:59:59"));
        details.setPlotCount(moreMonthInfo.getPlotCount());
        details.setRegionIds("0");
        details.setCardType(cardType);
        list.add(details);
        monthCardRequest.setDetails(list);
        return monthCardRequest;
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.集团月卡.getCmdType());
    }
}
