package com.icetech.park.service.down.p2c.impl;

import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.StrUtil;
import com.github.pagehelper.util.StringUtil;
import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.SendMessage;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.api.month.MonthCarService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.request.p2c.ClearListRequest;
import com.icetech.cloudcenter.domain.request.p2c.MonthCardRequest;
import com.icetech.cloudcenter.domain.request.p2c.QueryListRequest;
import com.icetech.cloudcenter.domain.response.MonthCardDto;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.BatchSendRepeatVO;
import com.icetech.cloudcenter.domain.vo.BatchSendVO;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.StringUtils;
import com.icetech.fee.dao.monthcar.MonthProductDao;
import com.icetech.fee.dao.monthcar.MonthRecordDao;
import com.icetech.fee.domain.entity.monthcar.MonthInfo;
import com.icetech.fee.domain.entity.monthcar.MonthProduct;
import com.icetech.fee.domain.entity.monthcar.MonthRecord;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.handle.BatchDownBizHandle;
import com.icetech.park.service.handle.P2cDownHandle;
import com.icetech.third.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.DEFAULT;
import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.MOOD_SUSPEND_TIME;
import static com.icetech.basics.constants.TextConstant.MOOD_TAKE_EFFECT_TIME;
import static com.icetech.basics.constants.TextConstant.ONE;

@Service("p2cMonthCardServiceImpl")
@RefreshScope
@Slf4j
public class MonthCardServiceImpl extends AbstractService implements ResponseService<String>, DownService<MonthCardRequest, Long> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private MonthRecordDao monthRecordDao;
    @Autowired
    private MonthProductDao monthProductDao;
    @Autowired
    private MonthCarService monthCarService;
    @Autowired
    private SendMsgServiceImpl sendMsgService;
    @Autowired
    private OrderService orderService;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private ClearListServiceImpl clearListService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private BatchDownBizHandle<MonthCardDto> batchDownBizHandle;
    @Value("${monthcard.down.alwaysDownParks}")
    private String alwaysDownParks;
    /**
     * 分时段卡
     */
    private static final int CARD_TYPE_2 = 2;
    /**
     * 月卡有效期开始前多久下发
     */
    private static final int BEFORE_TIME = 60 * 60;
    /**
     * 场内办卡延时多少秒下发
     */
    private static final int OPEN_IN_PARK_DELAY_TIME = 3600;
    //月卡数据头部
    private static final String[] MONTHCARD_HEADERS = {"id","cardId","plateNum","cardOwner","plotCount","cardType","startDate","endDate","startTime","endTime"};

    @Override
    public ObjectResponse<Long> send(SendRequest sendRequest, String parkCode) {
        Long serviceId = sendRequest.getServiceId();
        SendMessage message = SendMessage.builder()
                .type(ONE)
                .build();
        MonthRecord monthRecord = monthRecordDao.selectById(serviceId);
        if (monthRecord == null){
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(ONE, "月卡操作记录不存在"));
        }
        /*Integer plotCount = monthRecord.getPlotCount();
        String plateNum = monthRecord.getPlateNum();
        /**
         * 如果车位数和车牌数不一样，暂不下发
         */
        /*int plateNums = plateNum.split(",").length;
        if (plateNums > plotCount){
            log.info("<端云-月卡下发> 多车位多车暂不下发，参数:{}", monthRecord);
            return ObjectResponse.failed(CodeConstants.ERROR_406, "多车位多车暂不下发");
        }*/
        MonthProduct monthProduct = monthProductDao.selectById(monthRecord.getProductId());
        int cardType = monthProduct.getCardType();
        if (cardType == CARD_TYPE_2){
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(ONE, "错时月卡，当前数据无需进行下发"));
        }
        List<String> parkCodeList = Arrays.asList(alwaysDownParks.split(","));
        boolean alwaysDown = parkCodeList.contains(parkCode);
        if (alwaysDown) {
            Long fireTime = pauseValidate(monthRecord);
            if (fireTime != null) {
                message.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                        .failType(MOOD_SUSPEND_TIME)
                        .build()));
                return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(message), fireTime);
            }
            try {
                MonthCardRequest monthCardRequest = buildRequest(monthRecord, cardType);
                String regionIds = monthRecord.getRegionId();
                return send(sendRequest, parkCode, monthCardRequest, regionIds, true, monthProduct.getDiffWorkday());
            }catch (ResponseBodyException re){
                return ObjectResponse.failed(re.getErrCode(), re.getMessage());
            }
        }
        /*
         * 如果当前时间未到月卡有效期，则先不下发
         */
        if (monthRecord.getStartTime() == null) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(ONE, "生效日期不合法"));
        }
        long startTime = monthRecord.getStartTime().getTime() / 1000;
        if (startTime - BEFORE_TIME > DateTools.unixTimestamp()){
            log.info("<端云-月卡下发> 未到月卡有效期延迟下发, 参数:{}", monthRecord);
            long fireTime = startTime - BEFORE_TIME;
            message.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                    .failType(MOOD_TAKE_EFFECT_TIME)
                    .build()));
            return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(message), fireTime);
        }
        Long fireTime = pauseValidate(monthRecord);
        if (fireTime != null) {
            message.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                    .failType(MOOD_SUSPEND_TIME)
                    .build()));
            return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(message), fireTime);
        }
        try {
            MonthCardRequest monthCardRequest = buildRequest(monthRecord, cardType);
            String regionIds = monthRecord.getRegionId();

            return send(sendRequest, parkCode, monthCardRequest, regionIds, monthProduct.getDiffWorkday());
        }catch (ResponseBodyException re){
            return ObjectResponse.failed(re.getErrCode(), re.getMessage());
        }

    }

    private Long pauseValidate(MonthRecord monthRecord) {
        if (monthRecord.getCardOpertype() == 7 && monthRecord.getCardstopStart() != null) {
            long pauseStartTime = monthRecord.getCardstopStart().getTime() / 1000;
            if (pauseStartTime - BEFORE_TIME > DateTools.unixTimestamp()) {
                log.info("<端云-月卡下发> 未到月卡暂停开始时间延迟下发，参数:{}", monthRecord);
                return pauseStartTime - BEFORE_TIME;
            }
        }
        return null;
    }

    @Override
    public ObjectResponse<Long> batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode) {
        List<Long> ids = sendRequestList.stream().map(SendRequest::getServiceId).collect(Collectors.toList());
        List<MonthRecord> list = monthRecordDao.selectBatchIds(ids);
        if (CollectionUtils.isEmpty(list)) {
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }
        Map<Long, Long> recordIdMonthIdMap = list.stream().collect(
                Collectors.toMap(MonthRecord::getId, MonthRecord::getMonthId, (a, b) -> a));
        sendRequestList.forEach(sendRequest -> {
            ClearListRequest clearListRequest = new ClearListRequest();
            Long monthId = recordIdMonthIdMap.get(sendRequest.getServiceId());
            if (monthId != null) {
                clearListRequest.setId(monthId.intValue());
                clearListRequest.setDataType(QueryListRequest.DataType.月卡.getVal());
                sendRequest.setReqServiceType(P2cDownCmdEnum.名单数据清空.getCmdType());
                sendMsgService.send2Park(sendRequest, parkCode, clearListRequest);
                ThreadUtil.sleep(200);
            }
        });
        return ObjectResponse.success();
    }
    /**
     * 执行下发
     * @param sendRequest 下发参数
     * @param parkCode 车场编号
     * @param monthCardRequest 月卡记录
     * @return 响应
     */
    private ObjectResponse<Long> send(SendRequest sendRequest, String parkCode, MonthCardRequest monthCardRequest, String regionIds, Integer diffWorkday) {
        return send(sendRequest, parkCode, monthCardRequest, regionIds, false, diffWorkday);
    }
    private ObjectResponse<Long> send(SendRequest sendRequest, String parkCode, MonthCardRequest monthCardRequest, String regionIds,
                                      boolean alwaysDown, Integer diffWorkday) {
        SendMessage sendMessage = SendMessage.builder()
                .type(ONE)
                .build();
        List<MonthCardRequest.Details> details = monthCardRequest.getDetails();
        //新开卡时
        if (!alwaysDown && monthCardRequest.getCardOperType() == 1) {
            for (MonthCardRequest.Details mp : details) {
                //判断有没有在场内车辆，如果有，则延迟下发
                String plateNum = mp.getPlateNum();
                ObjectResponse<OrderInfo> infoObjectResponse = orderService.findInPark(plateNum, parkCode);
                if (ObjectResponse.isSuccess(infoObjectResponse)
                        && DateTools.getParse(mp.getStartDate()).getTime() / 1000 > infoObjectResponse.getData().getEnterTime()
                ) {
                    OrderInfo orderInfo = infoObjectResponse.getData();
                    //异常离场超过1天的马上下发
                    if (OrderStatusConstants.EXCEPTION == orderInfo.getServiceStatus()
                            && (orderInfo.getExitTime() == null || (DateTools.unixTimestamp() - orderInfo.getExitTime()) > (3600 * 24))) {
                        continue;
                    }
                    sendMessage.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                            .failType(DEFAULT)
                            .fixedDisplay("场内办卡-待离场后下发")
                            .build()));
                    return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(sendMessage),
                            DateTools.unixTimestamp() + OPEN_IN_PARK_DELAY_TIME);
                }
            }
        }
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(sendRequest.getParkId());
        Integer isInterior = parkObjectResponse.getData().getIsInterior();
        //场中场车场并且不是全车场月卡
        if (Integer.valueOf(1).equals(isInterior)
                && StringUtil.isNotEmpty(regionIds) && !"0".equals(regionIds)) {
            String[] regionIdArr = regionIds.split(",");
            ObjectResponse<List<ParkDevice>> listObjectResponse = parkDeviceService.getDeviceListByParkRegionId(sendRequest.getParkId(), Integer.valueOf(regionIdArr[0]), 1);
            ObjectResponse.notError(listObjectResponse, "区域ID[" + regionIdArr[0] + "]下未添加设备");
            List<ParkDevice> parkDevices = listObjectResponse.getData().stream()
                    .filter(pd -> P2cVersionEnum.版本17.getIndex() > P2cVersionEnum.getIndex(pd.getProtocolVer()))
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(parkDevices)) {
                ObjectResponse<Void> objectResponse = clearListService.execute(sendRequest.getParkId(), parkCode, QueryListRequest.DataType.月卡.getVal(), details.get(0).getCardId());
                log.info("[分区月卡下发] 已清除[{}]，响应：{}", details.get(0).getCardId(), objectResponse);
                for (int i = 0; i < regionIdArr.length; i++) {
                    String regionId = regionIdArr[i];
                    if (i != 0) {
                        listObjectResponse = parkDeviceService.getDeviceListByParkRegionId(sendRequest.getParkId(), Integer.valueOf(regionId), 1);
                    }
                    if (!ObjectResponse.isSuccess(listObjectResponse)) {
                        continue;
                    }
                    List<ParkDevice> parkDeviceList = listObjectResponse.getData();
                    for (ParkDevice parkDevice : parkDeviceList) {
                        Integer channelId = parkDevice.getChannelId();
                        if (channelId == null) {
                            continue;
                        }
                        /*
                         * 不给大场出口下发月卡，小场要收费，防止相机直接开闸
                         */
                        ObjectResponse<ParkInoutdevice> inoutdeviceObjectResponse = parkService.getInoutDeviceById(parkDevice.getChannelId().longValue());
                        if (!ObjectResponse.isSuccess(inoutdeviceObjectResponse)) {
                            continue;
                        }
                        ParkInoutdevice parkInoutdevice = inoutdeviceObjectResponse.getData();
                        //大场出口
                        if (parkInoutdevice.getInandoutType() == 2 && parkInoutdevice.getIsMaster() == 1) {
                            continue;
                        }
                        String serialNumber = parkDevice.getSerialNumber();
                        Message<MonthCardRequest> message = new Message<>(sendRequest.getTaskId(), sendRequest.getParkId(),
                                sendRequest.getServiceType(), sendRequest.getServiceId(), monthCardRequest);
                        p2cDownHandle.send(parkCode, serialNumber, message);
                    }
                }
                return ObjectResponse.success();
            }
        }
        ObjectResponse<Void> objectResponse;
        if (Integer.valueOf(1).equals(diffWorkday)){
            objectResponse = sendMsgService.send2ExcludeMasterExit(sendRequest, parkCode, monthCardRequest);
        } else {
            objectResponse = sendMsgService.send2Park(sendRequest, parkCode, monthCardRequest);
        }
        return ObjectResponse.instance(objectResponse.getCode(), objectResponse.getMsg(), null);

    }

    /**
     * 构建请求参数
     *
     * @param monthRecord 月卡记录
     * @return 响应
     */
    private MonthCardRequest buildRequest(MonthRecord monthRecord, int cardType) {
        Integer cardOpertype = monthRecord.getCardOpertype();
        MonthInfo monthInfo = monthCarService.selectMonthInfotById(monthRecord.getMonthId().intValue());
        if (monthInfo == null) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(ONE, "未找到月卡信息"));
        }
        if (Integer.valueOf(1).equals(monthInfo.getDeleteStatus()) && cardOpertype != 9) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(ONE, "当前月卡已被删除，无需下发"));
        }
        if (Integer.valueOf(0).equals(monthInfo.getDeleteStatus())) {
            int cardStatus = monthInfo.getCardStatus();
            if (Arrays.asList(3, 4, 5, 6).contains(cardStatus) && Arrays.asList(1, 2, 3, 6, 8).contains(cardOpertype)) {
                log.info("月卡操作和当前状态不对应, 操作记录[{}], 月卡信息[{}]", monthRecord, monthInfo);
                throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(ONE, "当前月卡已失效，无需下发"));
            }
        }
        MonthCardRequest monthCardRequest = new MonthCardRequest();
        if (cardOpertype == 6 || cardOpertype == 8){//解冻、恢复
            monthCardRequest.setCardOperType(1);
        }else if (cardOpertype == 5 || cardOpertype == 7){//冻结、暂停
            monthCardRequest.setCardOperType(4);
        }else{
            monthCardRequest.setCardOperType(cardOpertype);
        }
        List<MonthCardRequest.Details> list = new ArrayList<>();
        MonthCardRequest.Details details = new MonthCardRequest.Details();

        details.setPlateNum(monthRecord.getPlateNum());
        details.setCardId(monthRecord.getMonthId().intValue());
        details.setCardOwner(StringUtils.isBlank(monthRecord.getCardOwner()) ? "default" : monthRecord.getCardOwner());
        details.setStartDate(DateFormatUtils.format(monthRecord.getStartTime(), "yyyy-MM-dd 00:00:00"));
        details.setEndDate(DateFormatUtils.format(monthRecord.getEndTime(), "yyyy-MM-dd 23:59:59"));
        details.setPlotCount(monthRecord.getPlotCount());
        details.setRegionIds(StringUtils.isBlank(monthRecord.getRegionId()) ? "0" : monthRecord.getRegionId());

        details.setCardType(cardType);
        list.add(details);
        monthCardRequest.setDetails(list);
        return monthCardRequest;
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.月卡.getCmdType());
    }
    @Override
    public ObjectResponse<Long> batchSend(BatchSendVO vo){
        String parkCode = vo.getParkCode();
        Long parkId = vo.getParkId();

        ObjectResponse<List<MonthCardDto>> mothCardDtoRet = monthCarService.getValidByParkId(parkId);
        if (mothCardDtoRet.getCode().equals(CodeConstants.ERROR_404)){
            log.info("[批量月卡下发] 没有符合条件的月卡,忽略下发,车场编号[{}]", parkCode);
            return ObjectResponse.success();
        }
        if (!ObjectResponse.isSuccess(mothCardDtoRet)) {
            return ObjectResponse.failed(mothCardDtoRet.getCode(), mothCardDtoRet.getMsg());
        }
        List<MonthCardDto> monthCardDtos = mothCardDtoRet.getData();
        for (MonthCardDto monthCardDto : monthCardDtos) {
            if (StringUtils.isBlank(monthCardDto.getCardOwner())) {
                monthCardDto.setCardOwner("default");
            }
        }
        boolean hasDiffWorkday = monthCardDtos.stream().anyMatch(mcd -> Integer.valueOf(1).equals(mcd.getDiffWorkday()));
        if (hasDiffWorkday) {
            List<BatchSendVO.SubTaskInfo> subTaskInfos = vo.getSubTaskInfos();
            Map<Long, List<BatchSendVO.SubTaskInfo>> channelIdMap = subTaskInfos.stream().collect(Collectors.groupingBy(BatchSendVO.SubTaskInfo::getChannelId));
            List<Long> channelIds = subTaskInfos.stream().map(BatchSendVO.SubTaskInfo::getChannelId).collect(Collectors.toList());
            List<ParkInoutdevice> parkInoutDevices = parkService.getInoutDeviceByIds(channelIds);

            //非主出口下发所有月卡
            BatchSendVO vo1 = new BatchSendVO();
            BeanUtils.copyProperties(vo, vo1);
            List<BatchSendVO.SubTaskInfo> notMasterExitSubTask = new ArrayList<>();
            parkInoutDevices.stream().filter(p -> !(p.getIsMaster() == 1 && p.getInandoutType() == 2)).forEach(p -> {
                notMasterExitSubTask.addAll(channelIdMap.get(p.getId()));
            });
            if (CollectionUtils.isNotEmpty(notMasterExitSubTask)) {
                vo1.setSubTaskInfos(notMasterExitSubTask);
                List<Object> collect = monthCardDtos.stream().map(MonthCardDto::getCardId).collect(Collectors.toList());
                batchDownBizHandle.batchDown(vo1, monthCardDtos, collect, MONTHCARD_HEADERS, P2cDownCmdEnum.月卡.getCmdType());
            }

            //主出口下发内容不包括区分工作日非工作日月卡
            BatchSendVO vo2 = new BatchSendVO();
            BeanUtils.copyProperties(vo, vo2);
            List<BatchSendVO.SubTaskInfo> masterExitSubTask = new ArrayList<>();
            parkInoutDevices.stream().filter(p -> p.getIsMaster() == 1 && p.getInandoutType() == 2).forEach(p -> {
                masterExitSubTask.addAll(channelIdMap.get(p.getId()));
            });
            if (CollectionUtils.isNotEmpty(masterExitSubTask)) {
                vo2.setSubTaskInfos(masterExitSubTask);
                List<MonthCardDto> monthCardDtos2 = monthCardDtos.stream().filter(monthCardDto -> !Integer.valueOf(1).equals(monthCardDto.getDiffWorkday()))
                        .collect(Collectors.toList());
                if (CollectionUtils.isEmpty(monthCardDtos2)) {
                    log.info("区分工作日非作日月卡过滤后无待下发数据, 参数[{}]", vo);
                    return ObjectResponse.success();
                }
                List<Object> collect2 = monthCardDtos2.stream().map(MonthCardDto::getCardId).collect(Collectors.toList());
                batchDownBizHandle.batchDown(vo2, monthCardDtos2, collect2, MONTHCARD_HEADERS, P2cDownCmdEnum.月卡.getCmdType());
            }
        } else {
            List<Object> collect = monthCardDtos.stream().map(MonthCardDto::getCardId).collect(Collectors.toList());
            batchDownBizHandle.batchDown(vo, monthCardDtos, collect, MONTHCARD_HEADERS, P2cDownCmdEnum.月卡.getCmdType());
        }
        return ObjectResponse.success();
    }
    @Override
    public ObjectResponse<Long> batchSendRepeat(BatchSendRepeatVO vo){
        String failIds = vo.getFailIds();
        String parkCode = vo.getParkCode();
        ObjectResponse<List<MonthCardDto>> monthCardDtoRet;
        if (StrUtil.isNotBlank(vo.getFailIds())){
            monthCardDtoRet = monthCarService.getValidByIds(Arrays.asList(failIds.split(",")));
        }else{
            monthCardDtoRet = monthCarService.getValidByParkId(vo.getParkId());
        }
        if (monthCardDtoRet.getCode().equals(CodeConstants.ERROR_404)) {
            log.info("[批量月卡下发-重发] 没有符合条件的月卡,忽略下发,车场编号[{}]", parkCode);
            //TODO 更新进度为100%
            return ObjectResponse.success();
        }
        if (!ObjectResponse.isSuccess(monthCardDtoRet)) {
            return ObjectResponse.instance(monthCardDtoRet.getCode(), monthCardDtoRet.getMsg(), null);
        }
        List<MonthCardDto> monthCardDtos = monthCardDtoRet.getData();
        boolean hasDiffWorkday = monthCardDtos.stream().anyMatch(mcd -> Integer.valueOf(1).equals(mcd.getDiffWorkday()));
        if (hasDiffWorkday) {
            ObjectResponse<ParkInoutdevice> objectResponse = parkService.getInoutDeviceById(vo.getChannelId());
            if (!ObjectResponse.isSuccess(objectResponse)) {
                return ObjectResponse.failed(CodeConstants.ERROR_404, "通道ID不存在");
            }
            ParkInoutdevice data = objectResponse.getData();
            //主出口不下发区分工作日非工作日的月卡
            if (data.getIsMaster() == 1 && data.getInandoutType() == 2) {
                monthCardDtos = monthCardDtos.stream().filter(monthCardDto -> !Integer.valueOf(1).equals(monthCardDto.getDiffWorkday()))
                        .collect(Collectors.toList());
            }
            if (CollectionUtils.isEmpty(monthCardDtos)) {
                log.info("区分工作日非作日月卡过滤后无待下发数据, 参数[{}]", vo);
                return ObjectResponse.success();
            }
        }
        List<Object> collect = monthCardDtos.stream().map(MonthCardDto::getCardId).collect(Collectors.toList());
        batchDownBizHandle.repeatBatch(vo, monthCardDtos, collect, MONTHCARD_HEADERS, P2cDownCmdEnum.月卡.getCmdType());
        return ObjectResponse.success();
    }

}
