package com.icetech.park.service.down.p2c.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.basics.domain.entity.device.FleetModeLog;
import com.icetech.cloudcenter.api.park.FleetModeLogService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.request.p2c.FleetModeRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.validator.Validator;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service("p2cFleetModeServiceImpl")
@Slf4j
public class FleetModeServiceImpl extends AbstractService implements ResponseService<String>, DownService<FleetModeRequest, String> {
    @Autowired
    private P2cDownHandle p2cDownHandle;

    @Autowired
    private ParkService parkService;

    @Autowired
    private CacheHandle cacheHandle;

    @Autowired
    private FleetModeLogService fleetModeLogService;

    public ObjectResponse<Void> send(FleetModeRequest request, String sn, Long parkId) {
        if (!Validator.validate(request)) {
            return ObjectResponse.failed(CodeConstants.ERROR_400);
        }
        TokenDeviceVo deviceInfo = cacheHandle.getDeviceInfo(sn);
        if (deviceInfo == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_3003);
        }
        // 版本校验
        if (P2cVersionEnum.getIndex(deviceInfo.getVersion()) < P2cVersionEnum.版本19.getIndex()) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "当前设备版本不支持");
        }
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(parkId);
        if (!ObjectResponse.isSuccess(parkObjectResponse)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "车场不存在");
        }
        Park park = parkObjectResponse.getData();
        Message<FleetModeRequest> message = new Message<>(parkId, P2cDownCmdEnum.车队模式下发.getCmdType(), request);
        String messageId = p2cDownHandle.send(park.getParkCode(), sn, message);
        if (messageId == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_3005);
        }
        log.info("[车队模式]下发成功");
        // 获取响应结果
        ObjectResponse<String> responseObj = cacheHandle.getResponseFromRedis(messageId, 4000L);
        log.info("[车队模式]下发响应结果:{}", responseObj);
        if (ObjectResponse.isSuccess(responseObj)) {
            FleetModeLog fleetModeLog = fleetModeLogService.selectLimitOne(Wrappers.lambdaQuery(FleetModeLog.class)
                    .eq(FleetModeLog::getParkId, parkId)
                    .eq(FleetModeLog::getChannelCode, deviceInfo.getInandoutCode())
                    .eq(FleetModeLog::getDeleted, Boolean.FALSE));
            // 没有直接新增
            if (fleetModeLog == null) {
                fleetModeLogService.addFleetModeLog(FleetModeLog.builder()
                        .parkId(parkId)
                        .channelCode(deviceInfo.getInandoutCode())
                        .messageId(messageId)
                        .type(request.getEnable())
                        .deleted(Boolean.FALSE)
                        .build());
            } else {
                // 直接修改
                fleetModeLog.setMessageId(messageId);
                fleetModeLog.setType(request.getEnable());
                fleetModeLogService.modifyFleetModeLog(fleetModeLog);
            }
        }
        return ObjectResponse.success();
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String sn) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.车队模式下发.getCmdType());
    }
}
