package com.icetech.park.service.down.p2c.impl;

import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.StrUtil;
import com.icetech.basics.api.BlacklistService;
import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.entity.Blacklist;
import com.icetech.cloudcenter.domain.enumeration.P2cDownCmdEnum;
import com.icetech.cloudcenter.domain.request.p2c.BlacklistRequest;
import com.icetech.cloudcenter.domain.request.p2c.ClearListRequest;
import com.icetech.cloudcenter.domain.request.p2c.QueryListRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.BatchSendRepeatVO;
import com.icetech.cloudcenter.domain.vo.BatchSendVO;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.down.p2c.SendMsgServiceImpl;
import com.icetech.park.service.handle.BatchDownBizHandle;
import com.icetech.park.service.handle.P2cDownHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;

@Service("p2cBlacklistServiceImpl")
@Slf4j
public class BlacklistServiceImpl extends AbstractService implements ResponseService<String>, DownService<List<BlacklistRequest>, Void> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private BlacklistService blacklistService;
    @Autowired
    private SendMsgServiceImpl sendMsgService;
    @Autowired
    private BatchDownBizHandle<BlacklistRequest> batchDownBizHandle;
    //数据头部
    private static String[] HEADERS = {"id","blackId","plateNum"};

    @Override
    public ObjectResponse<Void> send(SendRequest sendRequest, String parkCode){

        ObjectResponse<Blacklist> blacklistResponse = blacklistService.getById(sendRequest.getServiceId());
        if (!ObjectResponse.isSuccess(blacklistResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(TextConstant.ONE, "黑名单数据不存在"));
        }
        ObjectResponse.notError(blacklistResponse);
        Blacklist blacklist = blacklistResponse.getData();
        //黑名单车辆删除，3表示可进可出
        if (blacklist.getType() == 3 || blacklist.getStatus() == 1){
            ClearListRequest clearListRequest = new ClearListRequest();
            clearListRequest.setDataType(QueryListRequest.DataType.黑名单.getVal());
            clearListRequest.setId(blacklist.getId());
            sendRequest.setReqServiceType(P2cDownCmdEnum.名单数据清空.getCmdType());
            return sendMsgService.send2Park(sendRequest, parkCode, clearListRequest);
        }else{
            BlacklistRequest blacklistRequest = new BlacklistRequest();
            blacklistRequest.setBlackId(blacklist.getId());
            blacklistRequest.setPlateNum(blacklist.getPlate());
            List<BlacklistRequest> list = new ArrayList<>();
            list.add(blacklistRequest);
            return sendMsgService.send2Park(sendRequest, parkCode, list);
        }
    }

    private ObjectResponse<Void> send(Long parkId, String parkCode, List<BlacklistRequest> blacklist) {

        String[] messageIdArr;
        Message<List<BlacklistRequest>> message = new Message<>(parkId, P2cDownCmdEnum.黑名单下发.getCmdType(), blacklist);
        messageIdArr = p2cDownHandle.send2Park(parkCode, message);

        if (ArrayUtils.isEmpty(messageIdArr)){
            return ObjectResponse.failed(CodeConstants.ERROR_3003);
        }else{
            return ObjectResponse.success();
        }
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, P2cDownCmdEnum.黑名单下发.getCmdType());
    }

    @Override
    public ObjectResponse<Void> batchSend(BatchSendVO vo){
        Long parkId = vo.getParkId();
        String parkCode = vo.getParkCode();
        ObjectResponse<List<Blacklist>> blacklistRet = blacklistService.getValidByParkId(parkId);
        if (blacklistRet.getCode().equals(CodeConstants.ERROR_404)){
            log.info("<批量黑名单下发> 没有符合比对条件的黑名单车，忽略下发，车场编号：{}", parkCode);
            return ObjectResponse.success();
        }else{
            if(ObjectResponse.isSuccess(blacklistRet)){
                List<Blacklist> blacklists = blacklistRet.getData();
                List<BlacklistRequest> list = new ArrayList<>();
                for (Blacklist blacklist : blacklists) {
                    BlacklistRequest blacklistRequest = new BlacklistRequest();
                    blacklistRequest.setBlackId(blacklist.getId());
                    blacklistRequest.setPlateNum(blacklist.getPlate());
                    list.add(blacklistRequest);
                }
                List<Object> idList = list.stream().map(BlacklistRequest::getBlackId).collect(Collectors.toList());
                batchDownBizHandle.batchDown(vo, list, idList, HEADERS, P2cDownCmdEnum.黑名单下发.getCmdType());
                return ObjectResponse.success();
            }else{
                return ObjectResponse.failed(blacklistRet.getCode(), blacklistRet.getMsg());
            }
        }
    }

    @Override
    public ObjectResponse<Void> batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode) {
        List<Long> ids = sendRequestList.stream().map(SendRequest::getServiceId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(ids)) {
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }
        sendRequestList.forEach(sendRequest -> {
            ClearListRequest clearListRequest = new ClearListRequest();
            clearListRequest.setId(sendRequest.getServiceId().intValue());
            clearListRequest.setDataType(QueryListRequest.DataType.黑名单.getVal());
            sendRequest.setReqServiceType(P2cDownCmdEnum.名单数据清空.getCmdType());
            sendMsgService.send2Park(sendRequest, parkCode, clearListRequest);
            ThreadUtil.sleep(200);
        });
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Void> batchSendRepeat(BatchSendRepeatVO vo){
        String parkCode = vo.getParkCode();
        ObjectResponse<List<Blacklist>> blacklistRet;
        if (StrUtil.isNotBlank(vo.getFailIds())) {
            blacklistRet = blacklistService.getValidByIds(Arrays.asList(vo.getFailIds().split(",")));
        }else{
            blacklistRet = blacklistService.getValidByParkId(vo.getParkId());
        }
        if (blacklistRet.getCode().equals(CodeConstants.ERROR_404)){
            log.info("<批量黑名单下发-重发> 没有符合比对条件的黑名单车，忽略下发，车场编号：{}", parkCode);
            //todo
            return ObjectResponse.success();
        }else{
            if(ObjectResponse.isSuccess(blacklistRet)){
                List<Blacklist> blacklists = blacklistRet.getData();
                List<BlacklistRequest> list = new ArrayList<>();
                for (Blacklist blacklist : blacklists) {
                    BlacklistRequest blacklistRequest = new BlacklistRequest();
                    blacklistRequest.setBlackId(blacklist.getId());
                    blacklistRequest.setPlateNum(blacklist.getPlate());
                    list.add(blacklistRequest);
                }
                List<Object> idList = list.stream().map(BlacklistRequest::getBlackId).collect(Collectors.toList());
                batchDownBizHandle.repeatBatch(vo, list, idList, HEADERS, P2cDownCmdEnum.黑名单下发.getCmdType());
                return ObjectResponse.success();
            }else{
                return ObjectResponse.failed(blacklistRet.getCode(), blacklistRet.getMsg());
            }
        }
    }
}
