package com.icetech.park.service.down.full.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.github.pagehelper.util.StringUtil;
import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.SendMessage;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.cloudcenter.api.month.MonthCarService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.third.SendInfoService;
import com.icetech.cloudcenter.domain.enumeration.FullCloudDownCmdEnum;
import com.icetech.cloudcenter.domain.request.full.WhiteListOperatorRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.fee.dao.monthcar.MonthProductDao;
import com.icetech.fee.dao.monthcar.MonthRecordDao;
import com.icetech.fee.domain.entity.monthcar.MonthInfo;
import com.icetech.fee.domain.entity.monthcar.MonthProduct;
import com.icetech.fee.domain.entity.monthcar.MonthRecord;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.down.full.FullCloudSendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.FullCloudDownHandle;
import com.icetech.third.domain.entity.third.SendInfo;
import com.icetech.third.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.DEFAULT;
import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.MOOD_SUSPEND_TIME;
import static com.icetech.basics.constants.TextConstant.MOOD_TAKE_EFFECT_TIME;
import static com.icetech.basics.constants.TextConstant.THREE;

/**
 * 全天月卡下发
 * @author fangct
 */
@Service
@RefreshScope
@Slf4j
public class FullCloudWhiteListOperatorServiceImpl implements ResponseService<String>, DownService<WhiteListOperatorRequest, Long> {

    @Autowired
    private FullCloudDownHandle downHandle;
    @Autowired
    private MonthRecordDao monthRecordDao;
    @Autowired
    private MonthProductDao monthProductDao;
    @Autowired
    private MonthCarService monthCarService;
    @Autowired
    private FullCloudSendMsgServiceImpl sendMsgService;
    @Autowired
    private OrderService orderService;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private ParkService parkService;
    @Resource
    private SendInfoService sendInfoService;
    @Value("${monthcard.down.alwaysDownParks}")
    private String alwaysDownParks;
    //分时段卡
    private static final int CARD_TYPE_2 = 2;
    //月卡有效期开始前多久下发
    private static final int BEFORE_TIME = 60 * 60;
    //场内办卡延时多少秒下发
    private static final int OPEN_IN_PARK_DELAY_TIME = 3600;

    @Override
    public ObjectResponse<Long> send(SendRequest sendRequest, String parkCode) {
        Long serviceId = sendRequest.getServiceId();
        SendMessage message = SendMessage.builder()
                .type(THREE)
                .build();
        MonthRecord monthRecord = monthRecordDao.selectById(serviceId);
        if (monthRecord == null) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, "月卡操作记录不存在"));
        }
        Integer plotCount = monthRecord.getPlotCount();
        String plateNum = monthRecord.getPlateNum();
        //如果车位数和车牌数不一样，暂不下发
        int plateNums = plateNum.split(",").length;
        if (plateNums > plotCount) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "多车位多车，当前数据无需进行下发"));
        }
        MonthProduct monthProduct = monthProductDao.selectById(monthRecord.getProductId());
        int cardType = monthProduct.getCardType();
        if (cardType == CARD_TYPE_2) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "错时月卡，当前数据无需进行下发"));
        }
        MonthInfo monthInfo = monthCarService.selectMonthInfotById(monthRecord.getMonthId().intValue());
        if (monthInfo == null) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "未找到月卡信息"));
        }
        if (Integer.valueOf(1).equals(monthInfo.getDeleteStatus())) {
            WhiteListOperatorRequest whiteListOperatorRequest = new WhiteListOperatorRequest();
            whiteListOperatorRequest.setOperator_type("delete");
            if (monthRecord.getPlateNum().contains(",")) {
                whiteListOperatorRequest.setPlate(Arrays.asList(monthRecord.getPlateNum().split(",")));
            } else {
                whiteListOperatorRequest.setPlate(monthRecord.getPlateNum());
            }

            return sendMsgService.send2Park(sendRequest, parkCode, whiteListOperatorRequest);
        }
        List<String> parkCodeList = Arrays.asList(alwaysDownParks.split(","));
        boolean alwaysDown = parkCodeList.contains(parkCode);
        if (alwaysDown) {
            Long fireTime = pauseValidate(monthRecord);
            if (fireTime != null) {
                message.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                        .failType(MOOD_SUSPEND_TIME)
                        .build()));
                return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(message), fireTime);
            }
            try {
                List<WhiteListOperatorRequest> whiteListOperatorRequest = buildRequest(monthRecord, monthInfo);
                String regionIds = monthRecord.getRegionId();
                return send(sendRequest, parkCode, whiteListOperatorRequest, regionIds, true, monthProduct.getDiffWorkday(), monthRecord);
            } catch (ResponseBodyException re) {
                return ObjectResponse.failed(re.getErrCode(), re.getMessage());
            }
        }
        /*
         * 如果当前时间未到月卡有效期，则先不下发
         */
        long startTime = monthRecord.getStartTime().getTime() / 1000;
        if (startTime - BEFORE_TIME > DateTools.unixTimestamp()) {
            log.info("<端云-月卡下发> 未到月卡有效期延迟下发, 参数:{}", monthRecord);
            long fireTime = startTime - BEFORE_TIME;
            message.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                    .failType(MOOD_TAKE_EFFECT_TIME)
                    .build()));
            return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(message), fireTime);
        }
        Long fireTime = pauseValidate(monthRecord);
        if (fireTime != null) {
            message.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                    .failType(MOOD_SUSPEND_TIME)
                    .build()));
            return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(message), fireTime);
        }
        try {
            List<WhiteListOperatorRequest> whiteListOperatorRequests = buildRequest(monthRecord, monthInfo);
            String regionIds = monthRecord.getRegionId();

            return send(sendRequest, parkCode, whiteListOperatorRequests, regionIds, monthProduct.getDiffWorkday(), monthRecord);
        } catch (ResponseBodyException re) {
            return ObjectResponse.failed(re.getErrCode(), re.getMessage());
        }

    }

    private Long pauseValidate(MonthRecord monthRecord) {
        if (monthRecord.getCardOpertype() == 7 && monthRecord.getCardstopStart() != null) {
            long pauseStartTime = monthRecord.getCardstopStart().getTime() / 1000;
            if (pauseStartTime - BEFORE_TIME > DateTools.unixTimestamp()) {
                log.info("<端云-月卡下发> 未到月卡暂停开始时间延迟下发，参数:{}", monthRecord);
                return pauseStartTime - BEFORE_TIME;
            }
        }
        return null;
    }

    /**
     * 执行下发
     *
     * @param sendRequest      下发参数
     * @param parkCode         车场编号
     * @param monthCardRequests 月卡记录
     * @return 响应
     */
    private ObjectResponse<Long> send(SendRequest sendRequest, String parkCode, List<WhiteListOperatorRequest> monthCardRequests, String regionIds,
                                      Integer diffWorkday, MonthRecord monthRecord) {
        return send(sendRequest, parkCode, monthCardRequests, regionIds, false, diffWorkday, monthRecord);
    }

    private ObjectResponse<Long> send(SendRequest sendRequest, String parkCode, List<WhiteListOperatorRequest> monthCardRequests, String regionIds,
                                      boolean alwaysDown, Integer diffWorkday, MonthRecord monthRecord) {
        SendMessage sendMessage = SendMessage.builder()
                .type(THREE)
                .build();
        //新开卡时
        WhiteListOperatorRequest monthCardRequest = monthCardRequests.size() == 1 ? monthCardRequests.get(0) : monthCardRequests.get(1);
        if (!alwaysDown && "update_or_add".equals(monthCardRequest.getOperator_type())) {
            //判断有没有在场内车辆，如果有，则延迟下发
            List<WhiteListOperatorRequest.DldbRec> dldbRecList = monthCardRequest.getDldb_rec();
            for (WhiteListOperatorRequest.DldbRec dldbRec : dldbRecList) {
                String plateNum = (String) dldbRec.getPlate();
                ObjectResponse<OrderInfo> infoObjectResponse = orderService.findInPark(plateNum, parkCode);
                if (ObjectResponse.isSuccess(infoObjectResponse)
                        && DateTools.getParse(dldbRec.getEnable_time()).getTime() / 1000 > infoObjectResponse.getData().getEnterTime()
                ) {
                    OrderInfo orderInfo = infoObjectResponse.getData();
                    //异常离场超过1天的马上下发
                    if (OrderStatusConstants.EXCEPTION != orderInfo.getServiceStatus()
                            || (orderInfo.getExitTime() == null || (DateTools.unixTimestamp() - orderInfo.getExitTime()) <= (3600 * 24))) {
                        sendMessage.setDescribes(Collections.singletonList(SendMessage.Describe.builder()
                                .failType(DEFAULT)
                                .fixedDisplay("场内办卡-待离场后下发")
                                .build()));
                        return ObjectResponse.instance(CodeConstants.ERROR_407, JsonUtils.toJson(sendMessage),
                                DateTools.unixTimestamp() + OPEN_IN_PARK_DELAY_TIME);
                    }
                }
            }
        }

        ObjectResponse<Park> parkObjectResponse = parkService.findByParkId(sendRequest.getParkId());
        Integer isInterior = parkObjectResponse.getData().getIsInterior();
        //场中场车场并且不是全车场月卡
        if (Integer.valueOf(1).equals(isInterior) && StringUtil.isNotEmpty(regionIds) && !"0".equals(regionIds)) {
            String[] regionIdArr = regionIds.split(",");
            List<String> sns = new ArrayList<>();
            for (String regionId : regionIdArr) {
                List<ParkDevice> parkDeviceList = parkDeviceService.getDeviceListByParkRegionId(sendRequest.getParkId(), Integer.valueOf(regionId), 1).getData();
                for (ParkDevice parkDevice : parkDeviceList) {
                    String serialNumber = parkDevice.getSerialNumber();
                    sns.add(serialNumber);
                }
            }
            return sendMsgService.send2Devices(String.join(";", sns), sendRequest, parkCode, monthCardRequests.toArray());
        }

        ObjectResponse<Void> objectResponse;
        if (Integer.valueOf(1).equals(diffWorkday)) {
            objectResponse = sendMsgService.send2ExcludeMasterExit(sendRequest, parkCode, monthCardRequests.toArray());
        } else {
            objectResponse = sendMsgService.send2Park(sendRequest, parkCode, monthCardRequests.toArray());
        }
        return ObjectResponse.instance(objectResponse.getCode(), objectResponse.getMsg(), null);

    }
    @Override
    public ObjectResponse<Long> batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode){
        for (SendRequest sendRequest : sendRequestList) {
            try {
                ObjectResponse<Long> objectResponse = send(sendRequest, parkCode);
                if (CodeConstants.ERROR_407.equals(objectResponse.getCode())) {
                    SendInfo sendInfo = new SendInfo();
                    sendInfo.setId(sendRequest.getTaskId());
                    sendInfo.setStatus(SendInfo.StatusEnum._DELAY.getCode());
                    sendInfo.setRemark(objectResponse.getMsg());
                    sendInfo.setNextFireTime(objectResponse.getData().intValue());
                    sendInfoService.update(sendInfo);
                }
            } catch (ResponseBodyException e) {
                log.warn("业务异常={}", e.getMessage());
                if (CodeConstants.ERROR_406.equals(e.getErrCode())) {
                    SendInfo sendInfo = new SendInfo();
                    sendInfo.setId(sendRequest.getTaskId());
                    sendInfo.setStatus(SendInfo.StatusEnum._PASS.getCode());
                    sendInfo.setRemark(e.getMessage());
                    sendInfoService.update(sendInfo);
                }
            } catch (Exception e) {
                log.error("e=", e);
            }
        }
        return ObjectResponse.success();
    }
    /**
     * 构建请求参数
     *
     * @param monthRecord 月卡记录
     * @return 响应
     */
    private List<WhiteListOperatorRequest> buildRequest(MonthRecord monthRecord, MonthInfo monthInfo) {
        Integer cardOpertype = monthRecord.getCardOpertype();
        if (Integer.valueOf(1).equals(monthInfo.getDeleteStatus()) && cardOpertype != 9) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "当前月卡已被删除，此操作无需下发"));
        }
        if (Integer.valueOf(0).equals(monthInfo.getDeleteStatus())) {
            int cardStatus = monthInfo.getCardStatus();
            if (Arrays.asList(3, 4, 5, 6).contains(cardStatus) && Arrays.asList(1, 2, 3, 6, 8).contains(cardOpertype)) {
                log.info("月卡操作和当前状态不对应, 操作记录[{}], 月卡信息[{}]", monthRecord, monthInfo);
                throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "当前月卡已失效，此操作无需下发"));
            }
        }

        List<WhiteListOperatorRequest> list = new ArrayList<>();
        //判断有没有修改车牌，如果有并且下发失败，需要下发旧车牌的删除
        if (cardOpertype == 3) {
            List<String> deletePlates = new ArrayList<>();
            List<MonthRecord> monthRecords = monthRecordDao.selectList(Wrappers.lambdaQuery(MonthRecord.class)
                    .eq(MonthRecord::getParkId, monthRecord.getParkId())
                    .eq(MonthRecord::getMonthId, monthRecord.getMonthId())
                    .ne(MonthRecord::getId, monthRecord.getId())
                    .notIn(MonthRecord::getSendStatus, 1));
            List<String> oldPlateNums = monthRecords.stream().map(MonthRecord::getPlateNum).distinct().collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(oldPlateNums)) {
                String joinPlates = String.join(",", oldPlateNums);
                String[] oldList = joinPlates.split(",");
                List<String> newList = Arrays.asList(monthRecord.getPlateNum().split(","));
                for (String oldPlateNum : oldList) {
                    if (!newList.contains(oldPlateNum)) {
                        deletePlates.add(oldPlateNum);
                    }
                }
                if (CollectionUtils.isNotEmpty(deletePlates)) {
                    WhiteListOperatorRequest whiteListOperatorRequest = new WhiteListOperatorRequest();
                    whiteListOperatorRequest.setOperator_type("delete");
                    if (deletePlates.size() == 1) {
                        whiteListOperatorRequest.setPlate(deletePlates.get(0));
                    } else {
                        whiteListOperatorRequest.setPlate(deletePlates);
                    }
                    list.add(whiteListOperatorRequest);
                }
            }
        }
        int cardStatus = monthInfo.getCardStatus();
        WhiteListOperatorRequest whiteListOperatorRequest = new WhiteListOperatorRequest();
        if (cardStatus != 1 && cardStatus != 2) {
            whiteListOperatorRequest.setOperator_type("delete");
            if (monthRecord.getPlateNum().contains(",")) {
                whiteListOperatorRequest.setPlate(Arrays.asList(monthRecord.getPlateNum().split(",")));
            } else {
                whiteListOperatorRequest.setPlate(monthRecord.getPlateNum());
            }
            list.add(whiteListOperatorRequest);
            return list;
        }
        List<WhiteListOperatorRequest.DldbRec> dldbRecList = new ArrayList<>();
        whiteListOperatorRequest.setOperator_type("update_or_add");

        for (String plateNum : monthRecord.getPlateNum().split(",")) {
            WhiteListOperatorRequest.DldbRec dldbRec = new WhiteListOperatorRequest.DldbRec();
            dldbRec.setPlate(plateNum);
            dldbRec.setIndex(monthInfo.getId().intValue());
            dldbRec.setCustomer_id(monthInfo.getId().intValue());
            dldbRec.setCreate_time(DateTools.getFormat(monthRecord.getCreateTime()));
            dldbRec.setEnable_time(DateFormatUtils.format(monthInfo.getStartTime(), "yyyy-MM-dd 00:00:00"));
            dldbRec.setOverdue_time(DateFormatUtils.format(monthInfo.getEndTime(), "yyyy-MM-dd 23:59:59"));
            dldbRec.setEnable(1);
            dldbRec.setTime_seg_enable(0);
            dldbRec.setNeed_alarm(0);
            dldbRecList.add(dldbRec);
        }
        whiteListOperatorRequest.setDldb_rec(dldbRecList);
        list.add(whiteListOperatorRequest);
        return list;
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String sn) {
        downHandle.dealResponse(p2cBaseResponse, parkId, FullCloudDownCmdEnum.名单下发.getCmdType());
    }

}
