package com.icetech.park.service.down.full.impl;

import cn.hutool.core.thread.ThreadUtil;
import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.api.lcd.LedService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.enumeration.FullCloudDownCmdEnum;
import com.icetech.cloudcenter.domain.request.full.WhiteListOperatorRequest;
import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.cloudcenter.domain.response.LedShowDto;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.domain.entity.DownSerialData;
import com.icetech.park.domain.entity.led.LedConfig;
import com.icetech.park.domain.entity.led.LedShow;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.domain.vo.full.DownSerialDataVO;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.down.ExHintService;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.full.FullCloudSendMsgServiceImpl;
import com.icetech.park.service.down.full.controlcard.ControlCardTypeServiceFactory;
import com.icetech.park.service.down.full.controlcard.IControlCardTypeBuilder;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.FullCloudDownHandle;
import com.icetech.park.service.handle.showsay.LedShowHandle;
import com.icetech.third.domain.entity.third.SendInfoRecord;
import com.icetech.third.service.third.MqPushService;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.THREE;
import static com.icetech.park.domain.constant.FullCloudConstants.FULL_BUSY_HINT;
import static com.icetech.park.domain.constant.FullCloudConstants.FULL_DOWN_SHOW_DATE_CHANNEL;
import static com.icetech.park.domain.constant.FullCloudConstants.FULL_SERIAL_DATA;
import static com.icetech.park.domain.constant.FullCloudConstants.FULL_SHOW_STATUS;

/**
 * 纯云下发接口
 *
 * @author wgq
 */
@Slf4j
@Service
@RefreshScope
public class FullCloudSerialDataServiceImpl implements ExHintService, ResponseService<String>, DownService<WhiteListOperatorRequest, Long> {

    @Autowired
    private FullCloudDownHandle downHandle;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private MqPushService mqPushService;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private LedService ledService;
    @Autowired
    private LedShowHandle ledShowHandle;
    @Autowired
    private ParkService parkService;
    @Autowired
    private FullCloudSendMsgServiceImpl sendMsgService;
    @Value("${fullcloud.dtong.lcd.maxWaitTime:1000}")
    private long maxWaitTime;

    @Override
    public ObjectResponse<Long> send(SendRequest sendRequest, String parkCode) {
        log.info("收到通道信息变更：{}", sendRequest);
        //获取车场信息
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(parkCode);
        if (!ObjectResponse.isSuccess(byParkCode)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, "车场信息不存在"));
        }
        //获取屏显语音配置
        Long channelId = sendRequest.getServiceId();
        ObjectResponse<LedConfig> objectResponse = ledService.getLedConfigByChannel(channelId);
        if (!ObjectResponse.isSuccess(objectResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, "屏显语音配置信息不存在"));
        }
        //获取通道信息
        ObjectResponse<ParkInoutdevice> channelResp = parkService.getInoutDeviceById(channelId);
        if (!ObjectResponse.isSuccess(channelResp)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, "通道信息不存在"));
        }
        //设置屏显消息下发
        redisUtils.hPut(FULL_SHOW_STATUS, String.valueOf(channelId), 0);
        //保存通道日期展示信息
        this.saveChannelDateConfig(channelResp.getData(), parkCode);
        //led配置
        LedConfig ledConfig = objectResponse.getData();
        //勿扰模式是否开启
        Integer quietHoursSwitch = ledConfig.getQuietHoursSwitch();
        DownSerialData serialData = new DownSerialData();
        List<DownSerialDataVO> dataVOList = new ArrayList<>();

        IControlCardTypeBuilder controlCardBuilder = ControlCardTypeServiceFactory.getControlCardTypeBuilder(channelResp.getData().getLedcardType());
        //初始化
        List<byte[]> volumeBytes = controlCardBuilder.initControlCard(ledConfig.getVolumeValue(),
                NumberUtils.toPrimitive(quietHoursSwitch) == 1,
                ledConfig.getQuietVolumeValue(), ledConfig.getQuietStartTime(), ledConfig.getQuietEndTime(), channelResp.getData().getTtsType());
        volumeBytes.forEach(bytes -> dataVOList.add(createDownSerialDataVO(bytes)));
        serialData.setSerialData(dataVOList);
        //给通道下发消息
        return sendMsgService.send2Channel(sendRequest, parkCode, channelResp.getData().getInandoutCode(), serialData);
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String sn) {
        downHandle.dealResponse(p2cBaseResponse, parkId, FullCloudDownCmdEnum.语音勿扰模式下发.getCmdType());
        SendInfoRecord<Object> sendinfoRecord = redisUtils.get(RedisKeyConstants.MQ_RECORD_PREFIX + p2cBaseResponse.getMessageId(), SendInfoRecord.class);
        if (sendinfoRecord != null && FullCloudDownCmdEnum.语音勿扰模式下发.getCmdType().equals(sendinfoRecord.getServiceType())) {
            ThreadUtil.sleep(500);
            busyChangeFree(parkId, sn);
        }
    }

    /**
     * 组装参数
     *
     * @param hintRequest 出入场返回
     * @param tokenInfo   设备信息
     * @param ledShowDto  led
     * @return List<DownSerialDataVO>
     */
    private DownSerialData assembleDownSerialData(HintRequest hintRequest, TokenDeviceVo tokenInfo, LedShowDto ledShowDto, boolean busyStatus) {
        log.info("准备下发屏显语音内容 {}", hintRequest);
        List<DownSerialDataVO> dataVOList = new ArrayList<>();
        DownSerialData serialData = new DownSerialData();

        IControlCardTypeBuilder controlCardTypeBuilder = getControlCardBuilder(tokenInfo.getParkId(), tokenInfo.getInandoutCode());

        //屏显
        LedConfig ledConfig = ledService.getLedConfigByChannel(tokenInfo.getId()).getData();
        List<byte[]> showDatas = controlCardTypeBuilder.buildShowBytes(ledShowDto.getLedColor(), ledConfig.getLedRestoreDefaultTime(), hintRequest, busyStatus);
        for (byte[] showData : showDatas) {
            dataVOList.add(createDownSerialDataVO(showData));
        }
        //语音
        if (StringUtils.isNotBlank(hintRequest.getSay())) {
            byte[] sayBytes = controlCardTypeBuilder.buildSayByte(hintRequest.getSay());
            dataVOList.add(createDownSerialDataVO(sayBytes));
        }
        serialData.setSerialData(dataVOList);
        serialData.setIsMultiplePackSleep(controlCardTypeBuilder.isMultiplePackSleep());
        return serialData;
    }

    private IControlCardTypeBuilder getControlCardBuilder(Long parkId, String channelCode) {
        ParkInoutdevice parkInoutdevice = parkService.getChannelByCodeAndParkId(parkId, channelCode);
        return ControlCardTypeServiceFactory.getControlCardTypeBuilder(parkInoutdevice.getLedcardType());
    }

    /**
     * 辅助方法，创建 DownSerialDataVO
     *
     * @param bytes byte
     * @return DownSerialDataVO
     */
    private DownSerialDataVO createDownSerialDataVO(byte[] bytes) {
        String encodedData = Base64.getEncoder().encodeToString(bytes);
        DownSerialDataVO downSerialData = new DownSerialDataVO();
        downSerialData.setData(encodedData);
        downSerialData.setDataLen(bytes.length);
        downSerialData.setSerialChannel(0);
        return downSerialData;
    }

    @Override
    public ObjectResponse<Void> execute(Long parkId, String parkCode, String serialNumber, HintRequest hintRequest) {
        String messageId = showAndSay(parkId, parkCode, serialNumber, hintRequest);
        if (messageId == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_3003);
        } else {
            return ObjectResponse.success();
        }
    }

    /**
     * 展示屏显信息
     *
     * @param parkId       车场ID
     * @param parkCode     车场代码
     * @param serialNumber 设备序列号
     * @param hintRequest  语音显示参数类
     * @return messageId
     */
    @Override
    public String showAndSay(Long parkId, String parkCode, String serialNumber, HintRequest hintRequest) {
        //获取通道信息
        TokenDeviceVo deviceInfo = cacheHandle.getDeviceInfo(serialNumber);
        ObjectResponse<LedShowDto> led;
        //出入通道展示忙时
        if (deviceInfo.getInandoutType() == 1) {
            led = ledService.getLedShowByType(deviceInfo.getId(), LedShow.DisplayTypeEnum.入场显示.type);
        } else {
            led = ledService.getLedShowByType(deviceInfo.getId(), LedShow.DisplayTypeEnum.出场显示.type);
        }

        //组装下发消息
        DownSerialData downSerialData = assembleDownSerialData(hintRequest, deviceInfo, led.getData(), true);

        String messageId = null;
        //多包下发中间增加延迟
        if (Boolean.TRUE.equals(downSerialData.getIsMultiplePackSleep()) && downSerialData.getSerialData().size() > 1) {
            for (DownSerialDataVO serialDatum : downSerialData.getSerialData()) {
                long beforeSendTime = System.currentTimeMillis();
                DownSerialData thisSerialDatum = new DownSerialData();
                thisSerialDatum.setSerialData(Collections.singletonList(serialDatum));
                Message<DownSerialData> message = new Message<>(parkId, FullCloudDownCmdEnum.屏显信息下发.getCmdType(), thisSerialDatum);
                messageId = downHandle.send(parkCode, serialNumber, message);
                //轮询获取串口返回
                getAndWait485Resp(FULL_SERIAL_DATA + serialNumber, beforeSendTime, maxWaitTime);
            }
        } else {
            downSerialData.setIsMultiplePackSleep(null);
            Message<DownSerialData> message = new Message<>(parkId, FullCloudDownCmdEnum.屏显信息下发.getCmdType(), downSerialData);
            messageId = downHandle.send(parkCode, serialNumber, message);
        }

        log.info("通道繁忙下发|通道ID：{}|messageId：{}", deviceInfo.getId(), messageId);
        //发送繁忙转空闲的延迟消息
        if (messageId != null) {
            //更新当前通道为忙时
            redisUtils.hPut(FULL_SHOW_STATUS, String.valueOf(deviceInfo.getId()), 1);
            // 获取屏显配置
            ObjectResponse<LedConfig> objectResponse = ledService.getLedConfigByChannel(deviceInfo.getId());
            int ledRestoreDefaultTime = Optional.ofNullable(objectResponse.getData().getLedRestoreDefaultTime()).orElse(90);
            redisUtils.set(FULL_BUSY_HINT + serialNumber, messageId, ledRestoreDefaultTime);
            // 下发空闲信息
            mqPushService.pushZsBusyHint(parkId, serialNumber, messageId, ledRestoreDefaultTime * 1000);
            log.info("延迟下发通道空闲信息|通道ID：{}|messageId：{}|延迟时间：{}s", deviceInfo.getId(), messageId, ledRestoreDefaultTime);
        }
        return messageId;
    }

    /**
     * 繁忙转空闲
     *
     * @param parkId       车场ID
     * @param serialNumber 设备序列号
     */
    public void busyChangeFree(Long parkId, String serialNumber) {
        // 获取通道信息
        TokenDeviceVo deviceInfo = cacheHandle.getDeviceInfo(serialNumber);
        if (deviceInfo == null) {
            log.warn("设备信息未找到，serialNumber: {}", serialNumber);
            return;
        }
        // 获取 LED 显示信息
        ObjectResponse<LedShowDto> ledVoObjectResponse = getLedShowDto(deviceInfo);
        LedShowDto ledShowDto = ledVoObjectResponse.getData();
        if (ledShowDto == null) {
            log.warn("LED 显示信息未找到，deviceId: {}", deviceInfo.getId());
            return;
        }
        // 获取显示内容
        String show = getShowContent(parkId, deviceInfo);
        // 组装并发送消息
        sendDownSerialData(parkId, serialNumber, deviceInfo, show, ledShowDto);
    }

    /**
     * 获取LED显示信息
     *
     * @param deviceInfo 通道信息
     * @return list
     */
    private ObjectResponse<LedShowDto> getLedShowDto(TokenDeviceVo deviceInfo) {
        if (deviceInfo.getInandoutType() == 1) {
            return ledService.getLedShowByType(deviceInfo.getId(), LedShow.DisplayTypeEnum.入口空闲显示.type);
        } else {
            return ledService.getLedShowByType(deviceInfo.getId(), LedShow.DisplayTypeEnum.出口空闲显示.type);
        }
    }

    /**
     * 获取显示内容
     *
     * @param parkId     车场ID
     * @param deviceInfo 通道信息
     * @return str
     */
    private String getShowContent(Long parkId, TokenDeviceVo deviceInfo) {
        if (deviceInfo.getInandoutType() == 1) {
            return ledShowHandle.enterFreeHandle(parkId, deviceInfo.getId(), new HashMap<>());
        } else {
            return ledShowHandle.exitFreeHandle(parkId, deviceInfo.getId(), new HashMap<>());
        }
    }

    /**
     * 下发通道空闲信息
     *
     * @param parkId       车场ID
     * @param serialNumber 设备序列号
     * @param deviceInfo   设备通道信息
     * @param show         显示内容
     * @param ledShowDto   led屏显信息
     */
    private void sendDownSerialData(Long parkId, String serialNumber, TokenDeviceVo deviceInfo, String show, LedShowDto ledShowDto) {
        try {
            //更新为闲时
            redisUtils.hPut(FULL_SHOW_STATUS, String.valueOf(deviceInfo.getId()), 0);
            HintRequest hintRequest = new HintRequest();
            hintRequest.setShow(show);
            DownSerialData downSerialData = assembleDownSerialData(hintRequest, deviceInfo, ledShowDto, false);
            Message<DownSerialData> message = new Message<>(parkId, FullCloudDownCmdEnum.屏显信息下发.getCmdType(), downSerialData);
            downHandle.send(deviceInfo.getParkCode(), serialNumber, message);
            log.info("通道空闲信息下发成功|通道ID：{}|messageId：{}", deviceInfo.getId(), message.getMessageId());
        } catch (ResponseBodyException e) {
            log.warn("设备不在线，serialNumber: {}", serialNumber, e);
        }
    }

    /**
     * 保存设备通道时间信息显示
     *
     * @param inoutDevice 通道信息
     */
    private void saveChannelDateConfig(ParkInoutdevice inoutDevice, String parkCode) {
        try {
            String serialNumber = cacheHandle.getSerialNumber(parkCode, inoutDevice.getInandoutCode());
            ObjectResponse<LedShowDto> led;
            if (inoutDevice.getInandoutType() == 1) {
                led = ledService.getLedShowByType(inoutDevice.getId(), LedShow.DisplayTypeEnum.入口空闲显示.type);
            } else {
                led = ledService.getLedShowByType(inoutDevice.getId(), LedShow.DisplayTypeEnum.出口空闲显示.type);
            }
            log.info("请求获取led信息：{}", led);
            if (ObjectResponse.isSuccess(led)) {
                LedShowDto ledShow = led.getData();
                log.info("收到led屏显下发内容：{}", ledShow.getContent());
                String[] contents = ledShow.getContent().split("/");
                int index = Arrays.stream(contents).map(String::trim)
                        .filter(part -> part.contains("{1}")).findFirst()
                        .map(part -> Arrays.asList(contents).indexOf(part)).orElse(-1);
                if (index == -1) {
                    //删除通道时间设置
                    redisUtils.hDelete(FULL_DOWN_SHOW_DATE_CHANNEL, String.valueOf(inoutDevice.getId()));
                    return;
                }
                Map<String, Object> map = new HashMap<>(16);
                map.put("parkId", inoutDevice.getParkId());
                map.put("sn", serialNumber);
                //保存通道时间设置
                redisUtils.hPut(FULL_DOWN_SHOW_DATE_CHANNEL, String.valueOf(inoutDevice.getId()), map);
            }
        } catch (Exception e) {
            log.info("保存设备通道时间信息显示异常", e);
        }
    }

    /**
     * 合并两个字节数组
     *
     * @param array1 第一个字节数组
     * @param array2 第二个字节数组 *
     * @return 合并后的字节数组
     */
    public static byte[] mergeByteArrays(byte[] array1, byte[] array2) {
        byte[] mergedArray = new byte[array1.length + array2.length];
        System.arraycopy(array1, 0, mergedArray, 0, array1.length);
        System.arraycopy(array2, 0, mergedArray, array1.length, array2.length);
        return mergedArray;
    }

    /**
     * 间隔查询时间，50ms
     */
    private static final int INTERVAL = 50;
    /**
     * 获取485串口响应信息
     * @param key redis KEY
     * @param beforeSendTime 发送前的时间
     * @param timeOut 最大等待时间
     * @return
     */
    public void getAndWait485Resp(String key, long beforeSendTime, long timeOut){
        long currentTimeMillis = System.currentTimeMillis();
        long lastTime = currentTimeMillis + timeOut;
        int n = 1;
        //查询redis中是否返回
        while (lastTime > currentTimeMillis) {
            Long data = redisUtils.get(key, Long.class);
            if (data != null) {
                if (data > beforeSendTime) {
                    log.info("第{}次从redis中读取到了key[{}]，相机485串口响应的时间：{}", n, key, data);
                    return;
                }
            }
            try {
                Thread.sleep(INTERVAL);
            } catch (InterruptedException e) {
                log.warn(String.valueOf(e.getMessage()), e);
            }
            currentTimeMillis = System.currentTimeMillis();
            n ++;
        }
        log.info("时限内未查询到key[{}]，相机485串口响应的信息！", key);
    }
}
