package com.icetech.park.service.down.full.impl;

import com.icetech.cloudcenter.api.AlarmService;
import com.icetech.cloudcenter.api.NotifyService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.enumeration.FullCloudDownCmdEnum;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.request.CloseBrakeRequest;
import com.icetech.cloudcenter.domain.request.OpenBrakeRequest;
import com.icetech.cloudcenter.domain.request.OpeningDtoRequest;
import com.icetech.cloudcenter.domain.request.p2c.RemoteSwitchRequest;
import com.icetech.cloudcenter.domain.request.p2c.SoftTriggerRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.response.p2c.RemoteSwitchResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.constants.LogWarnTypeEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.JsonUtils;
import com.icetech.park.component.AsyncNotifyClient;
import com.icetech.park.domain.entity.ChannelAlarm;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.domain.vo.full.GpIoOutVO;
import com.icetech.park.service.down.Message;
import com.icetech.park.service.down.RemoteOperaCommon;
import com.icetech.park.service.down.RemoteOperaService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.flow.p2c.FlowCondition;
import com.icetech.park.service.handle.FullCloudDownHandle;
import com.icetech.third.domain.entity.third.SendInfoRecord;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;


/**
 * 纯云下发接口
 *
 * @author wgq
 */
@Slf4j
@Service("fullCloudGpioOutServiceImpl")
public class FullCloudGpIoOutServiceImpl extends RemoteOperaCommon implements RemoteOperaService, NotifyService<GpIoOutVO>, ResponseService<String> {

    @Autowired
    private FullCloudDownHandle downHandle;
    @Autowired
    private AlarmService alarmService;
    @Resource
    private AsyncNotifyClient asyncNotifyClient;
    @Resource
    private FullCloudSnapShotServiceImpl fullCloudSnapShotService;

    @Override
    public ObjectResponse<RemoteSwitchResponse> execute(Integer switchType, String parkCode, String channelId, String plateNum) {
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkCode(parkCode);
        if (ObjectResponse.isSuccess(parkObjectResponse)){
            Long parkId = parkObjectResponse.getData().getId();
            String deviceNo;
            try {
                deviceNo = cacheHandle.getSerialNumber(parkCode, channelId);
            } catch (ResponseBodyException e) {
                return ObjectResponse.failed(CodeConstants.ERROR_3003);
            }
            return execute(switchType, parkId, parkCode, deviceNo, plateNum);
        }
        return ObjectResponse.failed(CodeConstants.ERROR_400);
    }
    @Override
    public ObjectResponse<RemoteSwitchResponse> execute(Integer switchType, Long parkId, String parkCode, String serialNumber, String plateNum) {

        int sequenceId = RandomUtils.nextInt(100000, 1000000);
        String messageId = open(switchType, parkId, parkCode, serialNumber, plateNum, sequenceId, null);
        if (messageId == null){
            TokenDeviceVo tokenInfo2 = cacheHandle.getDeviceInfo(serialNumber);
            if (tokenInfo2 != null){
                alarmService.switchFailHandler(parkCode, parkId, tokenInfo2.getInandoutCode());
            }
            return ObjectResponse.failed(CodeConstants.ERROR_3003);
        }else{
            ObjectResponse<String> objectResponse = cacheHandle.getResponseFromRedis(messageId, 5000L);
            if (ObjectResponse.isSuccess(objectResponse)){
                RemoteSwitchResponse remoteSwitchResponse = new RemoteSwitchResponse();
                remoteSwitchResponse.setResult(1);
                return ObjectResponse.success(remoteSwitchResponse);
            } else {
                log.info("开闸未响应，准备二次下发：{}", messageId);
                TokenDeviceVo tokenInfo2 = cacheHandle.getDeviceInfo(serialNumber);
                if (tokenInfo2 != null) {
                    String channelCode = tokenInfo2.getInandoutCode();
                    String msgId = open(switchType, parkId, parkCode, serialNumber, plateNum, sequenceId, null);
                    if (msgId != null) {
                        objectResponse = cacheHandle.getResponseFromRedis(messageId, 5000L);
                        log.info("msgId：{} 响应：{}", msgId, objectResponse);
                        if (!ObjectResponse.isSuccess(objectResponse)){
                            log.warn("[监控埋点] alarmType:[{}],keyword1:[{}]", LogWarnTypeEnum.开闸失败.name() + "-" + "纯云",
                                    "parkCode: " + parkCode + ", serialNumber: " + serialNumber + ", plateNum: " + plateNum);
                            alarmService.switchFailHandler(parkCode, parkId, channelCode);
                        } else {
                            RemoteSwitchResponse remoteSwitchResponse = new RemoteSwitchResponse();
                            remoteSwitchResponse.setResult(1);
                            return ObjectResponse.success(remoteSwitchResponse);
                        }
                    }
                }
                return ObjectResponse.failed(CodeConstants.ERROR_3002);
            }
        }
    }

    /**
     * 开闸指令下发
     * @param switchType 开关类型
     * @param parkCode 车场编号
     * @param serialNumber 相机SN
     * @return 响应
     */
    @Override
    public ObjectResponse<Void> execute(Integer switchType, String parkCode, String serialNumber, String plateNum,
                                        String topic, RemoteSwitchRequest.ExtraInfo extraInfo) {
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkCode(parkCode);
        if (!ObjectResponse.isSuccess(parkObjectResponse)) {
            throw new ResponseBodyException(CodeConstants.ERROR_402, "车场不存在");
        }
        Long parkId = parkObjectResponse.getData().getId();

        GpIoOutVO gpIoOutVO = assembleGpIoOut(2);
        Message<GpIoOutVO> message = new Message<>(parkId, FullCloudDownCmdEnum.远程开关闸.getCmdType(), gpIoOutVO);
        String messageId = downHandle.send(parkCode, serialNumber, message, topic, JsonUtils.toString(extraInfo));
        if (messageId == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_3002);
        }
        return ObjectResponse.failed(CodeConstants.ERROR_12002, messageId);
    }
    @Override
    public String open(Integer switchType, Long parkId, String parkCode, String serialNumber, String plateNum) {
        return open(switchType, parkId, parkCode, serialNumber, plateNum, RandomUtils.nextInt(100000, 1000000), null);
    }
    @Override
    public String open(Integer switchType, Long parkId, String parkCode, String serialNumber, String plateNum, String topic) {
        return open(switchType, parkId, parkCode, serialNumber, plateNum, RandomUtils.nextInt(100000, 1000000), topic);
    }
    @Override
    public String open(Integer switchType, Long parkId, String parkCode, String serialNumber, String plateNum, Integer sequenceId, String topic) {
        GpIoOutVO gpIoOutVO = assembleGpIoOut(switchType == 1 ? 2 : 0);
        Message<GpIoOutVO> message = new Message<>(parkId, FullCloudDownCmdEnum.远程开关闸.getCmdType(), gpIoOutVO);
        return downHandle.send(parkCode, serialNumber, message);
    }
    @Override
    public ObjectResponse<RemoteSwitchResponse> secondOpen(RemoteSwitchRequest remoteSwitchRequest, Long parkId, String parkCode, String serialNumber) {
        GpIoOutVO gpIoOutVO = assembleGpIoOut(remoteSwitchRequest.getSwitchType() == 1 ? 2 : 0);
        Message<GpIoOutVO> message = new Message<>(parkId, FullCloudDownCmdEnum.远程开关闸.getCmdType(), gpIoOutVO);
        String messageId = downHandle.send(parkCode, serialNumber, message);
        RemoteSwitchResponse remoteSwitchResponse = cacheHandle.getRemoteSwitch(messageId);
        if (remoteSwitchResponse != null){
            return ObjectResponse.success(remoteSwitchResponse);
        }
        return ObjectResponse.failed(CodeConstants.ERROR_3002);
    }

    /**
     * 组装参数
     *
     * @return List
     */
    private GpIoOutVO assembleGpIoOut(Integer value) {
        GpIoOutVO gpIoOut = new GpIoOutVO();
        gpIoOut.setIo(0);
        gpIoOut.setDelay(1000);
        gpIoOut.setValue(value);
        return gpIoOut;
    }
    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        downHandle.dealResponse(p2cBaseResponse, parkId, FullCloudDownCmdEnum.远程开关闸.getCmdType());
    }

    @Override
    public void notify(String messageId, ObjectResponse<String> response, SendInfoRecord<GpIoOutVO> sendInfoRecord) {
        boolean success = ObjectResponse.isSuccess(response);
        String serialNumber = sendInfoRecord.getTarget();
        TokenDeviceVo tokenInfo2 = cacheHandle.getDeviceInfo(serialNumber);
        if (!success) {
            log.warn("[监控埋点] alarmType:[{}],keyword1:[{}],keyword2:[{}]", LogWarnTypeEnum.开闸失败.name() + "-" + "纯云",
                    "parkCode: " + sendInfoRecord.getParkCode() +
                            ", serialNumber: " + serialNumber, response.getMsg());
            if (tokenInfo2 != null && tokenInfo2.getInandoutType() != null) {
                alarmService.switchFailHandler(sendInfoRecord.getParkCode(), sendInfoRecord.getParkId(), tokenInfo2.getInandoutCode());
            }
            asyncNotifyClient.callBack(messageId, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), ObjectResponse.failed(CodeConstants.ERROR_3002));
            return;
        }
        Long executeTime = DateTools.unixTimestamp();
        if (sendInfoRecord.getExtraInfo() == null) {
            return;
        }
        RemoteSwitchResponse remoteSwitchResponse = new RemoteSwitchResponse();
        remoteSwitchResponse.setExecuteTime(executeTime);
        remoteSwitchResponse.setResult(1);
        String image = redisUtils.get(RedisConstants.SOFT_TRIGGER_IMAGE_KEY + tokenInfo2.getInandoutCode(), String.class);
        remoteSwitchResponse.setImage(image);
        RemoteSwitchRequest.ExtraInfo extraInfo = JsonUtils.parseObject(sendInfoRecord.getExtraInfo(),
                RemoteSwitchRequest.ExtraInfo.class);
        if (RemoteSwitchRequest.ExtraInfoEnum.OPEN.val.equals(extraInfo.getBiz())) {
            OpenBrakeRequest openBrakeRequest = JsonUtils.parseObject(
                    JsonUtils.toString(extraInfo.getRequestVO()), OpenBrakeRequest.class);
            //记录类型，1：入，2：出
            if (openBrakeRequest.getRecordType() == 1){
                CarEnterRequest enterRequest = new CarEnterRequest();
                enterRequest.setEnterTime(System.currentTimeMillis()/1000);
                enterRequest.setType(3);
                enterRequest.setCarType(1);
                enterRequest.setInandoutCode(openBrakeRequest.getAisleCode());
                enterRequest.setParkCode(openBrakeRequest.getParkCode());
                enterRequest.setPlateNum(openBrakeRequest.getPlateNum());
                enterRequest.setProperty(2);
                enterRequest.setOpenFlag(1);
                enterRequest.setOrderNum(openBrakeRequest.getOrderNum());
                carOrderEnterService.sendWebsocketMessage(enterRequest,1);
            }else {
                CarExitRequest exitRequest = new CarExitRequest();
                exitRequest.setType(3);
                exitRequest.setCarType(1);
                exitRequest.setExitTime(System.currentTimeMillis());
                exitRequest.setInandoutCode(openBrakeRequest.getAisleCode());
                exitRequest.setParkCode(openBrakeRequest.getParkCode());
                exitRequest.setPlateNum(openBrakeRequest.getPlateNum());
                exitRequest.setOrderNum(openBrakeRequest.getOrderNum());
                exitRequest.setOpenFlag(1);
                carOrderExitService.sendWebsocketMessage(exitRequest, "5", FlowCondition.YES, FlowCondition.NO);
            }
            // 更新通道的报警记录为已处理
            ChannelAlarm channelAlarm = new ChannelAlarm();
            channelAlarm.setParkId(sendInfoRecord.getParkId());
            channelAlarm.setChannelCode(openBrakeRequest.getAisleCode());
            channelAlarm.setStatus(ChannelAlarm.Status.已处理.getStatus());
            channelAlarmDao.update(channelAlarm);
            if (delRemoteSwitchResponse(openBrakeRequest, sendInfoRecord.getParkCode(), remoteSwitchResponse, sendInfoRecord.getParkId())){
                asyncNotifyClient.callBack(messageId, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), ObjectResponse.success());
            } else {
                asyncNotifyClient.callBack(messageId, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), ObjectResponse.failed(CodeConstants.ERROR_3002));
            }
        } else if (RemoteSwitchRequest.ExtraInfoEnum.CLOSE.val.equals(extraInfo.getBiz())) {
            CloseBrakeRequest closeBrakeRequest = JsonUtils.parseObject(JsonUtils.toString(extraInfo.getRequestVO()),
                    CloseBrakeRequest.class);
            //添加开关闸记录
            addOpeningRecord(closeBrakeRequest, remoteSwitchResponse.getImage(), executeTime, closeBrakeRequest.getImagePath(), tokenInfo2.getParkId());
            asyncNotifyClient.callBack(messageId, sendInfoRecord.getTopic(), sendInfoRecord.getEnv(), ObjectResponse.success());
        }

    }
    @Override
    protected void addOpeningRecord(Object openOrCloseBrakeRequest, String image, Long executeTime, String imagePath, Long parkId) {
        //添加开闸或关闸记录
        OpeningDtoRequest openingDtoRequest = new OpeningDtoRequest();
        BeanUtils.copyProperties(openOrCloseBrakeRequest, openingDtoRequest);
        openingDtoRequest.setImgUrl(imagePath == null ? image : imagePath + "," + image);
        openingDtoRequest.setExecuteTime(executeTime);
        if (openOrCloseBrakeRequest instanceof CloseBrakeRequest) {
            openingDtoRequest.setSourcegate(4);
        }
        ObjectResponse<Integer> response = parkService.saveOpeningRecord(openingDtoRequest);
        if (ObjectResponse.isSuccess(response)) {
            //异步抓拍图片并更新记录表
            fullCloudSnapShotService.execute(parkId, openingDtoRequest.getParkCode(), openingDtoRequest.getAisleCode(), null,
                    SoftTriggerRequest.SwitchSnap.builder().biz(SoftTriggerRequest.ExtraInfoEnum.SWITCH_SNAP.val)
                            .id(response.getData()).build());
        }
    }
    /**
     * 请求开闸
     *
     * @param request 开闸入参
     * @return ObjectResponse
     */
    public ObjectResponse<Void> requestOpenBrake(OpenBrakeRequest request, Long parkId) {
        return handleBrakeRequest(request, parkId, request.getParkCode(), request.getAisleCode(), true);
    }

    /**
     * 请求关闸
     *
     * @param request 关闸入参* @return ObjectResponse
     */
    public ObjectResponse<Void> requestCloseBrake(CloseBrakeRequest request, Long parkId) {
        return handleBrakeRequest(request, parkId, request.getParkCode(), request.getAisleCode(), false);
    }
    /**
     * 请求开闸或关闸
     *
     * @param parkCode     车场
     * @param deviceNo     设备序列号
     * @param brakeRequest 闸门请求参数
     * @param open         是否开闸
     * @return ObjectResponse
     */
    public ObjectResponse<Void> handleBrakeRequest(Object brakeRequest, Long parkId, String parkCode, String deviceNo, boolean open) {
        try {
            log.info("收到：{}，信息：{}", open ? "开闸" : "关闸", brakeRequest);
            // 发送请求
            GpIoOutVO gpIoOutVO = assembleGpIoOut(2);
            Message<GpIoOutVO> message = new Message<>(parkId, FullCloudDownCmdEnum.远程开关闸.getCmdType(), gpIoOutVO);
            String messageId = downHandle.send(parkCode, deviceNo, message);
            if (messageId == null) {
                return ObjectResponse.failed(CodeConstants.ERROR, "纯云参数下发失败");
            }
            return ObjectResponse.success();
        } catch (Exception e) {
            log.error("闸门操作异常，车场编号[{}], 通道编号[{}], 参数[{}]", parkCode, deviceNo, brakeRequest, e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR);
        }
    }
}
