package com.icetech.park.service.down.full.impl;

import com.icetech.basics.constants.TextConstant;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.cloudcenter.api.month.VipCarService;
import com.icetech.cloudcenter.api.month.VipRegionService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.third.SendInfoService;
import com.icetech.cloudcenter.domain.enumeration.FullCloudDownCmdEnum;
import com.icetech.cloudcenter.domain.request.full.WhiteListOperatorRequest;
import com.icetech.cloudcenter.domain.request.p2c.VipInfoRequest;
import com.icetech.cloudcenter.domain.response.VipTypeDto;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.fee.domain.entity.vip.VipRecord;
import com.icetech.fee.domain.entity.vip.VipRegion;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.down.full.FullCloudSendMsgServiceImpl;
import com.icetech.park.service.down.p2c.DownService;
import com.icetech.park.service.down.p2c.ResponseService;
import com.icetech.park.service.handle.P2cDownHandle;
import com.icetech.third.domain.entity.third.SendInfo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;
import static com.icetech.basics.constants.TextConstant.THREE;

/**
 * 免费VIP下发
 * @author fangct
 */
@Service
@Slf4j
public class FullCloudFreeVipServiceImpl implements DownService<List<VipInfoRequest>, Long>, ResponseService<String> {

    @Autowired
    private P2cDownHandle p2cDownHandle;
    @Autowired
    private VipCarService vipCarService;
    @Autowired
    private VipRegionService vipRegionService;
    @Autowired
    private FullCloudSendMsgServiceImpl sendMsgService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Resource
    private SendInfoService sendInfoService;

    @Override
    public ObjectResponse<Long> send(SendRequest sendRequest, String parkCode) {
        sendRequest.setReqServiceType(FullCloudDownCmdEnum.名单下发.getCmdType());
        Long serviceId = sendRequest.getServiceId();
        ObjectResponse<VipRecord> recordObjectResponse = vipCarService.getOperatorRecordById(serviceId);
        if (!ObjectResponse.isSuccess(recordObjectResponse)) {
            return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, "vip操作记录不存在"));
        }
        VipRecord vipRecord = recordObjectResponse.getData();
        ObjectResponse<VipTypeDto> vipTypeObjectResponse = vipCarService.getVipTypeById(vipRecord.getTypeId());
        if (!ObjectResponse.isSuccess(vipTypeObjectResponse)) {
            return ObjectResponse.failed(ERROR_410, "未找到类型信息");
        }
        VipTypeDto vipType = vipTypeObjectResponse.getData();
        Integer discountType = vipType.getType();
        if (discountType != null && discountType != 1) {
            throw new ResponseBodyException(CodeConstants.ERROR_406, TextConstant.getDefaultMessage(THREE, "非全免VIP，无需下发"));
        }
        List<WhiteListOperatorRequest> whiteListOperatorRequests = buildRequest(vipRecord);

        ObjectResponse<Park> parkObjectResponse = parkService.findByParkCode(parkCode);
        ObjectResponse.notError(parkObjectResponse);
        Park park = parkObjectResponse.getData();
        if (park.getIsInterior() != null && park.getIsInterior() == 1) {
            List<VipRegion> vipRegionList = vipRegionService.getVipRegionsListByVipTypeId(vipType.getId());
            List<String> sns = new ArrayList<>();
            for (VipRegion vipRegion : vipRegionList) {
                List<ParkDevice> parkDeviceList = parkDeviceService.getDeviceListByParkRegionId(
                        sendRequest.getParkId(), vipRegion.getId(), 1).getData();
                if (parkDeviceList == null) {
                    continue;
                }
                for (ParkDevice parkDevice : parkDeviceList) {
                    String serialNumber = parkDevice.getSerialNumber();
                    sns.add(serialNumber);
                }
            }
            if (sns.isEmpty()) {
                return ObjectResponse.failed(ERROR_410, TextConstant.getDefaultMessage(THREE, "尚未添加相机设备"));
            }
            return sendMsgService.send2Devices(String.join(";", sns), sendRequest, parkCode, whiteListOperatorRequests.toArray());
        }
        return sendMsgService.send2Park(sendRequest, parkCode, whiteListOperatorRequests.toArray());
    }

    private List<WhiteListOperatorRequest> buildRequest(VipRecord vipRecord) {
        List<WhiteListOperatorRequest> list = new ArrayList<>();
        List<WhiteListOperatorRequest.DldbRec> dldbRecList = new ArrayList<>();
        Integer operType = vipRecord.getOperType();
        if (operType == 2) {
            List<String> deletePlates = new ArrayList<>();
            List<VipRecord> vipRecordList = vipCarService.getVipRecordsByVipId(vipRecord.getParkId(), vipRecord.getVipId());
            //和当前车牌不相同的其他车牌
            List<String> oldPlateNums = vipRecordList.stream().map(VipRecord::getPlateNum)
                    .filter(Objects::nonNull)
                    .filter(p -> !p.equals(vipRecord.getPlateNum())).distinct().collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(oldPlateNums)) {
                String joinPlates = String.join(",", oldPlateNums);
                String[] oldList = joinPlates.split(",");
                List<String> newList = Arrays.asList(vipRecord.getPlateNum().split(","));
                for (String oldPlateNum : oldList) {
                    if (!newList.contains(oldPlateNum)) {
                        deletePlates.add(oldPlateNum);
                    }
                }
                if (CollectionUtils.isNotEmpty(deletePlates)) {
                    WhiteListOperatorRequest whiteListOperatorRequest = new WhiteListOperatorRequest();
                    whiteListOperatorRequest.setOperator_type("delete");
                    if (deletePlates.size() == 1) {
                        whiteListOperatorRequest.setPlate(deletePlates.get(0));
                    } else {
                        whiteListOperatorRequest.setPlate(deletePlates);
                    }
                    list.add(whiteListOperatorRequest);
                }
            }
        }

        if (operType == 3) {//删除操作
            WhiteListOperatorRequest whiteListOperatorRequest = new WhiteListOperatorRequest();
            whiteListOperatorRequest.setOperator_type("delete");
            if (vipRecord.getPlateNum().contains(",")) {
                whiteListOperatorRequest.setPlate(Arrays.asList(vipRecord.getPlateNum().split(",")));
            } else {
                whiteListOperatorRequest.setPlate(vipRecord.getPlateNum());
            }
            list.add(whiteListOperatorRequest);
            return list;
        }
        if (vipRecord.getStartTime() == null || vipRecord.getEndTime() == null) {
            throw new ResponseBodyException(ERROR_410, TextConstant.getDefaultMessage(THREE, "日期不合法"));
        }

        WhiteListOperatorRequest whiteListOperatorRequest = new WhiteListOperatorRequest();
        whiteListOperatorRequest.setOperator_type("update_or_add");
        for (String plateNum : vipRecord.getPlateNum().split(",")) {
            WhiteListOperatorRequest.DldbRec dldbRec = new WhiteListOperatorRequest.DldbRec();
            dldbRec.setPlate(plateNum);
            dldbRec.setIndex(vipRecord.getVipId());
            dldbRec.setCustomer_id(vipRecord.getVipId());
            dldbRec.setCreate_time(DateTools.getFormat(vipRecord.getStartTime()));
            dldbRec.setOverdue_time(DateFormatUtils.format(vipRecord.getEndTime(), "yyyy-MM-dd 23:59:59"));
            dldbRec.setEnable(1);
            dldbRec.setTime_seg_enable(0);
            dldbRec.setEnable_time(DateFormatUtils.format(vipRecord.getStartTime(), "yyyy-MM-dd 00:00:00"));
            dldbRec.setNeed_alarm(0);
            dldbRecList.add(dldbRec);
        }
        whiteListOperatorRequest.setDldb_rec(dldbRecList);
        list.add(whiteListOperatorRequest);
        return list;
    }

    private ObjectResponse<Void> send(SendRequest sendRequest, String parkCode, List<VipInfoRequest> list) {
        return sendMsgService.send2Park(sendRequest, parkCode, list);
    }

    @Override
    public void dealResponse(P2cBaseResponse<String> p2cBaseResponse, Long parkId, String parkCode, String deviceNo) {
        p2cDownHandle.dealResponse(p2cBaseResponse, parkId, FullCloudDownCmdEnum.名单下发.getCmdType());
    }

    @Override
    public ObjectResponse<Long> batchSend(List<SendRequest> sendRequestList, Long parkId, String parkCode){
        for (SendRequest sendRequest : sendRequestList) {
            try {
                ObjectResponse<Long> objectResponse = send(sendRequest, parkCode);
                if (CodeConstants.ERROR_407.equals(objectResponse.getCode())) {
                    SendInfo sendInfo = new SendInfo();
                    sendInfo.setId(sendRequest.getTaskId());
                    sendInfo.setStatus(SendInfo.StatusEnum._DELAY.getCode());
                    sendInfo.setRemark(objectResponse.getMsg());
                    sendInfo.setNextFireTime(objectResponse.getData().intValue());
                    sendInfoService.update(sendInfo);
                }
            } catch (ResponseBodyException e) {
                log.warn("业务异常={}", e.getMessage());
                if (CodeConstants.ERROR_406.equals(e.getErrCode())) {
                    SendInfo sendInfo = new SendInfo();
                    sendInfo.setId(sendRequest.getTaskId());
                    sendInfo.setStatus(SendInfo.StatusEnum._PASS.getCode());
                    sendInfo.setRemark(TextConstant.getDefaultMessage(THREE, e.getMessage()));
                    sendInfoService.update(sendInfo);
                }
            } catch (Exception e) {
                log.error("e=", e);
            }
        }
        return ObjectResponse.success();
    }
}
