package com.icetech.park.service.down.full.controlcard.vertical_2x8;

import cn.hutool.core.io.checksum.crc16.CRC16Modbus;
import org.springframework.util.Assert;

import java.nio.charset.Charset;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Arrays;
import java.util.List;

/**
 * 方控语音控制卡(2x8标准竖屏)串口协议构建器
 *
 * @see SzfkznCmd
 */
public class SzfkznBuilder {
    /** 单个数据包的最大长度 255 字节 */
    private static final int MAX_LEN = 255;
    private static final byte[] PREFIX = {(byte) 0xAA, (byte) 0X55};
    private static final byte[] SUFFIX = {(byte) 0xAF};
    private static final int LEN = 2 + 1 + 1 + 1 + 1 + 2 + 2 + 1;
    private static final Charset CHARSET = Charset.forName("GBK");

    /**
     * <pre>
     * 构建<code>查询(心跳)</code>指令
     * 该命令查询版本信息等内容。该指令有强制返回数据，可作为心跳包。
     * </pre>
     *
     * @return 二进制指令包
     * @see SzfkznCmd#QUERY
     */
    public static byte[] buildQuery() {
        return build(SzfkznCmd.QUERY, null);
    }

    /**
     * 构建<code>设置时间</code>指令
     *
     * @param dateTime 日期时间，仅取年、月、日、时、分、秒
     * @return 二进制指令包
     * @see SzfkznCmd#SET_TIME
     */
    public static byte[] buildSetTime(LocalDateTime dateTime) {
        return build(SzfkznCmd.SET_TIME, new byte[]{
                (byte) (dateTime.getYear() % 100),
                (byte) dateTime.getMonthValue(),
                (byte) dateTime.getDayOfMonth(),
                (byte) dateTime.getHour(),
                (byte) dateTime.getMinute(),
                (byte) dateTime.getSecond(),
        });
    }

    /**
     * 构建<code>取消临显(即时显示)内容</code>指令
     *
     * @param one   取消第一行临显，恢复到广告内容
     * @param two   取消第一行临显，恢复到广告内容
     * @param three 取消第一行临显，恢复到广告内容
     * @param four  取消第一行临显，恢复到广告内容
     * @return 二进制指令包
     * @see SzfkznCmd#CANCEL_TMP_SHOW
     */
    public static byte[] buildCancelTmpShow(boolean one, boolean two, boolean three, boolean four) {
        int val = (one ? 1 : 0) << 7
                | (two ? 1 : 0) << 6
                | (three ? 1 : 0) << 5
                | (four ? 1 : 0) << 4;
        return build(SzfkznCmd.CANCEL_TMP_SHOW, new byte[]{(byte) val});
    }

    /**
     * <pre>
     * 构建<code>语音播放指令</code>指令
     * 该指令下发，直接中断原有语音，立即播报最新内容。
     * </pre>
     *
     * @param codes 约定的语音编码
     * @return 二进制指令包
     * @see SzfkznCmd#PLAY_AUDIO
     * @see #buildPlayAudio(List)
     * @see #buildPlayAudioTTS(String)
     */
    public static byte[] buildPlayAudio(AudioCode[] codes) {
        return buildPlayAudio(Arrays.asList(codes));
    }

    /**
     * <pre>
     * 构建<code>语音播放指令</code>指令
     * 该指令下发，直接中断原有语音，立即播报最新内容。
     * </pre>
     *
     * @param codes 约定的语音编码
     * @return 二进制指令包
     * @see SzfkznCmd#PLAY_AUDIO
     * @see #buildPlayAudio(AudioCode[])
     * @see #buildPlayAudioTTS(String)
     */
    public static byte[] buildPlayAudio(List<AudioCode> codes) {
        int length = 0;
        for (AudioCode code : codes) {
            length += code.getCode().length;
        }
        byte[] bytes = new byte[length];
        int offset = 0;
        for (AudioCode code : codes) {
            System.arraycopy(code.getCode(), 0, bytes, offset, code.getCode().length);
            offset += code.getCode().length;
        }
        return build(SzfkznCmd.PLAY_AUDIO, bytes);
    }

    /**
     * <pre>
     * 构建<code>加载广告内容</code>指令
     * <b>需间隔50ms，发下一条指令。该指令一定要做成手动触发。</b>
     * 第一行和第四行，最大可输入30个汉字(60字节)，小于2个汉字，则定屏显示，超过2个汉字，则滚动显示。
     * 第二第三行的广告内容，建议分别使用8个汉字(16字节)，对开显示.
     * </pre>
     *
     * @param line    行号，有效值1-4，其他值无效
     * @param color   显示颜色，有效期1-3，1=红色，2=绿色，3=黄色，其他值默认为1
     * @param content 广告内容
     * @return 二进制指令包
     * @see SzfkznCmd#LOAD_AD
     */
    public static byte[] buildLoadAd(int line, int color, String content) {
        byte[] context = content.getBytes(CHARSET);
        byte[] bytes = new byte[context.length + 3];
        Assert.isTrue(bytes.length <= SzfkznCmd.LOAD_AD.getMaxLen(), "数据超过长度限制");
        bytes[0] = (byte) line;
        bytes[1] = (byte) color;
        System.arraycopy(context, 0, bytes, 3, context.length);
        return build(SzfkznCmd.LOAD_AD, bytes);
    }

    /**
     * 构建<code>下发临显(即时显示)内容</code>指令
     * <ol>
     * <li>该指令使用时，第一行和第四行，最大可显示30 个汉字。</li>
     * <li>行号为2或者3时，都将清除广告内容，将对应的内容显示在对应的行位上，为了美观，行号为2
     * 或者3时，内容的最大长度为6个汉字，超出内容不显示，该两行不建议显示数字或者字符，该两行的时长
     * 和颜色控制字永远以后面那一条指令的参数为准，同时生效，该两行显示内容自动上下位置居中显示。
     * <li>该指令增加3个虚拟行，即第5-7行，3个虚拟行的显示可被行号为2或者3的取消指令取消掉。</li>
     * <li>第5 虚拟行，用于显示在第二第三行的中间位置，内容最大长度为6个汉字，超出无效。</li>
     * <li>专用于显示“剩余车位XXX”，内容最大4个字节的数字，数字即为车位数，其中数字的颜色可以由控
     * 制字4来设置，控制字4的值为0=数字的颜色和控制字3一样，=1数字为0色，=2数字为绿色，=3数字为黄
     * 色，其他值为默认红色。4个字节的数字必须是ASCII 码，自动屏蔽高位的0，自动左右居中显示。</li>
     * <li>专用于显示“请交费XXX元”，内容最大4个字节的数字或者‘.’其他控制方式和剩余车位一样</li>
     * </ol>
     *
     * @param line     显示行号
     * @param duration 显示时长，单位秒，该参数为 0 时，表示长期显示
     * @param color    显示颜色，1-3 有效，1=红色，2=绿色，3=黄色，其他默认为1
     * @param color2   显示行号为 6 和 7 的时候，生效。用于控制关键数字的颜色
     * @param content  显示内容
     * @return 二进制指令包
     * @see SzfkznCmd#TMP_SHOW
     */
    public static byte[] buildTmpShow(int line, int duration, int color, int color2, String content) {
        byte[] context = content.getBytes(CHARSET);
        byte[] bytes = new byte[context.length + 4];
        Assert.isTrue(bytes.length <= SzfkznCmd.TMP_SHOW.getMaxLen(), "数据超过长度限制");
        bytes[0] = (byte) line;
        bytes[1] = (byte) duration;
        bytes[2] = (byte) color;
        bytes[3] = (byte) color2;
        System.arraycopy(context, 0, bytes, 4, context.length);
        return build(SzfkznCmd.TMP_SHOW, bytes);
    }

    /**
     * <pre>
     * 构建<code>二维码显示</code>指令
     * 二维码显示后，建议把显示屏安装在铁箱里面，透过黑色的玻璃去测试扫码效果。
     * </pre>
     *
     * @param mode     二维码显示模式，有效值0-2。
     *                 =0 时表示二维码显示为两行居中模式；
     *                 =1 二维码显示为两行居左模式带4个汉字(8个字节)；
     *                 =2 时，表示二维码显示为三行模式。
     *                 两行模式下，二维码图象的最大容纳字符为49字节，三行模式下最大容纳字符为180字节
     * @param duration 表示二维码显示的时长，单位秒。=0 时，长期显示。二维码对应的行可以被“取消临显指令取消”。
     * @param color    二维码本身的颜色。= 1 时红色，=2 时绿色，=3 时黄色，其他无效
     * @param content  内容
     * @return 二进制指令包
     * @see SzfkznCmd#SHOW_QRCODE
     */
    public static byte[] buildShowQrcode(int mode, int duration, int color, String content) {
        byte[] context = content.getBytes(CHARSET);
        int len = context.length + 3;
        if (mode == 1) len += 8;
        Assert.isTrue(len <= SzfkznCmd.SHOW_QRCODE.getMaxLen(), "数据超过长度限制");
        byte[] bytes = new byte[len];
        bytes[0] = (byte) mode;
        bytes[1] = (byte) duration;
        bytes[2] = (byte) color;
        System.arraycopy(context, 0, bytes, 3, context.length);
        // 当 mode=1 时，内容最后面的8个字节是不参与图象绘制的，所以在下发内容的时候，一定要注意加上8个字节的内容，不足的用0x20补齐
        if (mode == 1) Arrays.fill(bytes, len - 8 - 1, len, (byte) 0x20);
        return build(SzfkznCmd.SHOW_QRCODE, bytes);
    }

    /**
     * <pre>
     * 构建<code>设置勿扰音量</code>指令
     * <b>需间隔50ms，发下一条指令。该指令一定要做成手动触发。</b>
     * </pre>
     *
     * @param enable 勿扰模式是否生效
     * @param volume 0~9，勿扰时段的音量，大于9 的值都写入为9,0 为静音
     * @param begin  起始时间，仅取小时及分钟
     * @param end    结束时间，仅取小时及分钟
     * @return 二进制指令包
     * @see SzfkznCmd#SET_DND_VOLUME
     */
    public static byte[] buildSetDNDVolume(boolean enable, int volume, LocalTime begin, LocalTime end) {
        byte[] bytes = {(byte) (enable ? 1 : 0), (byte) volume,
                (byte) begin.getHour(), (byte) begin.getMinute(),
                (byte) end.getHour(), (byte) end.getMinute()};
        return build(SzfkznCmd.SET_DND_VOLUME, bytes);
    }

    /**
     * <pre>
     * 构建<code>设置音量</code>指令
     * <b>需间隔50ms，发下一条指令。该指令一定要做成手动触发。</b>
     * </pre>
     *
     * @param volume 0-9,0 表示静音，9=最大音量
     * @return 二进制指令包
     * @see SzfkznCmd#SET_VOLUME
     */
    public static byte[] buildSetVolume(int volume) {
        return build(SzfkznCmd.SET_VOLUME, new byte[]{(byte) volume});
    }

    /**
     * <pre>
     * 构建<code>万能语音播放指令</code>指令
     * </pre>
     *
     * @param content 语音内容
     * @return 二进制指令包
     * @see SzfkznCmd#PLAY_AUDIO_TTS
     * @see #buildPlayAudio(AudioCode[])
     * @see #buildPlayAudio(List)
     */
    public static byte[] buildPlayAudioTTS(String content) {
        byte[] contentBytes = content.getBytes(CHARSET);
        int dataLen = 1 + 1 + contentBytes.length;
        byte[] bytes = new byte[1 + 2 + dataLen];
        int offset = 0;
        bytes[offset++] = (byte) 0xFD;  // 帧头
        bytes[offset++] = (byte) (dataLen >> 8 & 0xFF); // 数据区长度-高位
        bytes[offset++] = (byte) (dataLen & 0xFF);  // 数据区长度-低位
        bytes[offset++] = (byte) 0x01;  // 命令字：语音合成播放命令
        bytes[offset++] = (byte) 0x01;  // 命令参数：GBK编码
        System.arraycopy(contentBytes, 0, bytes, offset, contentBytes.length);
        return bytes;
    }

    /**
     * 构建指令
     *
     * @param cmd  指令类型
     * @param data 指令数据包
     * @return 二进制指令包
     * @see SzfkznCmd
     */
    public static byte[] build(SzfkznCmd cmd, byte[] data) {    // 14
        int dataLen = data == null || data.length == 0 ? 0 : data.length;
        int length = LEN + dataLen;
        Assert.isTrue(length <= MAX_LEN, "数据超过长度限制");
        byte[] bytes = new byte[length];
        int offset = 0;
        System.arraycopy(PREFIX, 0, bytes, offset, PREFIX.length);   // 包头
        offset += PREFIX.length;
        bytes[offset++] = 0; // 流水号：上位机确定的业务流水号，下位机返回应答时，流水号原数返回,可全为0
        bytes[offset++] = (byte) 0x64;   // 地址：下位机485地址，出厂默认100 = 0X64
        bytes[offset++] = 0; // 保留
        bytes[offset++] = cmd.getCmd();  // 命令
        if (dataLen > 0) {
            // 长度：2字节，指定数据内容的长度，高字节在前，低字节在后，比如 255 个字节的长度应表述为0x00 0xff
            bytes[offset++] = (byte) (dataLen >> 8 & 0xff);
            bytes[offset++] = (byte) (dataLen & 0xff);
            System.arraycopy(data, 0, bytes, offset, dataLen);
            offset += dataLen;
        } else {
            offset += 2; // 长度
        }
        offset += 2; // 越过CRC
        System.arraycopy(SUFFIX, 0, bytes, offset, SUFFIX.length);   // 包尾
        offset += SUFFIX.length;
        Assert.isTrue(offset == length, "数据长度核对失败");

        // 生成CRC校验，去掉包头及包尾的所有数据
        CRC16Modbus crc = new CRC16Modbus();
        crc.update(bytes, 2, length - PREFIX.length - SUFFIX.length);
        long crcVal = crc.getValue();
        bytes[length - 3] = (byte) (crcVal >> 8 & 0xff);
        bytes[length - 2] = (byte) (crcVal & 0xff);
        return bytes;
    }
}
