package com.icetech.park.service.down.full.controlcard.dtong_led;

import com.icetech.common.utils.DateTools;
import com.icetech.park.service.down.TagProcessor;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 道通非TTS动态内容转换器
 * @author fangct
 */
public class DtongLedDynamicContentConverter {

    // 定义变量编号与中文描述的映射
    private static final Map<Integer, String> VARIABLE_DESCRIPTIONS = new HashMap<>();

    static {
        VARIABLE_DESCRIPTIONS.put(4, "");
        VARIABLE_DESCRIPTIONS.put(6, "月卡车 剩余 ");
        VARIABLE_DESCRIPTIONS.put(7, "停车");
        VARIABLE_DESCRIPTIONS.put(8, "请扫码 交费 ");
        VARIABLE_DESCRIPTIONS.put(9, "一路平安");
        VARIABLE_DESCRIPTIONS.put(10, "欢迎光临");
        VARIABLE_DESCRIPTIONS.put(11, "请等待人工确认");
        VARIABLE_DESCRIPTIONS.put(12, "车位已满");
        VARIABLE_DESCRIPTIONS.put(13, "禁止通行");
        VARIABLE_DESCRIPTIONS.put(14, "临时车");
        VARIABLE_DESCRIPTIONS.put(15, "月租车");
    }

    /**
     * 将动态内容转换为中文字符串
     *
     * @param input 动态内容字符串，如 "<4>京A12345</4> <6>10</6> <9/>"
     * @return 转换后的中文字符串
     */
    public static String convertToChinese(String input) {
        if (input == null || input.isEmpty()) {
            return "";
        }

        // 正则表达式匹配动态内容
        Pattern pattern = Pattern.compile("<(\\d+)>(.*?)</\\1>|<(\\d+)/>");
        Matcher matcher = pattern.matcher(input);

        StringBuilder result = new StringBuilder();

        while (matcher.find()) {
            int variableId;
            String value = "";

            // 判断是带值还是无值的情况
            if (matcher.group(1) != null) {
                variableId = Integer.parseInt(matcher.group(1));
                value = matcher.group(2);
            } else {
                variableId = Integer.parseInt(matcher.group(3));
            }

            // 根据变量编号获取中文描述
            String description = VARIABLE_DESCRIPTIONS.get(variableId);
            if (description == null) {
                continue; // 如果未找到对应的描述，跳过
            }

            // 根据变量编号处理值
            switch (variableId) {
                // 车牌号
                case 4:
                    result.append(description).append(value).append(" ");
                    break;
                // 剩余天数
                case 6:
                    result.append(description).append(TagProcessor.convert(Double.parseDouble(value))).append("天 ");
                    break;
                // 停车时长（秒）
                case 7:
                    int seconds = Integer.parseInt(value);
                    result.append(description).append(DateTools.secondTotime(seconds)).append(" ");
                    break;
                // 需支付金额（0.1元）
                case 8:
                    int amount = Integer.parseInt(value);
                    if (amount > 0) {
                        double yuan = BigDecimal.valueOf(amount).multiply(BigDecimal.valueOf(0.1)).doubleValue();
                        result.append(description).append(TagProcessor.convert(yuan)).append("元 ");
                    }
                    break;
                default: // 其他无值的变量
                    result.append(description).append(" ");
                    break;
            }
        }
        //// 去除末尾空格
        return result.toString().trim();
    }
}