package com.icetech.park.service.down.full.controlcard.dtong_led;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.qrcode.QRCodeWriter;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.service.down.full.controlcard.IControlCardTypeBuilder;
import lombok.extern.slf4j.Slf4j;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 *  道通控制卡
 * @author fangct
 */
@Slf4j
public class DtongLedCardService implements IControlCardTypeBuilder {

    @Override
    public List<byte[]> initControlCard(int volume, boolean enable, int quietVolume, LocalTime startTime, LocalTime endTime, int tts) {
        //设置音量
        List<byte[]> bytes = buildSetVolume(volume, enable, quietVolume, startTime, endTime);
        //同步时间
        byte[] bytes1 = DtongLedBuilder.buildSetTime(LocalDateTime.now());
        bytes.add(bytes1);
        return bytes;
    }

    @Override
    public List<byte[]> buildSetVolume(int volume, boolean enable, int quietVolume, LocalTime startTime, LocalTime endTime) {
        List<byte[]> list = new ArrayList<>();
        byte[] bytes = DtongLedBuilder.buildSetVolume(volume  * 10);
        list.add(bytes);
        return list;
    }

    @Override
    public byte[] buildSayByte(String say) {
        byte[] sayByte = new byte[0];
        if (StringUtils.isNotBlank(say)) {
            // 正则表达式匹配标签
            Pattern pattern = Pattern.compile("^<.*>$");
            Matcher matcher = pattern.matcher(say);
            if (matcher.matches()) {
                // 标签处理
                sayByte = DtongLedBuilder.buildPlayAudio(DtongLedDynamicContentConverter.convertToChinese(say));
            } else {
                // TTS文本转换
                sayByte = DtongLedBuilder.buildPlayAudio(say);
            }
        }
        return sayByte;
    }

    @Override
    public List<byte[]> buildShowBytes(String ledColors, int ledRestoreDefaultTime, HintRequest hintRequest, boolean busyStatus) {
        List<byte[]> showData = new ArrayList<>();
        // 检查显示内容是否为空
        if (StringUtils.isBlank(hintRequest.getShow())) {
            return showData;
        }
        // 获取颜色配置和 LED 配置
        String[] colors = ledColors.split("/");
        //包含二维码
        if (hintRequest.getShowType() == 2 || hintRequest.getShowType() == 3) {
            processShowAndQrCode(hintRequest, busyStatus, showData, colors, ledRestoreDefaultTime);
        } else {
            processShowTexts(hintRequest, busyStatus, showData, colors, ledRestoreDefaultTime);
        }
        return showData;
    }

    /**
     * 处理显示内容和二维码
     *
     * @param hintRequest 提示请求
     * @param busyStatus  繁忙状态
     * @param showData    显示数据列表
     * @param colors      颜色数组
     * @param ledRestoreDefaultTime   LED 配置
     */
    private void processShowAndQrCode(HintRequest hintRequest, boolean busyStatus, List<byte[]> showData, String[] colors, int ledRestoreDefaultTime) {
        log.info("[道通控制卡] 生成文本和二维码[{}]", hintRequest.getShow());
        if (hintRequest.getShow().contains("|")) {
            String[] shows = hintRequest.getShow().split("\\|");
            String qrCode = shows[1];
            // 屏显数据
            String[] lines = shows[0].split("/");
            for (int i = 0; i < lines.length; i++) {
                String line = lines[i];
                byte[] showBytes = buildSingleLineShowBytes(busyStatus, i, ledRestoreDefaultTime, Integer.parseInt(colors[i]), line);
                showData.add(showBytes);
            }
            // 二维码展示
            byte[] qrBytes = DtongLedBuilder.buildQrCodePageDrawMode((byte) ledRestoreDefaultTime, false, hintRequest.getExtendShow(), genQRcode(qrCode));
            showData.add(qrBytes);
        } else {
            // 二维码展示
            byte[] qrBytes = DtongLedBuilder.buildQrCodePageDrawMode((byte) ledRestoreDefaultTime, false, hintRequest.getExtendShow(), genQRcode(hintRequest.getShow()));
            showData.add(qrBytes);
        }
    }

    /**
     * 处理纯文本显示内容
     *
     * @param hintRequest 提示请求
     * @param busyStatus  繁忙状态
     * @param showData    显示数据列表
     * @param colors      颜色数组
     * @param ledRestoreDefaultTime   恢复空闲的时间
     */
    private void processShowTexts(HintRequest hintRequest, boolean busyStatus, List<byte[]> showData, String[] colors, int ledRestoreDefaultTime) {
        log.info("[道通控制卡] 生成纯文本内容[{}]", hintRequest.getShow());
        String[] showTexts = hintRequest.getShow().split("/");

        List<byte[]> colorList = Arrays.stream(colors).map(color -> getRGBA(Integer.parseInt(color))).collect(Collectors.toList());
        int displayTime = busyStatus ? Math.min(ledRestoreDefaultTime + 3, 255) : 255;
        byte[] bytes = DtongLedBuilder.buildShowMultiLineWithAudio(Arrays.asList(showTexts), colorList, "", (byte) displayTime);
        showData.add(bytes);
    }

    /**
     * 构建单行显示字节数组
     *
     * @param busyStatus 繁忙状态
     * @param index      显示索引
     * @param ledRestoreDefaultTime  恢复空闲的时间
     * @param color      颜色值
     * @param showText   显示文本
     * @return 显示字节数组
     */
    private byte[] buildSingleLineShowBytes(boolean busyStatus, int index, int ledRestoreDefaultTime, int color, String showText) {
        int displayTime = busyStatus ? Math.min(ledRestoreDefaultTime + 3, 255) : 255;
        return DtongLedBuilder.buildDownloadTempText((byte) index, (byte) displayTime, getRGBA(color), showText);
    }

    /**
     * 根据颜色编号获取对应的 RGBA 值
     *
     * @param colorCode 颜色编号（1: 红色，2: 绿色，3: 黄色）
     * @return 对应的 RGBA 字节数组
     */
    public static byte[] getRGBA(int colorCode) {
        // 默认透明度为 255（不透明）
        int r = 0, g = 0, b = 0, a = 255;

        switch (colorCode) {
            // 红色
            case 1:
                r = 255;
                g = 0;
                b = 0;
                break;
            // 绿色
            case 2:
                r = 0;
                g = 255;
                b = 0;
                break;
            // 黄色
            case 3:
                r = 255;
                g = 255;
                b = 0;
                break;
            default:
                throw new IllegalArgumentException("无效的颜色编号: " + colorCode);
        }

        // 将 RGBA 值转换为 byte[]
        return new byte[]{
                (byte) r,
                (byte) g,
                (byte) b,
                (byte) a
        };
    }
    // 生成二维码
    public byte[] genQRcode(String url) {
        // 二维码的宽度
        int width = 48;
        // 二维码的高度
        int height = 48;
        // 定义二维码参数
        Map<EncodeHintType, Object> hints = new HashMap<>();
        // 设置字符集
        hints.put(EncodeHintType.CHARACTER_SET, "UTF-8");
        hints.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.L);
        hints.put(EncodeHintType.MARGIN, 0);

        try {
            // 创建QR码配置
            QRCodeWriter qrCodeWriter = new QRCodeWriter();
            BitMatrix bitMatrix = qrCodeWriter.encode(
                    url,
                    BarcodeFormat.QR_CODE,
                    width,
                    height,
                    hints
            );

            // 创建二维码图像
            BufferedImage qrImage = new BufferedImage(
                    bitMatrix.getWidth(),
                    bitMatrix.getHeight(),
                    BufferedImage.TYPE_BYTE_BINARY
            );

            // 填充二维码图像
            for (int x = 0; x < bitMatrix.getWidth(); x++) {
                for (int y = 0; y < bitMatrix.getHeight(); y++) {
                    qrImage.setRGB(x, y, bitMatrix.get(x, y) ? 0xFF000000 : 0xFFFFFFFF);
                }
            }

            // 转换为BMP格式
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            ImageIO.write(qrImage, "BMP", output);
            byte[] bmpData = output.toByteArray();

            // 转换为16进制字符串
            StringBuilder hexStr = new StringBuilder();
            for (byte b : bmpData) {
                hexStr.append(String.format("%02X ", b));
            }
            log.info(hexStr.toString());
            return bmpData;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
