package com.icetech.park.service.down.full.controlcard.dtong_lcd;

import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.service.down.full.controlcard.IControlCardTypeBuilder;
import lombok.extern.slf4j.Slf4j;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 *  道通控制卡
 * @author fangct
 */
@Slf4j
public class DtongLcdCardService implements IControlCardTypeBuilder {

    private static final String[] COLORS = {"#FF0000", "#00FF00", "#FFFF00"};

    @Override
    public List<byte[]> initControlCard(int volume, boolean enable, int quietVolume, LocalTime startTime, LocalTime endTime, int tts) {
        //设置音量
        List<byte[]> bytes = buildSetVolume(volume, enable, quietVolume, startTime, endTime);
        //同步时间
        byte[] bytes1 = DtongLcdBuilder.buildSetTime(LocalDateTime.now());
        bytes.add(bytes1);
        //设置TTS
        byte[] bytes2 = DtongLcdBuilder.buildTTS(tts);
        bytes.add(bytes2);
        return bytes;
    }

    @Override
    public List<byte[]> buildSetVolume(int volume, boolean enable, int quietVolume, LocalTime startTime, LocalTime endTime) {
        List<byte[]> list = new ArrayList<>();
        byte[] bytes = DtongLcdBuilder.buildSetVolume(volume  * 10, volume  * 10, enable,
                startTime.getHour(), endTime.getHour(), startTime.getMinute(), endTime.getMinute(), quietVolume * 10);
        list.add(bytes);
        return list;
    }

    @Override
    public byte[] buildSayByte(String say) {
        byte[] sayByte = new byte[0];
        if (StringUtils.isNotBlank(say)) {
            // 正则表达式匹配标签
            Pattern pattern = Pattern.compile("^<.*>$");
            Matcher matcher = pattern.matcher(say);
            if (matcher.matches()) {
                // 标签处理
                sayByte = DtongLcdBuilder.buildPlayAudio(DtongLcdDynamicContentConverter.convertToChinese(say));
            } else {
                // TTS文本转换
                sayByte = DtongLcdBuilder.buildPlayAudio(say);
            }
        }
        return sayByte;
    }

    @Override
    public List<byte[]> buildShowBytes(String ledColors, int ledRestoreDefaultTime, HintRequest hintRequest, boolean busyStatus) {
        List<byte[]> showData = new ArrayList<>();
        // 检查显示内容是否为空
        if (StringUtils.isBlank(hintRequest.getShow())) {
            return showData;
        }
        // 获取颜色配置和 LED 配置
        String[] colors = ledColors.split("/");
        //包含二维码
        if (hintRequest.getShowType() == 2 || hintRequest.getShowType() == 3) {
            processShowAndQrCode(hintRequest, busyStatus, showData, colors, ledRestoreDefaultTime);
        } else {
            processShowTexts(hintRequest, busyStatus, showData, colors, ledRestoreDefaultTime);
        }
        return showData;
    }

    @Override
    public Boolean isMultiplePackSleep() {
        return true;
    }

    /**
     * 处理显示内容和二维码
     *
     * @param hintRequest 提示请求
     * @param busyStatus  繁忙状态
     * @param showData    显示数据列表
     * @param colors      颜色数组
     * @param ledRestoreDefaultTime   LED 配置
     */
    private void processShowAndQrCode(HintRequest hintRequest, boolean busyStatus, List<byte[]> showData, String[] colors, int ledRestoreDefaultTime) {
        log.info("[道通控制卡] 生成文本和二维码[{}]", hintRequest.getShow());
        if (hintRequest.getShow().contains("|")) {
            String[] shows = hintRequest.getShow().split("\\|");
            String qrCode = shows[1];
            // 屏显数据
            String[] lines = shows[0].split("/");
            List<String> texts = new ArrayList<>();
            for (int i = 0; i < lines.length; i++) {
                texts.add(COLORS[Integer.parseInt(colors[i]) - 1] + " " + lines[i]);
            }
            // 二维码展示
            byte[] qrBytes = DtongLcdBuilder.buildQrCodePage((byte) ledRestoreDefaultTime,
                    new byte[]{(byte)255, (byte)255, (byte)255, 0}, qrCode, texts);
            showData.add(qrBytes);
        } else {
            // 二维码展示
            byte[] qrBytes = DtongLcdBuilder.buildQrCodePage((byte) ledRestoreDefaultTime,
                    new byte[]{(byte)255, (byte)255, (byte)255, 0}, hintRequest.getShow(), Collections.singletonList(" "));
            showData.add(qrBytes);
        }
    }

    /**
     * 处理纯文本显示内容
     *
     * @param hintRequest 提示请求
     * @param busyStatus  繁忙状态
     * @param showData    显示数据列表
     * @param colors      颜色数组
     * @param ledRestoreDefaultTime   恢复空闲的时间
     */
    private void processShowTexts(HintRequest hintRequest, boolean busyStatus, List<byte[]> showData, String[] colors, int ledRestoreDefaultTime) {
        log.info("[道通控制卡] 生成纯文本内容[{}]", hintRequest.getShow());
        String[] showTexts = hintRequest.getShow().split("/");

        List<byte[]> colorList = Arrays.stream(colors).map(color -> getRGBA(Integer.parseInt(color))).collect(Collectors.toList());
        int displayTime = busyStatus ? Math.min(ledRestoreDefaultTime + 3, 255) : 255;
        byte[] bytes = DtongLcdBuilder.buildShowMultiLineWithAudio(Arrays.asList(showTexts), colorList, "", (byte) displayTime);
        showData.add(bytes);
    }

    /**
     * 构建单行显示字节数组
     *
     * @param busyStatus 繁忙状态
     * @param index      显示索引
     * @param ledRestoreDefaultTime  恢复空闲的时间
     * @param color      颜色值
     * @param showText   显示文本
     * @return 显示字节数组
     */
    private byte[] buildSingleLineShowBytes(boolean busyStatus, int index, int ledRestoreDefaultTime, int color, String showText) {
        int displayTime = busyStatus ? Math.min(ledRestoreDefaultTime + 3, 255) : 255;
        return DtongLcdBuilder.buildDownloadTempText((byte) index, (byte) displayTime, getRGBA(color), showText);
    }

    /**
     * 根据颜色编号获取对应的 RGBA 值
     *
     * @param colorCode 颜色编号（1: 红色，2: 绿色，3: 黄色）
     * @return 对应的 RGBA 字节数组
     */
    public static byte[] getRGBA(int colorCode) {
        // 默认透明度为 255（不透明）
        int r = 0, g = 0, b = 0, a = 255;

        switch (colorCode) {
            // 红色
            case 1:
                r = 255;
                g = 0;
                b = 0;
                break;
            // 绿色
            case 2:
                r = 0;
                g = 255;
                b = 0;
                break;
            // 黄色
            case 3:
                r = 255;
                g = 255;
                b = 0;
                break;
            default:
                throw new IllegalArgumentException("无效的颜色编号: " + colorCode);
        }

        // 将 RGBA 值转换为 byte[]
        return new byte[]{
                (byte) r,
                (byte) g,
                (byte) b,
                (byte) a
        };
    }
}
