package com.icetech.park.service.down.full.controlcard.dtong_lcd;

import org.springframework.util.Assert;

import java.nio.charset.Charset;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 道通LCD 485通用协议指令构建器
 * @author fangct
 * @version V2.8
 */
public class DtongLcdBuilder {
    /** 字符编码 */
    private static final Charset CHARSET = Charset.forName("GB2312");

    /**
     * 构建<code>设置TTS</code>指令
     *
     * @param tts 播报语音的文字，0关 1开
     * @return 二进制指令包
     */
    public static byte[] buildTTS(int tts) {
        // 构建数据包
        byte[] data = new byte[1];
        data[0] = (byte) tts;

        return build(DtongLcdCmd.SET_TTS, data);
    }
    /**
     * 构建<code>同步时间</code>指令
     * 该指令用于从外部同步显示屏的系统时间。显示屏的系统时间每天的误差在+-5秒之内。
     * 在需要极高精度的时间应用场合，应该每隔一段时间对显示屏同步时间。
     *
     * @param dateTime 日期时间，包含年、月、日、时、分、秒
     * @return 二进制指令包
     */
    public static byte[] buildSetTime(LocalDateTime dateTime) {
        // 验证年份范围
        int year = dateTime.getYear();
        Assert.isTrue(year >= 1970 && year <= 2099, "年份必须在1970-2099之间");

        // 构建数据包
        byte[] data = new byte[8];
        int offset = 0;

        // 添加年份（小端模式）
        data[offset++] = (byte) (year & 0xFF);        // 低字节
        data[offset++] = (byte) ((year >> 8) & 0xFF); // 高字节

        // 添加月份
        data[offset++] = (byte) dateTime.getMonthValue();

        // 添加日期
        data[offset++] = (byte) dateTime.getDayOfMonth();

        // 添加星期（1=星期日，2-7=星期一至六）
        data[offset++] = (byte) dateTime.getDayOfWeek().getValue();

        // 添加时间
        data[offset++] = (byte) dateTime.getHour();
        data[offset++] = (byte) dateTime.getMinute();
        data[offset++] = (byte) dateTime.getSecond();

        return build(DtongLcdCmd.SYNC_TIME, data);
    }

    /**
     * 构建<code>播放语音</code>指令
     * 该指令用于播放显示屏系统预存的300句常用短语或词组。
     * 语音文本的匹配方式采用词组或者短语进行匹配，在一些含有变量信息的文字中，如金钱、车牌号、日期等，显示屏会自动处理。
     * 在发送语音文本时，需要将词组和短语之间加入逗号或者句号分隔，显示屏才能正确的匹配。
     *
     * @param text 播报语音的文字，最大长度为254字节，支持ASCII和GBK2312编码
     * @return 二进制指令包
     */
    public static byte[] buildPlayAudio(String text) {
        Assert.notNull(text, "语音文本不能为空");
        byte[] textBytes = text.getBytes(CHARSET);
        Assert.isTrue(textBytes.length <= 254, "语音文本长度不能超过254字节");

        // 构建数据包
        byte[] data = new byte[1 + textBytes.length];
        data[0] = 0x02;  // 操作选项
        System.arraycopy(textBytes, 0, data, 1, textBytes.length);

        return build(DtongLcdCmd.PLAY_VOICE, data);
    }

    /**
     * 构建<code>加载广告内容</code>指令
     * 需间隔50ms，发下一条指令。该指令一定要做成手动触发。
     * 第一行和第四行，最大可输入30个汉字(60字节)，小于2个汉字，则定屏显示，超过2个汉字，则滚动显示。
     * 第二第三行的广告内容，建议分别使用8个汉字(16字节)，对开显示.
     *
     * @param line    行号，有效值1-4，其他值无效
     * @param color   显示颜色，有效期1-3，1=红色，2=绿色，3=黄色，其他值默认为1
     * @param content 广告内容
     * @return 二进制指令包
     */
    public static byte[] buildLoadAd(int line, int color, String content) {
        byte[] context = content.getBytes(CHARSET);
        byte[] bytes = new byte[context.length + 3];
        Assert.isTrue(bytes.length <= DtongLcdCmd.SHOW_AD.getMaxLen(), "数据超过长度限制");
        bytes[0] = (byte) line;
        bytes[1] = (byte) color;
        System.arraycopy(context, 0, bytes, 3, context.length);
        return build(DtongLcdCmd.SHOW_AD, bytes);
    }

    /**
     * 构建<code>调整音量</code>指令
     * 该指令用于设置文本播报音量和广告视频播放音量，支持时间段控制
     *
     * @param textVolume 文本播报音量百分比，取值范围0-100
     * @param videoVolume 视频播放音量百分比，取值范围0-100
     * @param enableTimeControl 是否启用时间段控制
     * @param startHour 时间段开始小时，取值范围0-23
     * @param startMinute 时间段开始分钟，取值范围0-59
     * @param endHour 时间段结束小时，取值范围0-23
     * @param endMinute 时间段结束分钟，取值范围0-59
     * @param quietVolume 时间段内的音量值，取值范围0-100
     * @return 二进制指令包
     */
    public static byte[] buildSetVolume(int textVolume, int videoVolume,
                                        boolean enableTimeControl,
                                        Integer startHour, Integer startMinute,
                                        Integer endHour, Integer endMinute,
                                        Integer quietVolume) {
        // 验证基本参数
        Assert.isTrue(textVolume >= 0 && textVolume <= 100, "文本播报音量必须在0-100之间");
        Assert.isTrue(videoVolume >= 0 && videoVolume <= 100, "视频播放音量必须在0-100之间");

        // 如果启用时间段控制，验证相关参数
        if (enableTimeControl) {
            Assert.notNull(startHour, "启用时间段控制时，开始小时不能为空");
            Assert.notNull(startMinute, "启用时间段控制时，开始分钟不能为空");
            Assert.notNull(endHour, "启用时间段控制时，结束小时不能为空");
            Assert.notNull(endMinute, "启用时间段控制时，结束分钟不能为空");
            Assert.notNull(quietVolume, "启用时间段控制时，夜间音量不能为空");

            Assert.isTrue(startHour >= 0 && startHour <= 23, "开始小时必须在0-23之间");
            Assert.isTrue(startMinute >= 0 && startMinute <= 59, "开始分钟必须在0-59之间");
            Assert.isTrue(endHour >= 0 && endHour <= 23, "结束小时必须在0-23之间");
            Assert.isTrue(endMinute >= 0 && endMinute <= 59, "结束分钟必须在0-59之间");
            Assert.isTrue(quietVolume >= 0 && quietVolume <= 100, "夜间音量必须在0-100之间");
        }

        // 计算数据包长度
        int dataLength = enableTimeControl ? 8 : 2;
        byte[] data = new byte[dataLength];
        int offset = 0;

        // 添加基本音量参数
        data[offset++] = (byte) textVolume;   // 文本播报音量
        data[offset++] = (byte) videoVolume;  // 视频播放音量

        // 如果启用时间段控制，添加时间段参数
        if (enableTimeControl) {
            data[offset++] = 0x01;  // 时间段控制开关
            data[offset++] = (byte) startHour.intValue();    // 开始小时
            data[offset++] = (byte) startMinute.intValue();  // 开始分钟
            data[offset++] = (byte) endHour.intValue();      // 结束小时
            data[offset++] = (byte) endMinute.intValue();    // 结束分钟
            data[offset++] = (byte) quietVolume.intValue();  // 勿扰音量
        }

        return build(DtongLcdCmd.ADJUST_VOL, data);
    }

    /**
     * 构建<code>设置继电器状态</code>指令
     * 需间隔50ms，发下一条指令。该指令一定要做成手动触发。
     *
     * @param channel 继电器通道，1-4
     * @param state   状态，0=关闭，1=打开
     * @return 二进制指令包
     */
    public static byte[] buildSetRelay(int channel, int state) {
        return build(DtongLcdCmd.SET_RELAY, new byte[]{(byte) channel, (byte) state});
    }

    /**
     * 构建<code>停止语音播放</code>指令
     * 需间隔50ms，发下一条指令。该指令一定要做成手动触发。
     *
     * @return 二进制指令包
     */
    public static byte[] buildStopAudio() {
        return build(DtongLcdCmd.STOP_VOICE, new byte[0]);
    }

    /**
     * 构建<code>下载临时文本</code>指令
     * 该指令用于显示临时文字信息，掉电不会保存
     *
     * @param windowId 显示窗口ID，表示第几行
     * @param displayTime 界面显示时间(秒)，0=使用模板默认值，255=一直显示，1-254=用户自定义时间
     * @param textColor 文字颜色，RGBA格式，每个分量0-255
     * @param text 显示的文字内容，支持ASCII和GBK2312编码
     * @return 二进制指令包
     */
    public static byte[] buildDownloadTempText(byte windowId,
                                               byte displayTime, byte[] textColor, String text) {
        Assert.isTrue(textColor.length == 4, "文字颜色必须是RGBA格式(4字节)");
        Assert.notNull(text, "显示文字不能为空");

        // 计算数据长度
        byte[] textBytes = text.getBytes(CHARSET);

        // 构建数据包
        byte[] data = new byte[1 + 1 + 1 + 1 + 1 + 2 + 1 + 1 + 4 + 4 + 1 + 1 + textBytes.length];
        int offset = 0;

        // 添加基本参数
        data[offset++] = windowId;     // 显示窗口ID
        data[offset++] = (byte) 0;  // 显示方式
        data[offset++] = 1;   // 进入速度
        data[offset++] = displayTime;  // 显示时间
        data[offset++] = 1;     // 停留时间

        // 添加保留字节RA1[2]
        for (int i = 0; i < 2; i++) {
            data[offset++] = 0x00;
        }

        // 添加字体和显示次数
        data[offset++] = 0x05;     // 字体索引
        data[offset++] = 0;   // 显示次数

        // 添加文字颜色
        System.arraycopy(textColor, 0, data, offset, 4);
        offset += 4;

        // 添加保留字节RA2[4]
        for (int i = 0; i < 4; i++) {
            data[offset++] = 0x00;
        }

        // 添加文字长度和保留字节
        data[offset++] = (byte)textBytes.length;  // 文字长度
        data[offset++] = 0x00;  // 保留字节R3

        // 添加文字内容
        System.arraycopy(textBytes, 0, data, offset, textBytes.length);

        return build(DtongLcdCmd.DOWNLOAD_TEMP, data);
    }

    /**
     * 构建<code>显示多行并播放语音</code>指令
     * 该指令用于单包多行显示，每行的内容用'\n'换行符分隔，最后一行用'\0'结束，语音用'\a'开始,'\0'结束
     * 文本头和文字以及语音的长度之和不能超出单包的最大长度(255字节)
     *
     * @param texts 多行文本
     * @param textColors 每行颜色，RGBA格式，每个分量0-255
     * @param voiceText 语音文本
     * @param displayTime 界面显示时间(秒)，0-5=使用模板默认值，254-255=一直显示，6-253=自定义时间
     * @return 二进制指令包
     */
    public static byte[] buildShowMultiLineWithAudio(List<String> texts, List<byte[]> textColors, String voiceText, int displayTime) {
        Assert.notEmpty(texts, "显示内容不能为空");
        Assert.isTrue(texts.size() <= 4, "最多支持4行显示");

        // 计算总长度
        int totalLen = 1 + 1 + 1;  // SF + GST + TEXT_CONTEXT_NUMBER

        // 计算每行文本长度
        for (String text : texts) {
            byte[] textBytes = text.getBytes(CHARSET);
            totalLen += 1 + 1 + 1 + 1 + 1 + 1 + 1 + 4 + 1 + textBytes.length + 1;
            // LID + DM + DS + DT + DR + FINDEX + FLAGS + TC[4] + TL + TEXT + 0x0D/0x00
        }

        // 计算语音部分长度
        int voiceLen = 0;
        byte[] voiceBytes = null;
        if (voiceText != null && !voiceText.isEmpty()) {
            voiceBytes = voiceText.getBytes(CHARSET);
            voiceLen = 1 + 1 + voiceBytes.length + 1;  // VF + VTL + VOICE + 0x00
        }

        // 计算图标部分长度
        int iconLen = 3;// IF + IN + IC

        // 验证总长度
        Assert.isTrue(totalLen + voiceLen + iconLen <= 255, "数据包总长度不能超过255字节");

        // 构建数据包
        byte[] data = new byte[totalLen + voiceLen + iconLen];
        int offset = 0;

        // 添加基本参数
        data[offset++] = (byte) 0;  // SF
        data[offset++] = (byte) displayTime;  // GST
        data[offset++] = (byte)texts.size();  // TEXT_CONTEXT_NUMBER

        // 添加每行文本
        for (int i = 0; i < texts.size(); i++) {
            byte[] textBytes = texts.get(i).getBytes(CHARSET);

            data[offset++] = (byte)i;  // LID
            data[offset++] = 0x00;  // DM
            data[offset++] = 0;  // DS
            data[offset++] = (byte) displayTime;  // DT
            data[offset++] = 0;  // DR
            data[offset++] = 0x05;  // FINDEX
            data[offset++] = 0x00;  // FLAGS

            // TC[4] - RGBA
            System.arraycopy(textColors.get(i), 0, data, offset, 4);
            offset += 4;

            data[offset++] = (byte)textBytes.length;  // TL
            System.arraycopy(textBytes, 0, data, offset, textBytes.length);
            offset += textBytes.length;

            // 添加分隔符
            data[offset++] = (byte)(i == texts.size() - 1 ? 0x00 : 0x0D);
        }

        // 添加语音信息
        if (voiceBytes != null) {
            data[offset++] = 0x0A;  // VF
            data[offset++] = (byte)voiceBytes.length;  // VTL
            System.arraycopy(voiceBytes, 0, data, offset, voiceBytes.length);
            offset += voiceBytes.length;
            data[offset++] = 0x00;  // 语音结束符
        }

        // 添加图标信息
        data[offset++] = 0x10;  // IF
        data[offset++] = (byte)0x00;                 // P字图标 // IN
        data[offset++] = 0x00;  // IC // 红色

        return build(DtongLcdCmd.MULTI_LINE, data);
    }

    /**
     * 构建<code>扫码支付页面</code>指令 (0xE1)
     * 该指令用于调用扫码支付页面，显示二维码和文本信息
     *
     * @param displayTime 显示时间，单位为秒，0为一直显示
     * @param qrc 二维码颜色，4字节RGBA格式，R=红色分量，G=绿色分量，B=蓝色分量，A保留值，各取值范围0-255
     * @param qrCodeMessage 二维码字符串
     * @param texts 文本信息字符串
     * @return 二进制指令包
     */
    public static byte[] buildQrCodePage(byte displayTime, byte[] qrc, String qrCodeMessage, List<String> texts) {
        // 参数验证
        Assert.notNull(qrc, "二维码颜色不能为空");
        Assert.notNull(qrCodeMessage, "二维码字符串不能为空");

        // 计算数据长度
        byte[] messageBytes = qrCodeMessage.getBytes(CHARSET);
        // 计算每行文本长度
        int totalTextLen = 0;
        for (String text : texts) {
            byte[] textBytes = text.getBytes(CHARSET);
            totalTextLen += textBytes.length + 1;
        }

        // 构建数据包
        byte[] data = new byte[1 + 1 + 1 + 1 + 1 + 4 + 4 + 1 + 1 + 1 + 1 + 15 +
                (messageBytes.length + 1) + totalTextLen];
        int offset = 0;

        // SF: 显示标志，固定为1
        data[offset++] = 0x00;

        // EM: 进入模式，固定为0
        data[offset++] = 0x00;

        // ETM: 退出模式，固定为0
        data[offset++] = 0x00;

        // ST: 显示时间
        data[offset++] = displayTime;

        // NI: 下一个界面索引，固定为0
        data[offset++] = 0x00;

        // QRC[4]: 二维码颜色(RGBA)
        System.arraycopy(qrc, 0, data, offset, 4);
        offset += 4;

        // RA1[4]: 保留字节
        for (int i = 0; i < 4; i++) {
            data[offset++] = 0x00;
        }

        // ML: 二维码信息长度
        data[offset++] = (byte) messageBytes.length;

        // TL: 文本信息长度
        data[offset++] = (byte) (totalTextLen - 1);

        // FLAGS: 标志域
        data[offset++] = (byte) 0x80;

        // QRSIZE: 二维码尺寸
        data[offset++] = 0;

        // RA2[15]: 保留字节
        for (int i = 0; i < 15; i++) {
            data[offset++] = 0x00;
        }

        // MSG: 二维码字符串
        System.arraycopy(messageBytes, 0, data, offset, messageBytes.length);
        offset += messageBytes.length;
        data[offset++] = 0x00;  // 结束符

        // TEXT: 文本信息
        // 添加每行文本
        for (int i = 0; i < texts.size(); i++) {
            byte[] textBytes = texts.get(i).getBytes(CHARSET);

            System.arraycopy(textBytes, 0, data, offset, textBytes.length);
            offset += textBytes.length;

            // 添加分隔符
            data[offset++] = (byte)(i == texts.size() - 1 ? 0x00 : 0x0D);
        }

        return build(DtongLcdCmd.QR_PAY, data);
    }

    /**
     * 构建<code>扫码支付页面-绘图模式</code>指令
     * 该指令通过绘图方式调用扫码支付页面，支持29x29和45x45的二维码位图
     *
     * @param displayFlag 显示标志，1为显示，0为不显示
     * @param colorIndex 二维码颜色索引，0=红，1=绿，2=蓝，3=黄，4=青，5=紫，6=白
     * @param displayTime 显示时间，单位为秒，0为一直显示
     * @param playVoice 是否播报语音
     * @param textMessage 文本信息字符串
     * @param qrCodeBitmap 二维码单色位图数据，支持29x29和45x45
     * @return 二进制指令包
     */
    public static byte[] buildQrCodePageDrawMode(byte displayFlag, byte colorIndex, byte displayTime,
                                                 boolean playVoice, String textMessage, byte[] qrCodeBitmap) {
        Assert.notNull(textMessage, "文本信息不能为空");
        Assert.notNull(qrCodeBitmap, "二维码位图数据不能为空");

        // 计算数据长度
        byte[] textBytes = textMessage.getBytes(CHARSET);
        int dataLength = 1 + 1 + 1 + 1 + 1 + 1 + 1 + textBytes.length + qrCodeBitmap.length;

        // 构建数据包
        byte[] data = new byte[dataLength];
        int offset = 0;

        // 添加固定参数
        data[offset++] = displayFlag;  // 显示标志
        data[offset++] = 0x00;         // 进入模式，固定为0
        data[offset++] = colorIndex;   // 二维码颜色索引
        data[offset++] = displayTime;  // 显示时间
        data[offset++] = 0x00;         // 下一个界面索引，固定为0
        data[offset++] = playVoice ? (byte)0x01 : (byte)0x00;  // 语音播报开关
        data[offset++] = (byte)textBytes.length;  // 文本长度

        // 添加文本信息
        System.arraycopy(textBytes, 0, data, offset, textBytes.length);
        offset += textBytes.length;

        // 添加二维码位图数据
        System.arraycopy(qrCodeBitmap, 0, data, offset, qrCodeBitmap.length);

        return build(DtongLcdCmd.QR_PAY_V2, data);
    }

    /**
     * 构建<code>显示无牌车扫码页面</code>指令
     * 该指令用于显示无牌车扫码页面，包含二维码、文本和语音信息
     *
     * @param displayTime 显示时间，单位为秒，0为一直显示
     * @param qrCodeMessage 二维码字符串
     * @param textMessage 文本信息字符串，多行用0x0a或0x0d分行，最后一行用0x00结束
     * @param voiceMessage 语音文本字符串，包含最后结束符00
     * @return 二进制指令包
     */
    public static byte[] buildNoPlateQrCodePage(byte displayTime, String qrCodeMessage,
                                                String textMessage, String voiceMessage) {
        Assert.notNull(qrCodeMessage, "二维码字符串不能为空");
        Assert.notNull(textMessage, "文本信息不能为空");
        Assert.notNull(voiceMessage, "语音文本不能为空");

        // 计算数据长度
        byte[] qrBytes = qrCodeMessage.getBytes(CHARSET);
        byte[] textBytes = textMessage.getBytes(CHARSET);
        byte[] voiceBytes = voiceMessage.getBytes(CHARSET);

        // 构建数据包
        byte[] data = new byte[3 + 1 + 5 + 1 + 1 + 1 +
                (qrBytes.length + 1) +
                (textBytes.length + 1) +
                (voiceBytes.length + 1)];
        int offset = 0;

        // 添加保留字节RA1[3]
        for (int i = 0; i < 3; i++) {
            data[offset++] = 0x00;
        }

        // 添加显示时间
        data[offset++] = displayTime;

        // 添加保留字节RA2[5]
        for (int i = 0; i < 5; i++) {
            data[offset++] = 0x00;
        }

        // 添加长度信息
        data[offset++] = (byte)qrBytes.length;    // 二维码长度
        data[offset++] = (byte)textBytes.length;  // 文本长度
        data[offset++] = (byte)voiceBytes.length; // 语音长度

        // 添加二维码信息
        System.arraycopy(qrBytes, 0, data, offset, qrBytes.length);
        offset += qrBytes.length;
        data[offset++] = 0x00;  // 结束符

        // 添加文本信息
        System.arraycopy(textBytes, 0, data, offset, textBytes.length);
        offset += textBytes.length;
        data[offset++] = 0x00;  // 结束符

        // 添加语音信息
        System.arraycopy(voiceBytes, 0, data, offset, voiceBytes.length);
        offset += voiceBytes.length;
        data[offset++] = 0x00;  // 结束符

        return build(DtongLcdCmd.NO_PLATE_QR, data);
    }

    /**
     * 构建指令
     *
     * @param cmd  指令类型
     * @param data 指令数据包
     * @return 二进制指令包
     */
    public static byte[] build(DtongLcdCmd cmd, byte[] data) {
        int dataLen = data == null || data.length == 0 ? 0 : data.length;
        Assert.isTrue(dataLen <= cmd.getMaxLen(), "数据超过命令允许的最大长度");

        // 构建数据包
        byte[] bytes = new byte[dataLen + 8]; // DA(1) + VR(1) + PN(2) + CMD(1) + DL(1) + DATA(n) + CRC(2)
        int offset = 0;
        bytes[offset++] = 0x00;  // DA: 设备地址，默认0x00
        bytes[offset++] = (byte) 0x64;  // VR: 协议版本，默认0x64(100)
        bytes[offset++] = (byte) 0XFFFF;  // PN: 包序列
        bytes[offset++] = (byte) 0XFFFF;  // PN: 包序列
        bytes[offset++] = cmd.getCmd();  // CMD: 命令字节
        bytes[offset++] = (byte) dataLen;  // DL: 数据长度

        if (dataLen > 0) {
            System.arraycopy(data, 0, bytes, offset, dataLen);
            offset += dataLen;
        }

        // 计算CRC16校验码
        int crc = calculateCRC16(bytes, 0, bytes.length - 2);
        bytes[offset++] = (byte) (crc & 0xFF);  // CRC低字节
        bytes[offset++] = (byte) ((crc >> 8) & 0xFF);  // CRC高字节

        return bytes;
    }

    public static byte[] hexToBytes(String hex) {
        int len = hex.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(hex.charAt(i), 16) << 4)
                    + Character.digit(hex.charAt(i + 1), 16));
        }
        return data;
    }
    /**
     * 查表法计算CRC16校验
     *
     * @param data 需要计算的字节数组
     */
    public static int calculateCRC16(byte[] data, int offset, int len) {
        byte[] crc16H = {
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40,
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40,
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41,
                (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x01, (byte) 0xC0, (byte) 0x80, (byte) 0x41, (byte) 0x00, (byte) 0xC1, (byte) 0x81, (byte) 0x40
        };

        byte[] crc16L = {
                (byte) 0x00, (byte) 0xC0, (byte) 0xC1, (byte) 0x01, (byte) 0xC3, (byte) 0x03, (byte) 0x02, (byte) 0xC2, (byte) 0xC6, (byte) 0x06, (byte) 0x07, (byte) 0xC7, (byte) 0x05, (byte) 0xC5, (byte) 0xC4, (byte) 0x04,
                (byte) 0xCC, (byte) 0x0C, (byte) 0x0D, (byte) 0xCD, (byte) 0x0F, (byte) 0xCF, (byte) 0xCE, (byte) 0x0E, (byte) 0x0A, (byte) 0xCA, (byte) 0xCB, (byte) 0x0B, (byte) 0xC9, (byte) 0x09, (byte) 0x08, (byte) 0xC8,
                (byte) 0xD8, (byte) 0x18, (byte) 0x19, (byte) 0xD9, (byte) 0x1B, (byte) 0xDB, (byte) 0xDA, (byte) 0x1A, (byte) 0x1E, (byte) 0xDE, (byte) 0xDF, (byte) 0x1F, (byte) 0xDD, (byte) 0x1D, (byte) 0x1C, (byte) 0xDC,
                (byte) 0x14, (byte) 0xD4, (byte) 0xD5, (byte) 0x15, (byte) 0xD7, (byte) 0x17, (byte) 0x16, (byte) 0xD6, (byte) 0xD2, (byte) 0x12, (byte) 0x13, (byte) 0xD3, (byte) 0x11, (byte) 0xD1, (byte) 0xD0, (byte) 0x10,
                (byte) 0xF0, (byte) 0x30, (byte) 0x31, (byte) 0xF1, (byte) 0x33, (byte) 0xF3, (byte) 0xF2, (byte) 0x32, (byte) 0x36, (byte) 0xF6, (byte) 0xF7, (byte) 0x37, (byte) 0xF5, (byte) 0x35, (byte) 0x34, (byte) 0xF4,
                (byte) 0x3C, (byte) 0xFC, (byte) 0xFD, (byte) 0x3D, (byte) 0xFF, (byte) 0x3F, (byte) 0x3E, (byte) 0xFE, (byte) 0xFA, (byte) 0x3A, (byte) 0x3B, (byte) 0xFB, (byte) 0x39, (byte) 0xF9, (byte) 0xF8, (byte) 0x38,
                (byte) 0x28, (byte) 0xE8, (byte) 0xE9, (byte) 0x29, (byte) 0xEB, (byte) 0x2B, (byte) 0x2A, (byte) 0xEA, (byte) 0xEE, (byte) 0x2E, (byte) 0x2F, (byte) 0xEF, (byte) 0x2D, (byte) 0xED, (byte) 0xEC, (byte) 0x2C,
                (byte) 0xE4, (byte) 0x24, (byte) 0x25, (byte) 0xE5, (byte) 0x27, (byte) 0xE7, (byte) 0xE6, (byte) 0x26, (byte) 0x22, (byte) 0xE2, (byte) 0xE3, (byte) 0x23, (byte) 0xE1, (byte) 0x21, (byte) 0x20, (byte) 0xE0,
                (byte) 0xA0, (byte) 0x60, (byte) 0x61, (byte) 0xA1, (byte) 0x63, (byte) 0xA3, (byte) 0xA2, (byte) 0x62, (byte) 0x66, (byte) 0xA6, (byte) 0xA7, (byte) 0x67, (byte) 0xA5, (byte) 0x65, (byte) 0x64, (byte) 0xA4,
                (byte) 0x6C, (byte) 0xAC, (byte) 0xAD, (byte) 0x6D, (byte) 0xAF, (byte) 0x6F, (byte) 0x6E, (byte) 0xAE, (byte) 0xAA, (byte) 0x6A, (byte) 0x6B, (byte) 0xAB, (byte) 0x69, (byte) 0xA9, (byte) 0xA8, (byte) 0x68,
                (byte) 0x78, (byte) 0xB8, (byte) 0xB9, (byte) 0x79, (byte) 0xBB, (byte) 0x7B, (byte) 0x7A, (byte) 0xBA, (byte) 0xBE, (byte) 0x7E, (byte) 0x7F, (byte) 0xBF, (byte) 0x7D, (byte) 0xBD, (byte) 0xBC, (byte) 0x7C,
                (byte) 0xB4, (byte) 0x74, (byte) 0x75, (byte) 0xB5, (byte) 0x77, (byte) 0xB7, (byte) 0xB6, (byte) 0x76, (byte) 0x72, (byte) 0xB2, (byte) 0xB3, (byte) 0x73, (byte) 0xB1, (byte) 0x71, (byte) 0x70, (byte) 0xB0,
                (byte) 0x50, (byte) 0x90, (byte) 0x91, (byte) 0x51, (byte) 0x93, (byte) 0x53, (byte) 0x52, (byte) 0x92, (byte) 0x96, (byte) 0x56, (byte) 0x57, (byte) 0x97, (byte) 0x55, (byte) 0x95, (byte) 0x94, (byte) 0x54,
                (byte) 0x9C, (byte) 0x5C, (byte) 0x5D, (byte) 0x9D, (byte) 0x5F, (byte) 0x9F, (byte) 0x9E, (byte) 0x5E, (byte) 0x5A, (byte) 0x9A, (byte) 0x9B, (byte) 0x5B, (byte) 0x99, (byte) 0x59, (byte) 0x58, (byte) 0x98,
                (byte) 0x88, (byte) 0x48, (byte) 0x49, (byte) 0x89, (byte) 0x4B, (byte) 0x8B, (byte) 0x8A, (byte) 0x4A, (byte) 0x4E, (byte) 0x8E, (byte) 0x8F, (byte) 0x4F, (byte) 0x8D, (byte) 0x4D, (byte) 0x4C, (byte) 0x8C,
                (byte) 0x44, (byte) 0x84, (byte) 0x85, (byte) 0x45, (byte) 0x87, (byte) 0x47, (byte) 0x46, (byte) 0x86, (byte) 0x82, (byte) 0x42, (byte) 0x43, (byte) 0x83, (byte) 0x41, (byte) 0x81, (byte) 0x80, (byte) 0x40
        };
        int ucCRCHi = 0x00ff;
        int ucCRCLo = 0x00ff;
        int iIndex;
        for (int i = offset; i < len; ++i) {
            iIndex = (ucCRCLo ^ data[i]) & 0x00ff;
            ucCRCLo = ucCRCHi ^ crc16H[iIndex];
            ucCRCHi = crc16L[iIndex];
        }
        return  ((ucCRCHi & 0x00ff) << 8) | (ucCRCLo & 0x00ff) & 0xffff;
    }
} 