package com.icetech.park.service.down;

import com.icetech.park.service.down.full.controlcard.vertical_2x8.AudioCode;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * TTS 转中文
 *
 * @author wgq
 */
@Slf4j
public class TagProcessor {
    // 存储标签及其对应的处理逻辑的有序Map
    private static final Map<String, TagHandler> TAG_HANDLERS = new LinkedHashMap<>();

    private static final String[] UNIT = {"", "十", "百", "千", "万"};

    private static final String[] DIGITS = {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9"};

    // 初始化标签和处理逻辑
    static {
        initTagHandlers();
    }

    /**
     * 初始化标签和处理逻辑
     */
    private static void initTagHandlers() {
        TAG_HANDLERS.put("4", new TagHandler(Collections.emptyList(), Collections.emptyList(), 1.0));
        TAG_HANDLERS.put("6", new TagHandler(Arrays.asList(AudioCode._34, AudioCode._61), Collections.singletonList(AudioCode._163), 1.0));
        TAG_HANDLERS.put("7", new TagHandler(Collections.singletonList(AudioCode._46), Collections.emptyList(), 1.0));
        TAG_HANDLERS.put("8", new TagHandler(Collections.singletonList(AudioCode._11), Collections.singletonList(AudioCode._200), 0.1));
        TAG_HANDLERS.put("9", new TagHandler(Collections.singletonList(AudioCode._2)));
        TAG_HANDLERS.put("10", new TagHandler(Collections.singletonList(AudioCode._1)));
        TAG_HANDLERS.put("11", new TagHandler(Collections.singletonList(AudioCode._3)));
        TAG_HANDLERS.put("12", new TagHandler(Collections.singletonList(AudioCode._10)));
        TAG_HANDLERS.put("13", new TagHandler(Collections.singletonList(AudioCode._43)));
        TAG_HANDLERS.put("14", new TagHandler(Collections.singletonList(AudioCode._33)));
        TAG_HANDLERS.put("15", new TagHandler(Collections.singletonList(AudioCode._34)));
    }

    /**
     * 处理输入字符串，提取标签内容并生成对应的中文提示信息。
     * 如果没有标签，则直接返回输入的文本。
     *
     * @param input 输入的字符串
     * @return 处理后的字节数组
     */
    public static List<AudioCode> processTags(String input) {
        List<AudioCode> audioCodes = new ArrayList<>();
        // 匹配所有标签的正则表达式
        Pattern pattern = Pattern.compile("<(\\d+)/>|<(\\d+)>([^<]*)</\\d+>|</(\\d+)>");
        Matcher matcher = pattern.matcher(input);
        // 遍历匹配到的所有标签
        while (matcher.find()) {
            if (matcher.group(1) != null) {
                // 匹配这种标签<10/>
                getAudioCode(audioCodes, matcher.group(1), null);
            } else if (matcher.group(2) != null) {
                // 匹配这种标签<4>粤BF65234</4>
                getAudioCode(audioCodes, matcher.group(2), matcher.group(3));
            } else if (matcher.group(4) != null) {
                // 匹配这种标签</9>
                getAudioCode(audioCodes, matcher.group(4), null);
            }
        }
        return audioCodes;
    }

    /**
     * 组装AudioCode
     *
     * @param audioCodes list
     * @param tag        标签
     * @param content    内容
     */
    private static void getAudioCode(List<AudioCode> audioCodes, String tag, String content) {
        TagHandler handler = TAG_HANDLERS.get(tag);
        if (handler != null) {
            // 使用处理器处理标签内容并添加到结果列表中
            audioCodes.addAll(handler.handle(tag, content));
        }
    }

    /**
     * 标签处理类
     */
    private static class TagHandler {
        private final List<AudioCode> prefix;
        private final List<AudioCode> suffix;
        private final double multiplier;

        // 处理简单标签的构造函数
        TagHandler(List<AudioCode> message) {
            this(message, Collections.emptyList(), 1.0);
        }

        // 处理复杂标签的构造函数
        TagHandler(List<AudioCode> prefix, List<AudioCode> suffix, double multiplier) {
            this.prefix = prefix;
            this.suffix = suffix;
            this.multiplier = multiplier;
        }

        // 处理标签内容的方法，返回字节数组
        List<AudioCode> handle(String tag, String content) {
            List<AudioCode> audioCodes = new ArrayList<>();
            // 根据需要进行转换
            if (multiplier != 1.0) {
                double d = Double.parseDouble(String.format("%.2f", Double.parseDouble(content) * multiplier));
                content = String.valueOf(convert(d));
            }
            if (isNumeric(content) && !"7".equals(tag)) {
                double d = Double.parseDouble(content);
                content = String.valueOf(convert(d));
            }
            // 写入前缀、内容和后缀
            if (!prefix.isEmpty()) {
                audioCodes.addAll(prefix);
            }
            stringGetBytes(audioCodes, tag, content);
            if (!suffix.isEmpty()) {
                audioCodes.addAll(suffix);
            }
            return audioCodes;
        }
    }

    /**
     * 将字符串转换为字节数组的辅助方法
     */
    @SneakyThrows
    private static void stringGetBytes(List<AudioCode> audioCodes, String tag, String content) {
        // 定义需要处理的标签集合
        Set<String> tagsToProcess = new HashSet<>(Arrays.asList("4", "6", "8"));
        // 检查标签是否在需要处理的集合中
        if ("7".equals(tag)) {
            assembleDate(audioCodes, content);
            return;
        }
        if (!tagsToProcess.contains(tag)) {
            audioCodes.add(AudioCode._64);
            return;
        }
        // 遍历字符串中的每个字符
        content.chars().mapToObj(c -> AudioCode.resolveCode(String.valueOf((char) c)))
                .forEach(audioCode -> audioCodes.add(audioCode != null ? audioCode : AudioCode._64));

    }

    /**
     * 转换成中文
     *
     * @param number 数字
     * @return str
     */
    public static String convert(double number) {
        if (number == 0) {
            return DIGITS[0];
        }
        StringBuilder result = new StringBuilder();
        String[] parts = String.valueOf(number).split("\\.");
        int intPart = Integer.parseInt(parts[0]);
        String numStr = String.valueOf(intPart);
        int length = numStr.length();
        for (int i = 0; i < length; i++) {
            int digit = numStr.charAt(i) - '0';
            if (digit != 0) {
                result.append(DIGITS[digit]).append(UNIT[length - i - 1]);
            } else if (i + 1 < length && numStr.charAt(i + 1) != '0') {
                result.append(DIGITS[digit]);
            }
        }
        String converted = result.toString().replaceAll("0+", "0").replaceAll("0$", "");
        converted = converted.startsWith("10") ? converted.substring(1) : converted;
        if (parts.length > 1 && !parts[1].matches("0+")) {
            if (intPart == 0) {
                result = new StringBuilder("0").append("点");
            } else {
                result = new StringBuilder(converted).append("点");
            }
            for (char ch : parts[1].toCharArray()) {
                result.append(DIGITS[ch - '0']);
            }
            converted = result.toString();
        }
        return converted;
    }

    /**
     * 时间转换
     *
     * @param audioCodes 下发内容
     * @param content    内容
     */
    private static void assembleDate(List<AudioCode> audioCodes, String content) {
        int time = Integer.parseInt(content);
        int days = time / (24 * 3600);
        int hours = (time % (24 * 3600)) / 3600;
        int minutes = (time % 3600) / 60;
        int seconds = time % 60;
        if (days > 0) {
            convert(days).chars().mapToObj(c -> AudioCode.resolveCode(String.valueOf((char) c)))
                    .forEach(audioCode -> audioCodes.add(audioCode != null ? audioCode : AudioCode._64));
            audioCodes.add(AudioCode._163);
        }
        if (hours > 0) {
            convert(hours).chars().mapToObj(c -> AudioCode.resolveCode(String.valueOf((char) c)))
                    .forEach(audioCode -> audioCodes.add(audioCode != null ? audioCode : AudioCode._64));
            audioCodes.add(AudioCode._47);
        }
        if (minutes > 0) {
            convert(minutes).chars().mapToObj(c -> AudioCode.resolveCode(String.valueOf((char) c)))
                    .forEach(audioCode -> audioCodes.add(audioCode != null ? audioCode : AudioCode._64));
            audioCodes.add(AudioCode._203);
        }
        convert(seconds).chars().mapToObj(c -> AudioCode.resolveCode(String.valueOf((char) c)))
                .forEach(audioCode -> audioCodes.add(audioCode != null ? audioCode : AudioCode._64));
        audioCodes.add(AudioCode._237);
    }

    /**
     * 是否是纯数字
     *
     * @param str 内容
     * @return true false
     */
    private static boolean isNumeric(String str) {
        return str != null && str.matches("[0-9]+");
    }
}
