package com.icetech.park.service.down;

import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.down.p2c.impl.ClearListServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.icetech.basics.constants.TextConstant.ERROR_410;

@Slf4j
public class SendMsgCommon implements SendMsgService {

    @Autowired
    protected ParkService parkService;
    @Autowired
    protected ParkDeviceService parkDeviceService;
    @Autowired
    protected CacheHandle cacheHandle;
    @Autowired
    protected ClearListServiceImpl clearListService;

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2Park(SendRequest sendRequest, String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getParkDevices(sendRequest);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2MasterChannel(SendRequest sendRequest, String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getMasterChannelAndEnExTypeDevices(sendRequest, null);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2Channel(SendRequest sendRequest, String parkCode, String channelCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getParkDevicesByChannelCode(sendRequest, channelCode);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2MasterExit(SendRequest sendRequest, String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getMasterChannelAndEnExTypeDevices(sendRequest, 2);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2RegionSubChannel(SendRequest sendRequest, String parkCode, Long regionId, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getRegionSubChannelDevices(sendRequest, regionId);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2ExcludeMasterExit(SendRequest sendRequest, String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getExcludeMasterExitDevices(sendRequest);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2OtherMasterExit(String serialNumber, SendRequest sendRequest, String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getMasterChannelAndEnExTypeDevices(sendRequest, 2);
        String newDevices = sendDevices.replaceAll(serialNumber + ";", "");
        if (newDevices.isEmpty()){
            return ObjectResponse.failed(CodeConstants.ERROR_406);
        }
        return send(message, newDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2Exit(SendRequest sendRequest, String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getParkDevicesByEnexType(message, sendRequest, 2, null);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2Enter(SendRequest sendRequest, String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getParkDevicesByEnexType(message, sendRequest, 1, null);
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2ParkOtherExit(String serialNumber, SendRequest sendRequest,
                                                             String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        String sendDevices = getParkDevicesByEnexType(message, sendRequest, 2, serialNumber);
        String newDevices = sendDevices.replaceAll(serialNumber + ";", "");
        if (newDevices.isEmpty()){
            return ObjectResponse.failed(CodeConstants.ERROR_406);
        }
        return send(message, newDevices, parkCode);
    }

    @SafeVarargs
    @Override
    public final <T, R> ObjectResponse<R> send2Devices(String sendDevices, SendRequest sendRequest,
                                                       String parkCode, T... objPara) {
        Message<T> message = buildMessage(sendRequest, objPara);
        boolean isSaveCache = sendRequest.getServiceId() != null;
        if (isSaveCache) {
            String newSendDevices = cacheHandle.getSendDevices(sendRequest);
            if (newSendDevices != null){
                sendDevices = newSendDevices;
            }else{
                cacheHandle.setSendDevices(sendRequest, sendDevices);
            }
        }
        return send(message, sendDevices, parkCode);
    }

    @SafeVarargs
    private final <T> Message<T> buildMessage(SendRequest sendRequest, T... objPara) {
        Message<T> message = new Message<>();
        BeanUtils.copyProperties(sendRequest, message);
        message.setPayload(objPara[0]);
        if (objPara.length > 1) {
            if (objPara[1] != null) {
                log.info("设置下一个下发的内容[{}]", objPara[1]);
                message.setNextPayLoad(objPara[1]);
            }
        }
        return message;
    }

    private <T> String getParkDevicesByEnexType(Message<T> message, SendRequest sendRequest,
                                            int enexType, String excludeSerialNumber) {
        ObjectResponse<List<ParkInoutdevice>> objectResponse =
                parkService.getChannelByType(message.getParkId(), enexType);
        ObjectResponse.notError(objectResponse, "未获取到通道信息");
        boolean isSaveCache = sendRequest.getServiceId() != null;
        String sendDevices = null;
        if (isSaveCache) {
            sendDevices = cacheHandle.getSendDevices(sendRequest);
        }
        if (sendDevices != null){
            return sendDevices;
        }
        sendDevices = "";
        List<ParkInoutdevice> inoutdevices = objectResponse.getData();
        for (ParkInoutdevice inoutdevice : inoutdevices) {
            ObjectResponse<List<ParkDevice>> deviceListResponse =
                    parkDeviceService.getDeviceListByParkChannelId(message.getParkId(),
                            inoutdevice.getId().intValue(), 1);
            if (ObjectResponse.isSuccess(deviceListResponse)) {
                List<ParkDevice> parkDevices = deviceListResponse.getData();
                for (ParkDevice parkDevice : parkDevices) {
                    String serialNumber = parkDevice.getSerialNumber();
                    if (excludeSerialNumber == null || !excludeSerialNumber.contains(serialNumber)) {
                        sendDevices += serialNumber + ";";
                    }
                }
            } else {
                log.info("通道{}下尚未添加设备", inoutdevice.getInandoutCode());
            }
        }
        if (isSaveCache) {
            cacheHandle.setSendDevices(sendRequest, sendDevices);
        }
        return sendDevices;
    }

    private String getParkDevices(SendRequest sendRequest) {
        String sendDevices = null;
        boolean isSaveCache = sendRequest.getServiceId() != null;
        if (isSaveCache){
            sendDevices = cacheHandle.getSendDevices(sendRequest);
        }
        if (sendDevices != null){
            return sendDevices;
        }
        //第一次下发
        ObjectResponse<List<ParkDevice>> objectResponse =
                parkDeviceService.getDeviceListByType(sendRequest.getParkId(), 1);
        if (!ObjectResponse.isSuccess(objectResponse)) {
            throw new ResponseBodyException(ERROR_410, "未添加识别摄像机");
        }
        List<ParkDevice> parkDevices = objectResponse.getData();
        sendDevices = parkDevices.stream()
                .filter(Objects::nonNull)
                .map(ParkDevice::getSerialNumber)
                .filter(Objects::nonNull)
                .collect(Collectors.joining(";")) + ";";
        if (isSaveCache) {
            cacheHandle.setSendDevices(sendRequest, sendDevices);
        }
        return sendDevices;
    }
    private String getParkDevicesByChannelCode(SendRequest sendRequest, String channelCode) {
        String sendDevices = null;
        boolean isSaveCache = sendRequest.getServiceId() != null;
        if (isSaveCache){
            sendDevices = cacheHandle.getSendDevices(sendRequest);
        }
        if (sendDevices != null){
            return sendDevices;
        }
        //第一次下发
        ObjectResponse<List<ParkDevice>> objectResponse =
                parkDeviceService.getDeviceByChannel(sendRequest.getParkId(), channelCode, 1);
        if (!ObjectResponse.isSuccess(objectResponse)) {
            throw new ResponseBodyException(ERROR_410, "未添加识别摄像机");
        }
        List<ParkDevice> parkDevices = objectResponse.getData();
        sendDevices = parkDevices.stream()
                .filter(Objects::nonNull)
                .map(ParkDevice::getSerialNumber)
                .filter(Objects::nonNull)
                .collect(Collectors.joining(";")) + ";";
        if (isSaveCache) {
            cacheHandle.setSendDevices(sendRequest, sendDevices);
        }
        return sendDevices;
    }

    private String getMasterChannelAndEnExTypeDevices(SendRequest sendRequest, Integer enExType) {
        ObjectResponse<List<ParkInoutdevice>> objectResponse =
                parkService.getChannelByType(sendRequest.getParkId(), enExType);
        ObjectResponse.notError(objectResponse, "未获取到通道信息");
        boolean isSaveCache = sendRequest.getServiceId() != null;
        String sendDevices = null;
        if (isSaveCache) {
            sendDevices = cacheHandle.getSendDevices(sendRequest);
        }
        if (sendDevices != null){
            return sendDevices;
        }
        sendDevices = "";
        List<ParkInoutdevice> inoutdevices = objectResponse.getData();
        inoutdevices = inoutdevices.stream()
                .filter(inout -> Integer.valueOf(1).equals(inout.getIsMaster()))
                .collect(Collectors.toList());
        for (ParkInoutdevice inoutdevice : inoutdevices) {
            ObjectResponse<List<ParkDevice>> deviceListResponse =
                    parkDeviceService.getDeviceListByParkChannelId(sendRequest.getParkId(),
                            inoutdevice.getId().intValue(), 1);
            if (ObjectResponse.isSuccess(deviceListResponse)) {
                List<ParkDevice> parkDevices = deviceListResponse.getData();
                for (ParkDevice parkDevice : parkDevices) {
                    String serialNumber = parkDevice.getSerialNumber();
                    sendDevices += serialNumber + ";";
                }
            } else {
                log.info("通道{}下尚未添加设备", inoutdevice.getInandoutCode());
            }
        }
        if (isSaveCache) {
            cacheHandle.setSendDevices(sendRequest, sendDevices);
        }
        return sendDevices;
    }

    private String getRegionSubChannelDevices(SendRequest sendRequest, Long regionId) {
        ObjectResponse<List<ParkInoutdevice>> objectResponse =
                parkService.getAllChannel(sendRequest.getParkId());
        ObjectResponse.notError(objectResponse, "未获取到通道信息");
        boolean isSaveCache = sendRequest.getServiceId() != null;
        String sendDevices = null;
        if (isSaveCache) {
            sendDevices = cacheHandle.getSendDevices(sendRequest);
        }
        if (sendDevices != null){
            return sendDevices;
        }
        sendDevices = "";
        List<ParkInoutdevice> inoutdevices = objectResponse.getData();
        inoutdevices = inoutdevices.stream()
                .filter(inout -> Integer.valueOf(0).equals(inout.getIsMaster()))
                .filter(inout -> inout.getRegionId().equals(regionId))
                .collect(Collectors.toList());
        for (ParkInoutdevice inoutdevice : inoutdevices) {
            ObjectResponse<List<ParkDevice>> deviceListResponse =
                    parkDeviceService.getDeviceListByParkChannelId(sendRequest.getParkId(),
                            inoutdevice.getId().intValue(), 1);
            if (ObjectResponse.isSuccess(deviceListResponse)) {
                List<ParkDevice> parkDevices = deviceListResponse.getData();
                for (ParkDevice parkDevice : parkDevices) {
                    String serialNumber = parkDevice.getSerialNumber();
                    sendDevices += serialNumber + ";";
                }
            } else {
                log.info("通道{}下尚未添加设备", inoutdevice.getInandoutCode());
            }
        }
        if (isSaveCache) {
            cacheHandle.setSendDevices(sendRequest, sendDevices);
        }
        return sendDevices;
    }
    private String getExcludeMasterExitDevices(SendRequest sendRequest) {
        ObjectResponse<List<ParkInoutdevice>> objectResponse =
                parkService.getAllChannel(sendRequest.getParkId());
        ObjectResponse.notError(objectResponse, "未获取到通道信息");
        boolean isSaveCache = sendRequest.getServiceId() != null;
        String sendDevices = null;
        if (isSaveCache) {
            sendDevices = cacheHandle.getSendDevices(sendRequest);
        }
        if (sendDevices != null){
            return sendDevices;
        }
        sendDevices = "";
        List<ParkInoutdevice> inoutdevices = objectResponse.getData();
        inoutdevices = inoutdevices.stream()
                .filter(inout -> !(Integer.valueOf(1).equals(inout.getIsMaster()) && inout.getInandoutType() == 2))
                .collect(Collectors.toList());
        for (ParkInoutdevice inoutdevice : inoutdevices) {
            ObjectResponse<List<ParkDevice>> deviceListResponse =
                    parkDeviceService.getDeviceListByParkChannelId(sendRequest.getParkId(),
                            inoutdevice.getId().intValue(), 1);
            if (ObjectResponse.isSuccess(deviceListResponse)) {
                List<ParkDevice> parkDevices = deviceListResponse.getData();
                for (ParkDevice parkDevice : parkDevices) {
                    String serialNumber = parkDevice.getSerialNumber();
                    sendDevices += serialNumber + ";";
                }
            } else {
                log.info("通道{}下尚未添加设备", inoutdevice.getInandoutCode());
            }
        }
        if (isSaveCache) {
            cacheHandle.setSendDevices(sendRequest, sendDevices);
        }
        return sendDevices;
    }

    protected  <T, R> ObjectResponse<R> send(Message<T> message, String sendDevices, String parkCode) {
        return ObjectResponse.success();
    }
}
