package com.icetech.park.service.charge;

import com.icetech.basics.dao.charge.*;
import com.icetech.basics.domain.entity.charge.*;
import com.icetech.cloudcenter.api.fee.ChargeService;
import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.basics.dao.charge.ParkChargeconfigDao;
import com.icetech.basics.dao.park.ParkRegionDao;
import com.icetech.basics.dao.park.RegionChargeconfigDao;
import com.icetech.fee.dao.storecard.StoreCardDao;
import com.icetech.cloudcenter.domain.charge.detail.Charge24HourDetail;
import com.icetech.cloudcenter.domain.charge.detail.ChargeDayNightDetail;
import com.icetech.cloudcenter.domain.charge.detail.ChargeNaturalDayDetail;
import com.icetech.cloudcenter.domain.charge.detail.ChargeTypeDetail;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.basics.domain.entity.park.ParkChargeconfig;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.basics.domain.entity.RegionChargeconfig;
import com.icetech.fee.domain.entity.storecard.StoreCard;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.constants.PlateColorEnum;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.redis.handle.RedisHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 计费规则配置查询
 *
 * @author fangct
 */
@Slf4j
@Service
public class ChargeServiceImpl implements ChargeService {

    @Autowired
    private ParkChargeconfigDao parkChargeconfigDao;
    @Autowired
    private ChargeNaturaldayDao chargeNaturaldayDao;
    @Autowired
    private ChargeDaynightDao chargeDaynightDao;
    @Autowired
    private Charge24chargeDao charge24chargeDao;
    @Autowired
    private Charge24chargeSetDao charge24chargeSetDao;
    @Autowired
    private ChargeOnceDao chargeOnceDao;
    @Autowired
    private ChargeDyrationDao chargeDyrationDao;
    @Autowired
    private RegionChargeconfigDao regionChargeconfigDao;
    @Resource
    private RedisHandle redisHandle;
    @Autowired
    private StoreCardDao storeCardDao;
    @Autowired
    private OrderCarInfoDao orderCarInfoDao;

    @Autowired
    private ParkRegionDao parkRegionDao;

    /**
     * 电动汽车标识
     */
    private List<String> electricVehicle = Arrays.asList("D", "A", "B", "C", "E");

    /**
     * 油电混合标识
     */
    private List<String> blendVehicle = Arrays.asList("F", "G", "H", "J", "K");


    @Override
    public ObjectResponse<List<RegionChargeconfig>> getRegionConfigs(Long parkId) {
        String cacheKey = RedisKeyConstants.KEY_PREFIX_CHARGE_REGION_PARK + parkId;
        List<RegionChargeconfig> configs = redisHandle.cacheList(cacheKey, RegionChargeconfig.class,
                () -> regionChargeconfigDao.selectByParkId(parkId), RedisKeyConstants.EXPIRE_CHARGE_REGION);
        if (CollectionUtils.isEmpty(configs)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        return ObjectResponse.success(configs);
    }

    @Override
    public ObjectResponse<ParkChargeconfig> getConfigsBycode(Long parkId, String billCode) {
        ObjectResponse<List<ParkChargeconfig>> parkChargeconfigsResp = getConfigs(parkId);
        if (!ObjectResponse.isSuccess(parkChargeconfigsResp)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        for (int i = 0; i < parkChargeconfigsResp.getData().size(); i++) {
            ParkChargeconfig parkChargeconfig = parkChargeconfigsResp.getData().get(i);
            if (parkChargeconfig.getBilltypecode().equals(billCode)) {
                return ObjectResponse.success(parkChargeconfig);
            }
        }
        return ObjectResponse.failed(CodeConstants.ERROR_404);
    }

    @Override
    public ObjectResponse<List<ParkChargeconfig>> getConfigs(Long parkId) {
        String cacheKey = RedisKeyConstants.KEY_PREFIX_CHARGE_PARK + parkId;
        List<ParkChargeconfig> configs = redisHandle.cacheList(cacheKey, ParkChargeconfig.class,
                () -> parkChargeconfigDao.selectByParkId(parkId), RedisKeyConstants.EXPIRE_CHARGE_PARK);
        if (CollectionUtils.isEmpty(configs)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        return ObjectResponse.success(configs);
    }

    @Override
    public ObjectResponse<ParkChargeconfig> selectDefaultBill(Long parkId) {
        ObjectResponse<List<ParkChargeconfig>> configsResp = getConfigs(parkId);
        if (!ObjectResponse.isSuccess(configsResp))
            return ObjectResponse.failed(configsResp.getCode(), configsResp.getMsg());
        for (ParkChargeconfig config : configsResp.getData()) {
            if (config.getDefaultCharge() == 1 && config.getStatus() == 0) {
                return ObjectResponse.success(config);
            }
        }
        //没有默认规则时，取第一个规则
        if (configsResp.getData() != null) {
            ParkChargeconfig parkChargeconfig = configsResp.getData().get(0);
            return ObjectResponse.success(parkChargeconfig);
        }

        return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
    }

    @Override
    public ObjectResponse<ChargeNaturalDayDetail> getNaturalday(String billtypecode) {
        String cacheKey = RedisKeyConstants.KEY_PREFIX_CHARGE_NATURAL_CODE + billtypecode;
        ChargeNaturalDayDetail detail = redisHandle.cacheObject(cacheKey, ChargeNaturalDayDetail.class, () -> {
            ChargeNaturalday naturalDay = chargeNaturaldayDao.selectByCode(billtypecode);
            if (naturalDay == null) {
                return null;
            }
            Map<Integer, ChargeOnce> onceMarkMap = chargeOnceDao.selectByCode(billtypecode).stream().collect(Collectors.toMap(ChargeOnce::getDayornightfeemark, Function.identity(), (old, newer) -> newer));
            Map<Integer, List<ChargeDuration>> durationMarkMap = chargeDyrationDao.selectByCode(billtypecode).stream().collect(Collectors.groupingBy(ChargeDuration::getDayornightfeemark));

            ChargeNaturalDayDetail naturalDayDetail = new ChargeNaturalDayDetail();
            BeanUtils.copyProperties(naturalDay, naturalDayDetail);
            naturalDayDetail.setWorkdayDetail(toChargeTypeDetails(onceMarkMap, durationMarkMap, naturalDay.getBillmethod(), 1));
            if (naturalDay.getIsspecialdaycharge() == 1) {
                naturalDayDetail.setHolidayDetail(toChargeTypeDetails(onceMarkMap, durationMarkMap, naturalDay.getNoworkBillmethod(), 2));
            }
            return naturalDayDetail;
        }, RedisKeyConstants.EXPIRE_CHARGE_NATURAL);
        if (detail == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        return ObjectResponse.success(detail);
    }

    @Override
    public ObjectResponse<ChargeDayNightDetail> getDaynight(String billtypecode) {
        String cacheKey = RedisKeyConstants.KEY_PREFIX_CHARGE_DAY_NIGHT_CODE + billtypecode;
        ChargeDayNightDetail detail = redisHandle.cacheObject(cacheKey, ChargeDayNightDetail.class, () -> {
            ChargeDaynight dayNight = chargeDaynightDao.selectByCode(billtypecode);
            if (dayNight == null) {
                return null;
            }
            Map<Integer, ChargeOnce> onceMarkMap = chargeOnceDao.selectByCode(billtypecode).stream().collect(Collectors.toMap(ChargeOnce::getDayornightfeemark, Function.identity(), (old, newer) -> newer));
            Map<Integer, List<ChargeDuration>> durationMarkMap = chargeDyrationDao.selectByCode(billtypecode).stream().collect(Collectors.groupingBy(ChargeDuration::getDayornightfeemark));

            ChargeDayNightDetail dayNightDetail = new ChargeDayNightDetail();
            BeanUtils.copyProperties(dayNight, dayNightDetail);
            dayNightDetail.setDailyDetail(toChargeTypeDetails(onceMarkMap, durationMarkMap, dayNight.getBillmethodday(), 1));
            dayNightDetail.setNightlyDetail(toChargeTypeDetails(onceMarkMap, durationMarkMap, dayNight.getBillmethodnight(), 2));
            return dayNightDetail;
        }, RedisKeyConstants.EXPIRE_CHARGE_DAY_NIGHT);
        if (detail == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        return ObjectResponse.success(detail);
    }

    @Override
    public ObjectResponse<Charge24HourDetail> get24Hours(String billtypecode) {
        String cacheKey = RedisKeyConstants.KEY_PREFIX_CHARGE_24HOUR_CODE + billtypecode;
        Charge24HourDetail detail = redisHandle.cacheObject(cacheKey, Charge24HourDetail.class, () -> {
            Charge24charge hourCharge = charge24chargeDao.selectByCode(billtypecode);
            if (hourCharge == null) {
                return null;
            }
            Charge24HourDetail hourDetail = new Charge24HourDetail();
            BeanUtils.copyProperties(hourCharge, hourDetail);
            List<Charge24chargeSet> chargeDetails = charge24chargeSetDao.selectByCode(billtypecode);
            hourDetail.setChargeDetails(chargeDetails);
            return hourDetail;
        }, RedisKeyConstants.EXPIRE_CHARGE_24HOUR);
        if (detail == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        return ObjectResponse.success(detail);
    }

    /**
     * 根据车牌颜色获取计费规则
     */
    @Override
    public ParkChargeconfig getParkChargeConfigByPlateColor(Long regionId, String orderNum, String plateNum, Long parkId){
        log.info("获取计费规则，参数：regionId：{}, orderNum：{}, plateNum：{}, parkId：{}", regionId, orderNum, plateNum, parkId);
        if (regionId == null) {
            ParkRegion parkRegion = parkRegionDao.selectOutByParkid(parkId);
            if (parkRegion != null) {
                regionId = parkRegion.getId();
            }
        }
        if (plateNum != null){
            StoreCard storeCard = storeCardDao.selectValidateCard(parkId, plateNum);
            if (storeCard != null && regionId != null && regionId != 0){
                //获取储值卡的计费规则
                RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 6);
                if (Objects.nonNull(regionChargeconfig)){
                    ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                    if (Objects.nonNull(parkChargeconfig)){
                        log.info("使用储值卡车计费规则，orderNum：{}，parkChargeconfig：{}", orderNum, parkChargeconfig);
                        return parkChargeconfig;
                    }
                }
            }
        }
        //获取区域信息
        OrderCarInfo carInfo = orderCarInfoDao.selectByOrderNum(orderNum);
        if (carInfo != null && regionId != null && regionId != 0){
            String plateColor = carInfo.getPlateColor();
            // 绿牌车
            if (PlateColorEnum.GREEN.getDesc().equals(plateColor)){ // 优先取新能源计费
                ParkRegion region = getParkRegion(regionId);
                // 不区分油电混合
                if (region.getIsNewEnergyDiffBill() == 0 ) {
                    RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 1);
                    if (Objects.nonNull(regionChargeconfig)){
                        ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                        if (Objects.nonNull(parkChargeconfig)){
                            log.info("不区分油电混合使用新能源车计费规则, orderNum: {}, parkChargeConfig: {}",
                                    orderNum,
                                    parkChargeconfig);
                            return parkChargeconfig;
                        }
                    }
                }
                // 区分油电混合
                if (region.getIsNewEnergyDiffBill() == 1 && StringUtils.isNotBlank(plateNum)) {
                    // 鲁AD08Q7 获取第三位D
                    String letter = StringUtils.substring(plateNum, 2, 3).toUpperCase();
                    // 纯电动
                    if (electricVehicle.contains(letter)) {
                        RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 7);
                        if (Objects.nonNull(regionChargeconfig)){
                            ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                            if (Objects.nonNull(parkChargeconfig)){
                                log.info("区分油电混合使用新能源(纯电动)车计费规则, orderNum: {}, parkChargeConfig: {}",
                                        orderNum,
                                        parkChargeconfig);
                                return parkChargeconfig;
                            }
                        }
                    }
                    // 油电混合
                    if (blendVehicle.contains(letter)) {
                        RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 1);
                        if (Objects.nonNull(regionChargeconfig)){
                            ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                            if (Objects.nonNull(parkChargeconfig)){
                                log.info("区分油电混使用新能源车(油电混合)计费规则, orderNum: {}, parkChargeConfig: {}",
                                        orderNum,
                                        parkChargeconfig);
                                return parkChargeconfig;
                            }
                        }
                    }
                }
            }
            if (PlateColorEnum.BULE.getDesc().equals(plateColor)){
                //获取蓝牌的计费规则
                RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 3);
                if (Objects.nonNull(regionChargeconfig)){
                    ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                    if (Objects.nonNull(parkChargeconfig)){
                        log.info("使用蓝牌车计费规则，orderNum：{}，parkChargeconfig：{}", orderNum, parkChargeconfig);
                        return parkChargeconfig;
                    }
                }
            }
            if (PlateColorEnum.YELLOW.getDesc().equals(plateColor)){
                //获取黄牌的计费规则
                RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 4);
                if (Objects.nonNull(regionChargeconfig)){
                    ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                    if (Objects.nonNull(parkChargeconfig)){
                        log.info("使用黄牌车计费规则，orderNum：{}，parkChargeconfig：{}", orderNum, parkChargeconfig);
                        return parkChargeconfig;
                    }
                }
            }
            // 黄绿牌
            if (PlateColorEnum.YELLOW_GREEN.getDesc().equals(plateColor)) {
                ParkRegion region = getParkRegion(regionId);
                // 不区分油电混合
                if (region.getIsNewEnergyDiffBill() == 0 ) {
                    RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 1);
                    if (Objects.nonNull(regionChargeconfig)){
                        ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                        if (Objects.nonNull(parkChargeconfig)){
                            log.info("不区分油电混合使用大型新能源汽车计费规则, orderNum: {}, parkChargeConfig: {}",
                                    orderNum,
                                    parkChargeconfig);
                            return parkChargeconfig;
                        }
                    }
                }
                // 区分油电混合
                if (region.getIsNewEnergyDiffBill() == 1 && StringUtils.isNotBlank(plateNum)) {
                    // 鲁AD08Q7D 获取最后一位D
                    String letter = StringUtils.substring(plateNum, 8, 9).toUpperCase();
                    // 纯电动
                    if (electricVehicle.contains(letter)) {
                        RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 7);
                        if (Objects.nonNull(regionChargeconfig)){
                            ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                            if (Objects.nonNull(parkChargeconfig)){
                                log.info("区分油电混合使用大型新能源汽车(纯电动)车计费规则, orderNum: {}, parkChargeConfig: {}",
                                        orderNum,
                                        parkChargeconfig);
                                return parkChargeconfig;
                            }
                        }
                    }
                    // 油电混合
                    if (blendVehicle.contains(letter)) {
                        RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 1);
                        if (Objects.nonNull(regionChargeconfig)){
                            ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId,regionChargeconfig.getBilltypecode());
                            if (Objects.nonNull(parkChargeconfig)){
                                log.info("区分油电混使用大型新能源汽车(油电混合)计费规则, orderNum: {}, parkChargeConfig: {}",
                                        orderNum,
                                        parkChargeconfig);
                                return parkChargeconfig;
                            }
                        }
                    }
                }
            }
            //获取区域的默认规则
            RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 0);
            if (Objects.nonNull(regionChargeconfig)) {
                ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId, regionChargeconfig.getBilltypecode());
                if (Objects.nonNull(parkChargeconfig)) {
                    log.info("使用区域默认计费规则，orderNum：{}，parkChargeconfig：{}", orderNum, parkChargeconfig);
                    return parkChargeconfig;
                }
            }

            //获取车场默认的计费规则
            ObjectResponse<ParkChargeconfig> parkChargeconfig = selectDefaultBill(parkId);
            log.info("使用车场默认计费规则，orderNum：{}，parkChargeconfig：{}", orderNum, parkChargeconfig);
            return parkChargeconfig.getData();
        }else {
            if (regionId != null && regionId != 0) {//没有车牌颜色
                //获取区域的默认规则
                RegionChargeconfig regionChargeconfig = regionChargeconfigDao.getRegionChargeconfig(parkId, regionId, 0);
                if (Objects.nonNull(regionChargeconfig)) {
                    ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCode(parkId, regionChargeconfig.getBilltypecode());
                    if (Objects.nonNull(parkChargeconfig)) {
                        log.info("使用区域默认计费规则，orderNum：{}，parkChargeconfig：{}", orderNum, parkChargeconfig);
                        return parkChargeconfig;
                    }
                }
            }
            //获取车场的默认计费规则
            ObjectResponse<ParkChargeconfig> parkChargeconfig = selectDefaultBill(parkId);
            log.info("使用车场默认计费规则，orderNum：{}，parkChargeconfig：{}", orderNum, parkChargeconfig);
            return parkChargeconfig.getData();
        }
    }

    @Override
    public ParkChargeconfig getParkChargeConfigById(Integer id) {
        return parkChargeconfigDao.selectByPrimaryKey(id);
    }

    @Override
    public ObjectResponse<ParkChargeconfig> getConfigsBycodeWithoutDelFlag(Long parkId, String billCode) {
        ParkChargeconfig parkChargeconfig = parkChargeconfigDao.selectByParkIdAndBillCodeWithoutDelFlag(parkId, billCode);
        return ObjectResponse.returnNotFoundIfNull(parkChargeconfig);
    }

    public ParkRegion getParkRegion(Long regionId) {
        ParkRegion region = parkRegionDao.selectById(regionId);
        if (region == null) {
            throw new ResponseBodyException("1000", "停车场分区获取失败");
        }
        if (region.getIsNewEnergyDiffBill() == null) {
            region.setIsNewEnergyDiffBill(0);
        }
        return region;
    }

    private ChargeTypeDetail toChargeTypeDetails(Map<Integer, ChargeOnce> onceMarkMap, Map<Integer, List<ChargeDuration>> durationMarkMap, int billMethod, int flag) {
        ChargeTypeDetail typeDetail = new ChargeTypeDetail();
        typeDetail.setChargeType(billMethod);
        if (billMethod == 1) {
            typeDetail.setOnce(onceMarkMap.get(flag));
        } else {
            List<ChargeDuration> durations = durationMarkMap.get(flag);
            if(CollectionUtils.isNotEmpty(durations)) {
                typeDetail.setPeriodDurations(new ArrayList<>(durations.size()));
                for (ChargeDuration duration : durations) {
                    if (duration.getIsOverTimeSet() == 1) {
                        typeDetail.setDuration(duration);
                    } else {
                        typeDetail.getPeriodDurations().add(duration);
                    }
                }
            }
        }
        return typeDetail;
    }
}
