package com.icetech.park.service.car.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.icetech.third.anno.DS_SLAVE;
import com.icetech.cloudcenter.api.car.ICarRegisterService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.park.domain.entity.car.CarClean;
import com.icetech.park.domain.entity.car.CarRegister;
import com.icetech.cloudcenter.domain.enumeration.OrderOddStatusEnum;
import com.icetech.park.service.car.CarCleanService;
import com.icetech.park.service.car.CarRegisterService;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 对外暴露接口盘点车辆登记表 服务实现类
 * <p>
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wanggang
 * @version 1.0 @date 2022-08-23
 */
@Service("iCarRegisterServiceImpl")
public class ICarRegisterServiceImpl implements ICarRegisterService {

    @Autowired
    private CarRegisterService carRegisterService;

    @Autowired
    private CarCleanService carCleanService;

    @Autowired
    private OrderService orderService;


    @Override
    @DS_SLAVE
    public Boolean checkPlateNumIsRegister(Long parkId, String plateNum) {
        int count = carRegisterService.count(Wrappers.lambdaQuery(CarRegister.class)
                .eq(CarRegister::getParkId, parkId)
                .eq(CarRegister::getPlateNum, plateNum)
                .eq(CarRegister::getBatchCode, "0"));
        return count > 0;
    }

    @Override
    public ObjectResponse<CarRegister> save(CarRegister carRegister) {
        return carRegisterService.save(carRegister) ? ObjectResponse.success(carRegister) : ObjectResponse.failed(CodeConstantsEnum.ERROR);
    }

    @Override
    @DS_SLAVE
    public com.icetech.common.domain.Page<CarRegister> getRegisterPage(Long parkId, String plateNum, Integer pageNo, Integer pageSize, Integer type, String startTime, String endTime) {
        IPage<CarRegister> page = new Page<>();
        page.setCurrent(pageNo);
        page.setSize(pageSize);
        LambdaQueryWrapper<CarRegister> queryWrapper = Wrappers.lambdaQuery(CarRegister.class);
        queryWrapper.eq(CarRegister::getParkId, parkId);
        queryWrapper.orderByDesc(CarRegister::getId);
        if (StringUtils.isNotEmpty(plateNum)) {
            queryWrapper.eq(CarRegister::getPlateNum, plateNum);
        }
        if (type == 1) {
            queryWrapper.eq(CarRegister::getBatchCode, "0");
        } else {
            queryWrapper.ne(CarRegister::getBatchCode, "0");
        }
        if (StringUtils.isNotEmpty(startTime)) {
            queryWrapper.ge(CarRegister::getCreateTime, startTime);
        }
        if (StringUtils.isNotEmpty(endTime)) {
            queryWrapper.le(CarRegister::getCreateTime, endTime);
        }
        IPage<CarRegister> result = carRegisterService.page(page, queryWrapper);
        return com.icetech.common.domain.Page.instance(Math.toIntExact(result.getPages()), result.getTotal(), result.getRecords());
    }

    @Override
    @DS_SLAVE
    public ObjectResponse<CarRegister> findById(Long id) {
        CarRegister carRegister = carRegisterService.getCarRegisterById(id);
        if (carRegister == null) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        }
        return ObjectResponse.success(carRegister);
    }

    @Override
    @DS_SLAVE
    public ObjectResponse<Integer> getCurrentRegisterCarNum(Long parkId) {
        return ObjectResponse.success(carRegisterService.count(Wrappers.lambdaQuery(CarRegister.class)
                .eq(CarRegister::getParkId, parkId)
                .eq(CarRegister::getBatchCode, "0")));
    }

    @Override
    @DS_SLAVE
    public ObjectResponse<CarRegister> getRegisterOneBy(Long parkId) {
        CarRegister carRegister = carRegisterService.selectLimitOne(Wrappers.lambdaQuery(CarRegister.class)
                .eq(CarRegister::getParkId, parkId)
                .eq(CarRegister::getBatchCode, "0")
                .orderByAsc(CarRegister::getId));
        return carRegister == null ? ObjectResponse.failed(CodeConstantsEnum.ERROR_404) : ObjectResponse.success(carRegister);
    }

    @Override
    public com.icetech.common.domain.Page<CarClean> getCleanPage(Long parkId, String plateNum, Integer pageNo, Integer pageSize, String startTime, String endTime) {
        IPage<CarClean> page = new Page<>();
        page.setCurrent(pageNo);
        page.setSize(pageSize);
        LambdaQueryWrapper<CarClean> queryWrapper = Wrappers.lambdaQuery(CarClean.class);
        queryWrapper.eq(CarClean::getParkId, parkId);
        queryWrapper.eq(CarClean::getType, 1);
        queryWrapper.orderByDesc(CarClean::getId);
        if (StringUtils.isNotEmpty(plateNum)) {
            queryWrapper.eq(CarClean::getPlateNum, plateNum);
        }
        if (StringUtils.isNotEmpty(startTime)) {
            queryWrapper.ge(CarClean::getCreateTime, startTime);
        }
        if (StringUtils.isNotEmpty(endTime)) {
            queryWrapper.le(CarClean::getCreateTime, endTime);
        }
        IPage<CarClean> result = carCleanService.page(page, queryWrapper);
        return com.icetech.common.domain.Page.instance(Math.toIntExact(result.getPages()), result.getTotal(), result.getRecords());
    }

    @Override
    public ObjectResponse<CarClean> getCleanDetail(Long id) {
        CarClean carClean = carCleanService.getCarCleanById(id);
        return carClean == null ? ObjectResponse.failed(CodeConstantsEnum.ERROR_404) : ObjectResponse.success(carClean);
    }

    @Override
    public ObjectResponse<List<CarRegister>> getCarRegisterListBy(Long parkId, String batchCode) {
        List<CarRegister> registers = carRegisterService.list(Wrappers.lambdaQuery(CarRegister.class)
                .eq(CarRegister::getParkId, parkId)
                .eq(CarRegister::getBatchCode, batchCode));
        return CollectionUtils.isEmpty(registers) ? ObjectResponse.failed() : ObjectResponse.success(registers);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ObjectResponse<Boolean> batchUpdateCarRegisters(Long parkId, String batchCode) {
        LambdaUpdateWrapper<CarRegister> wrapper = Wrappers.lambdaUpdate();
        wrapper.set(CarRegister::getBatchCode, batchCode);
        wrapper.eq(CarRegister::getParkId, parkId);
        wrapper.eq(CarRegister::getBatchCode, "0");
        return carRegisterService.update(wrapper) ? ObjectResponse.success(Boolean.TRUE) : ObjectResponse.success(Boolean.FALSE);
    }

    @Override
    public ObjectResponse<Boolean> saveAllCarClean(List<CarClean> carCleanList) {
        return carCleanService.saveBatch(carCleanList) ? ObjectResponse.success(Boolean.TRUE) : ObjectResponse.success(Boolean.FALSE);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ObjectResponse cleanCar(Long parkId, String batchCode, List<CarClean> carCleanList) {
        ObjectResponse<Boolean> updateObject = batchUpdateCarRegisters(parkId, batchCode);
        if (ObjectResponse.isSuccess(updateObject) && !updateObject.getData()) {
            throw new ResponseBodyException("10005", "盘点失败,请稍后重试");
        }
        clear(carCleanList);
        return ObjectResponse.success();
    }

    public void clear(List<CarClean> carCleanList) {
        List<String> orderNums = carCleanList.stream()
                .map(CarClean::getOrderNum)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        orderService.batchExitOrder(orderNums, OrderOddStatusEnum.车辆盘点.getVal(), null);
        CompletableFuture.runAsync(() -> saveAllCarClean(carCleanList));
    }

}
