package com.icetech.park.rpc;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Lists;
import com.icetech.cloudcenter.api.fee.QueryOrderFeeService;
import com.icetech.cloudcenter.api.order.OrderPayService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.paycode.PayCodePrePayService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.request.QueryOrderFeeRequest;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderNotpay;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.cloudcenter.domain.pay.Notification4PayRequest;
import com.icetech.cloudcenter.domain.pay.PayTo;
import com.icetech.cloudcenter.domain.request.DownPayCodeRequest;
import com.icetech.cloudcenter.domain.response.NotPayDetail;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.cloudcenter.domain.vo.NotPayDetailVo;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.impl.DownPncPayCodeServiceImpl;
import com.icetech.order.service.OrderNotpayService;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.constants.PayChannelConstants;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.common.domain.AsyncNotifyInterface;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.MoneyTool;
import com.icetech.paycenter.api.IPayCenterService;
import com.icetech.paycenter.domain.PlatformPayType;
import com.icetech.paycenter.domain.normalpay.request.UnifiedOrderRequest;
import com.icetech.paycenter.domain.normalpay.response.UnifiedOrderResponse;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.math.RoundingMode;
import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import static com.icetech.cloudcenter.domain.constants.RedisConstants.PAY_LOCK_KEY;

/**
 * @Description: 反扫预缴费下单
 * @Author: wangzhiwei
 * @CreateTime: 2023-04-03  15:47
 * @Version: 1.0
 */
@Service
@Slf4j
public class PayCodePrePayServiceImpl implements PayCodePrePayService {
    @Autowired
    private ParkService parkService;
    @Autowired
    private IPayCenterService payCenterService;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private NotifyServiceImpl notifyService;
    @Autowired
    private OrderService orderService;
    @Autowired
    private OrderPayService orderPayService;
    @Autowired
    private OrderNotpayService orderNotpayService;
    @Autowired
    private StringRedisTemplate redisTemplate;
    @Autowired
    private DownPncPayCodeServiceImpl downPncPayCodeService;
    @Autowired
    private ThreadPoolExecutor asyncMethodExecutor;
    @Autowired
    private QueryOrderFeeService queryOrderFeeService;
    @Autowired
    private RedisUtils redisUtils;

    @Override
    public ObjectResponse<Void> prePay(String parkCode, String deviceNo,String channelCode,String payCode) {
        String parks = redisTemplate.opsForValue().get(RedisConstants.PNC_THIRD_PARTY_DOWN_PARKS);
        // 端网云下发支付码(支付码反扫)
        if (parks != null && parks.contains(parkCode)) {
            // 下发
            DownPayCodeRequest downPayCodeRequest = new DownPayCodeRequest();
            downPayCodeRequest.setChannelId(channelCode); // 出入口编号
            downPayCodeRequest.setPayCode(payCode); // 支付码
            downPncPayCodeService.downPayCode(downPayCodeRequest, parkCode);
            return ObjectResponse.success();
        }
        QueryOrderFeeRequest queryOrderFeeRequest = new QueryOrderFeeRequest();
        queryOrderFeeRequest.setParkCode(parkCode);
        queryOrderFeeRequest.setChannelId(channelCode);
        queryOrderFeeRequest.setTopic(AsyncNotifyInterface.getTopic());
        queryOrderFeeRequest.setWithNotPay(true);
        ObjectResponse<QueryOrderFeeResponse> objectResponse = queryOrderFeeService.queryOrderFee(queryOrderFeeRequest);
        //获取回调的查费结果
        QueryOrderFeeResponse channelFee;

        if(CodeConstants.ERROR_12002.equals(objectResponse.getCode())) {
            String messageId = objectResponse.getMsg();
            objectResponse = AsyncNotifyInterface.wait(messageId, 6000L, () -> {
                ObjectResponse<QueryOrderFeeResponse> response = redisUtils.get(AsyncNotifyInterface.getMessageKey(messageId),
                        new TypeReference<ObjectResponse<QueryOrderFeeResponse>>() {});
                log.info("[端云费用返回] response {}",response);
                if (response != null) {
                    return response;
                }
                return ObjectResponse.failed(CodeConstantsEnum.ERROR_405.getCode(), "未查询到待缴费");
            });
            channelFee = objectResponse.getData();
        }else {
            channelFee = objectResponse.getData();
        }

        if (channelFee == null){
            log.warn("[立柱付款码接口] 通道费用已经清空, 设备信息[{}]", channelCode);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_405.getCode(), "未查询到待缴费");
        }
        //获取车场支持的类型
        ObjectResponse<String> response= parkService.selectGroupPayType(parkCode);
        if (!ObjectResponse.isSuccess(response)){
            log.warn("[立柱付款码接口] 该车场不支持电子缴费, parkCode {} 设备信息[{}]", parkCode,channelCode);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_405.getCode(), "未查询到待缴费");
        }

        String orderNum = channelFee.getOrderNum();
        if (StringUtils.isBlank(orderNum)) {
            log.warn("[立柱付款码接口] 通道费用已经清空, 设备信息[{}]", channelCode);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_405.getCode(), "未查询到待缴费");
        }
        ObjectResponse<OrderInfo> orderInfoObjectResponse = orderService.findByOrderNum(orderNum);
        if (!ObjectResponse.isSuccess(orderInfoObjectResponse)){
            log.warn("[立柱付款码接口] 无效的订单号，订单号[{}]", orderNum);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_405.getCode(), "订单号不存在");
        }

        OrderInfo orderInfo = orderInfoObjectResponse.getData();
        try {
            //判断支付锁，是否可以扣费
            lockJudge(parkCode, orderNum, orderInfo);
        } catch (ResponseBodyException e) {
            return ObjectResponse.failed(e.getErrCode(), e.getMessage());
        }
        UnifiedOrderRequest unifiedOrderRequest = new UnifiedOrderRequest();
        int payWay = UnifiedOrderRequest.getWxAliPayCode(payCode);
        unifiedOrderRequest.setSelectTradeType(PlatformPayType.getTradeTypeScan(response.getData(),payWay));
        String tradeNo = CodeTools.GenerateTradeNo();
        PayTo payTo = new PayTo();
        payTo.setParkCode(parkCode);
        payTo.setTradeNo(tradeNo);
        payTo.setOrderNum(orderNum);
        payTo.setChannelId(channelCode);
        payTo.setPlateNum(orderInfo.getPlateNum());
        payTo.setEnterTime(orderInfo.getEnterTime());
        payTo.setIsOffline(false);
        List<NotPayDetail> notPayDetails = addOrderPay(channelFee, orderInfo, PlatformPayType.getPayAisle(response.getData(), payWay), tradeNo, channelCode, deviceNo, payWay);
        if (CollectionUtil.isNotEmpty(notPayDetails)){
            payTo.setNotPayDetails(BeanUtil.copyToList(notPayDetails, NotPayDetailVo.class));
        }
        unifiedOrderRequest.setExtraInfo(JSONUtil.toJsonStr(payTo));
        unifiedOrderRequest.setTradeNo(tradeNo);
        unifiedOrderRequest.setParkCode(parkCode);
        unifiedOrderRequest.setPayCode(payCode);

        String parkName = channelFee.getParkName();
        if (StringUtils.isBlank(parkName)) {
            ObjectResponse<Park> parkObjectResponse = parkService.findByParkCode(parkCode);
            if (ObjectResponse.isSuccess(parkObjectResponse)) {
                parkName = parkObjectResponse.getData().getParkName();
            }
        }
        String productInfo = unifiedOrderRequest.genProductInfo(parkName, orderInfo.getPlateNum(), "停车费");
        unifiedOrderRequest.setProductInfo(productInfo);
        unifiedOrderRequest.setPrice(MoneyTool.fromYuanToFen(channelFee.getSumPrice()));
        ObjectResponse<UnifiedOrderResponse> unifiedOrderResponseObjectResponse = payCenterService.doUnifiedOrder(unifiedOrderRequest);
        if (ObjectResponse.isSuccess(unifiedOrderResponseObjectResponse)){
            asyncMethodExecutor.execute(ThreadUtils.wrapTrace(() -> {
                UnifiedOrderResponse unifiedOrderResponse = unifiedOrderResponseObjectResponse.getData();
                String outTradeNo = unifiedOrderResponse != null ? unifiedOrderResponse.getOutTradeNo() : null;
                log.info("[反扫异步通知开始] tradeNo {}",tradeNo);
                Notification4PayRequest request = new Notification4PayRequest();
                request.setTradeNo(tradeNo);
                request.setOutTradeNo(StringUtils.isNotBlank(outTradeNo) ? outTradeNo : unifiedOrderRequest.getParkCode()+unifiedOrderRequest.getTradeNo());
                request.setPrice(unifiedOrderRequest.getPrice());
                request.setPayTime(DateUtil.currentSeconds());
                request.setExtraInfo(unifiedOrderRequest.getExtraInfo());
                ObjectResponse<Void> voidObjectResponse = notifyService.payNotify(request);
                log.info("[反扫异步通知结束] voidObjectResponse {}",voidObjectResponse);
            }));
            return ObjectResponse.success();
        }
        redisTemplate.delete(PAY_LOCK_KEY + parkCode + orderNum);
        return ObjectResponse.failed();
    }

    @Override
    public ObjectResponse pncPrePay(String parkCode, String channelCode,String orderId,String payCode,String price) {
        //获取车场支持的类型
        ObjectResponse<Park> parkObjectResponse = parkService.findByParkCode(parkCode);
        ObjectResponse<String> response= parkService.selectGroupPayType(parkCode);
        OrderInfo orderInfo = new OrderInfo();
        orderInfo.setLocalOrderNum(orderId);
        orderInfo.setParkId(parkObjectResponse.getData().getId());
        ObjectResponse<OrderInfo> orderInfoObjectResponse = orderService.findByOrderInfo(orderInfo);

        orderInfo = orderInfoObjectResponse.getData();
        String orderNum = orderInfo.getOrderNum();
        try {
            lockJudge(parkCode, orderNum, orderInfo);
        } catch (ResponseBodyException e) {
            return ObjectResponse.failed(e.getErrCode(), e.getMessage());
        }

        UnifiedOrderRequest unifiedOrderRequest = new UnifiedOrderRequest();
        int payWay = UnifiedOrderRequest.getWxAliPayCode(payCode);
        unifiedOrderRequest.setSelectTradeType(PlatformPayType.getTradeTypeScan(response.getData(),payWay));
        String tradeNo = CodeTools.GenerateTradeNo();
        PayTo payTo = new PayTo();
        payTo.setParkCode(parkCode);
        payTo.setTradeNo(tradeNo);
        payTo.setOrderNum(orderInfo.getOrderNum());
        payTo.setChannelId(channelCode);
        payTo.setPlateNum(orderInfo.getPlateNum());
        payTo.setEnterTime(orderInfo.getEnterTime());
        payTo.setIsOffline(false);
        unifiedOrderRequest.setExtraInfo(JSONUtil.toJsonStr(payTo));
        unifiedOrderRequest.setTradeNo(tradeNo);
        unifiedOrderRequest.setParkCode(parkCode);
        unifiedOrderRequest.setPayCode(payCode);
        String productInfo = unifiedOrderRequest.genProductInfo(parkObjectResponse.getData().getParkName(), orderInfo.getPlateNum(), "停车费");
        unifiedOrderRequest.setProductInfo(productInfo);
        unifiedOrderRequest.setPrice(price);
        ObjectResponse<UnifiedOrderResponse> unifiedOrderResponseObjectResponse = payCenterService.doUnifiedOrder(unifiedOrderRequest);
        if (ObjectResponse.isSuccess(unifiedOrderResponseObjectResponse)){
            return ObjectResponse.success(unifiedOrderResponseObjectResponse.getData());
        }
        return ObjectResponse.failed(unifiedOrderResponseObjectResponse.getCode(), unifiedOrderResponseObjectResponse.getMsg(), unifiedOrderResponseObjectResponse.getData());
    }

    private void lockJudge(String parkCode, String orderNum, OrderInfo orderInfo) {
        String etcLock = redisTemplate.opsForValue().get(RedisConstants.ETC_PAY_LOCK_KEY + orderNum);
        if (StringUtils.isNotBlank(etcLock)) {
            log.info("[立柱付款码接口] 订单号[{}]正在进行ETC扣款, 请稍后再试，", orderNum);
            throw new ResponseBodyException(CodeConstants.ERROR_402, "车牌号[" + orderInfo.getPlateNum() + "]正在进行其他ETC扣款");
        } else {
            Boolean payLock = redisTemplate.opsForValue().setIfAbsent(PAY_LOCK_KEY + parkCode + orderNum, "etc", 20L, TimeUnit.SECONDS);
            if (!Boolean.TRUE.equals(payLock)) {
                log.info("[立柱付款码接口] 订单号[{}]正在进行扫码支付, 请稍后再试", orderNum);
                throw new ResponseBodyException(CodeConstants.ERROR_402, "车牌号[" + orderInfo.getPlateNum() + "]正在进行扫码支付");
            }
        }
    }

    private List<NotPayDetail> addOrderPay(QueryOrderFeeResponse channelFee, OrderInfo orderInfo, Integer payAisle, String tradeNo, String channelCode, String deviceNo, Integer payWay){
        OrderPay orderPay = new OrderPay();
        orderPay.setOrderNum(orderInfo.getOrderNum());
        orderPay.setPayStatus(PayStatusConstants.PAYING);
        orderPay.setParkId(orderInfo.getParkId());

        float discountPrice = Float.parseFloat(channelFee.getDiscountPrice());
        float unpayPrice = Float.parseFloat(channelFee.getUnpayPrice());
        orderPay.setDiscountPrice(channelFee.getDiscountPrice());
        orderPay.setPaidPrice(channelFee.getUnpayPrice());
        orderPay.setPayChannel(PayChannelConstants.TALKBACK);
        orderPay.setPayTerminal(deviceNo);
        orderPay.setPayTime(channelFee.getQueryTime());
        orderPay.setOrderTime(channelFee.getQueryTime());
        orderPay.setPayWay(payWay);
        float totalPrice = discountPrice + unpayPrice;
        orderPay.setTotalPrice(String.valueOf(totalPrice));
        orderPay.setTradeNo(tradeNo);
        orderPay.setIsSync(1);
        orderPay.setPayAisle(payAisle);
        orderPay.setChannelId(channelCode);
        orderPayService.addOrderPay(orderPay);
        if (CollectionUtil.isNotEmpty(channelFee.getNotPayDetails())){
            channelFee.getNotPayDetails().forEach(notPayDetail -> {
                OrderNotpay orderNotpay = orderNotpayService.getOrderNotpayByOrderNum(notPayDetail.getOrderNum());
                OrderPay notPay = new OrderPay();
                notPay.setParkId(orderNotpay.getParkId());
                notPay.setPayStatus(PayStatusConstants.PAYING);
                notPay.setOrderNum(orderNotpay.getOrderNum());
                notPay.setChannelId(channelCode);
                notPay.setPayWay(payWay);
                notPay.setTradeNo(CodeTools.GenerateTradeNo());
                notPay.setLastPayTime(orderNotpay.getLastPayTime());
                notPay.setTotalPrice(orderNotpay.getTotalPrice().setScale(2, RoundingMode.HALF_UP).toString());
                notPay.setDiscountPrice(orderNotpay.getDiscountPrice().setScale(2, RoundingMode.HALF_UP).toString());
                notPay.setPaidPrice(orderNotpay.getTotalPrice().subtract(orderNotpay.getDiscountPrice()).setScale(2, RoundingMode.HALF_UP).toString());
                notPay.setOrderTime(DateUtil.currentSeconds());
                notPay.setPayTime(DateUtil.currentSeconds());
                notPay.setPayChannel(PayChannelConstants.TALKBACK);
                notPay.setPayType(1);
                notPay.setPayAisle(payAisle);
                notPay.setPayTerminal(deviceNo);
                orderPayService.addOrderPay(notPay);
                notPayDetail.setTradeNo(notPay.getTradeNo());
            });
            return channelFee.getNotPayDetails();
        }
        return Lists.newArrayList();
    }
}
