package com.icetech.park.rpc;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.TypeReference;
import com.icetech.cloudcenter.api.INotifyService;
import com.icetech.cloudcenter.api.ManageService;
import com.icetech.cloudcenter.api.catched.NoplateRecordService;
import com.icetech.cloudcenter.api.discount.MerchantUserService;
import com.icetech.cloudcenter.api.month.MonthCarService;
import com.icetech.cloudcenter.api.order.OrderPayService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.parkvip.IParkVipService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.pay.*;
import com.icetech.cloudcenter.domain.request.MonthPayRequest;
import com.icetech.cloudcenter.domain.request.NoplateExitRequest;
import com.icetech.cloudcenter.domain.request.OfflineExitRequest;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.common.domain.AsyncNotifyInterface;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.JsonUtils;
import com.icetech.order.domain.entity.OrderNotpay;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.order.service.OrderNotpayService;
import com.icetech.park.domain.entity.catched.NoplateRecord;
import com.icetech.park.service.down.NoplateExitServiceImpl;
import com.icetech.park.service.impl.DownSendServiceImpl;
import com.icetech.redis.lock.RedissonDistributedLock;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author kate
 */
@Slf4j
@Service
public class NotifyServiceImpl implements INotifyService {
    @Autowired
    private OrderPayService orderPayService;
    @Autowired
    private MonthCarService monthCarService;
    @Autowired
    private StringRedisTemplate redisTemplate;
    @Autowired
    private ManageService manageService;
    @Autowired
    private NoplateExitServiceImpl noplateExitService;
    @Autowired
    private NoplateRecordService noplateRecordService;
    @Autowired
    private IParkVipService parkVipService;
    @Autowired
    private ThreadPoolExecutor asyncMethodExecutor;
    @Autowired
    private MerchantUserService merchantUserService;
    @Autowired
    private DownSendServiceImpl downSendService;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private RedissonDistributedLock redissonDistributedLock;
    @Autowired
    private ParkService parkService;

    @Autowired
    private OrderNotpayService orderNotpayService;

    @Override
    public ObjectResponse<Void> payNotify(Notification4PayRequest bizContent) {
        try {
            log.info("[支付异步通知] 参数[{}]", bizContent);
            if (bizContent == null){
                return ObjectResponse.failed(CodeConstants.ERROR_400);
            }
            if (StringUtils.isBlank(bizContent.getTradeNo())) {
                return ObjectResponse.failed(CodeConstants.ERROR_400,"交易流水号不能为空");
            }
            return redissonDistributedLock.locked(() -> {
                //解析不同的业务类型进行不同的处理
                PayTo payTo = JsonUtils.parseObject(bizContent.getExtraInfo(), PayTo.class);
                if (payTo == null){
                    return ObjectResponse.failed(CodeConstants.ERROR_400, "缺少额外数据包参数");
                }
                if (payTo.getBizType() == BizType.PARK_PAY.getBizType()) {
                    payTo.setOutTradeNo(bizContent.getOutTradeNo());
                    payTo.setBankOrderId(bizContent.getBankOrderId());
                    if (payTo.getExType() == null || payTo.getExType() == 2) {
                        ObjectResponse<Void> orderPayResponse = currentParkOrderHandle(bizContent, payTo);
                        //欠费追缴
                        if (CollectionUtils.isNotEmpty(payTo.getNotPayDetails())){
                            orderPayService.finishPayNotPay(payTo);
                        }
                        return orderPayResponse;
                    } else {
                        //欠费追缴
                        if (CollectionUtils.isNotEmpty(payTo.getNotPayDetails())){
                            orderPayService.finishPayNotPay(payTo);
                        }
                        return ObjectResponse.success();
                    }
                } else if (payTo.getBizType() == BizType.MONTH_PAY.getBizType()){
                    MonthPayInfo monthPayInfo = JsonUtils.parseObject(bizContent.getExtraInfo(), MonthPayInfo.class);
                    MonthPayRequest monthPayRequest = new MonthPayRequest();
                    BeanUtil.copyProperties(monthPayInfo,monthPayRequest);
                    monthPayRequest.setTradeNo(bizContent.getTradeNo());
                    monthPayRequest.setOutTradeNo(bizContent.getOutTradeNo());
                    monthPayRequest.setPayTime(bizContent.getPayTime());
                    ObjectResponse<Void> response = monthCarService.finishMonthPay(monthPayRequest);
                    log.info("[月卡续费异步通知] 参数[{}], 返回[{}]", monthPayRequest, JsonUtils.toString(response));
                    return ObjectResponse.success();
                }else if (payTo.getBizType() == BizType.PARK_VIP_PAY.getBizType()) {
                    OpenParkVipDto openParkVipDto = JsonUtils.parseObject(bizContent.getExtraInfo(), OpenParkVipDto.class);
                    ObjectResponse<Void> voidObjectResponse = parkVipService.openVip(openParkVipDto);
                    log.info("[车场开通VIP]请求[{}]|返回[{}]", openParkVipDto, voidObjectResponse);
                    return ObjectResponse.success();
                }else if (payTo.getBizType() == BizType.DISCOUNT_PAY.getBizType()){
                    MerchantRechargeDto merchantRechargeDto = JsonUtils.parseObject(bizContent.getExtraInfo(), MerchantRechargeDto.class);
                    ObjectResponse<Void> voidObjectResponse = merchantUserService.finishRecharge(merchantRechargeDto);
                    log.info("[车场商户充值]请求[{}]|返回[{}]", merchantRechargeDto, voidObjectResponse);
                    return ObjectResponse.success();
                }else if (payTo.getBizType() == BizType.MONTH_OPEN_PAY.getBizType()){
                    MonthOpenCardDto monthOpenCardDto = JsonUtils.parseObject(bizContent.getExtraInfo(), MonthOpenCardDto.class);
                    monthOpenCardDto.setOutTradeNo(bizContent.getOutTradeNo());
                    monthOpenCardDto.setPayTime(bizContent.getPayTime());
                    ObjectResponse<Object> objectObjectResponse = monthCarService.openCard(monthOpenCardDto, Boolean.FALSE);
                    log.info("[月卡开卡]请求[{}]|返回[{}]", monthOpenCardDto, objectObjectResponse);
                    return ObjectResponse.success();
                }
                else if (payTo.getBizType() == BizType.MINI_OPEN_PAY.getBizType()){
                    MiniOrdersDTO dto = JsonUtils.parseObject(bizContent.getExtraInfo(), MiniOrdersDTO.class);
                    dto.setOutTradeNo(bizContent.getOutTradeNo());
                    dto.setPayTime(bizContent.getPayTime());
                    ObjectResponse<Void> objectObjectResponse = currentParkOrderHandle(dto, bizContent.getTradeNo(), bizContent.getBankOrderId());
                    log.info("[小程序批量支付]请求[{}]|返回[{}]", dto, objectObjectResponse);
                    return ObjectResponse.success();
                }
                return ObjectResponse.failed(CodeConstants.ERROR_402,"无法处理的业务类型[" + payTo.getBizType() + "]");
            }, "LOCK:PAY:" + bizContent.getTradeNo());
        }catch (Exception e){
            log.error("[异步通知异常] tradeNo {}",bizContent.getTradeNo(),e);
            return ObjectResponse.failed();
        }
    }
    private ObjectResponse<Void> currentParkOrderHandle(MiniOrdersDTO dto, String tradeNo,String bankOrderId) {
        if (StringUtils.isNotEmpty(dto.getOrderNum())) {
            OrderPay orderPay = new OrderPay();
            orderPay.setTradeNo(tradeNo);
            orderPay.setOrderNum(dto.getOrderNum());
            ObjectResponse<OrderPay> orderPayResponse = orderPayService.findOne(orderPay);
            if (ObjectResponse.isSuccess(orderPayResponse)) {
                OrderPay data = orderPayResponse.getData();
                if (data.getPayStatus() == PayStatusConstants.PAYING) {
                    data.setPayTime(dto.getPayTime());
                    data.setBankOrderId(bankOrderId);
                    data.setPayStatus(PayStatusConstants.PAID);
                    data.setThirdTradeNo(dto.getOutTradeNo());
                    orderPayService.updateOrderPayWithPush(data);
                }
            }
        }
        if (CollectionUtils.isNotEmpty(dto.getOrderNums())) {
            orderPayService.finishPayNotPay(dto,bankOrderId);
        }
        return ObjectResponse.success();
    }
    private ObjectResponse<Void> currentParkOrderHandle(Notification4PayRequest bizContent, PayTo payTo) {
        //停车缴费异步通知处理
        OrderPay orderPay = new OrderPay();
        orderPay.setTradeNo(bizContent.getTradeNo());
        ObjectResponse<OrderPay> orderPayResponse = orderPayService.findOne(orderPay);
        if (!ObjectResponse.isSuccess(orderPayResponse)) {
            log.info("[停车费异步通知] 未查找到交易流水, tradeNo[{}], 返回[{}]", bizContent.getTradeNo(), orderPayResponse);
            return ObjectResponse.failed(orderPayResponse.getCode(), orderPayResponse.getMsg());
        }
        OrderPay data = orderPayResponse.getData();
        if (data.getPayStatus() == PayStatusConstants.PAID) {
            log.info("[停车费异步通知] 重复通知不处理, 直接返回, tradeNo[{}]", bizContent.getTradeNo());
            return ObjectResponse.success();
        }
        //清除支付锁
        redisTemplate.delete(Arrays.asList(
                RedisConstants.PAY_LOCK_KEY + payTo.getParkCode() + data.getOrderNum(),
                RedisConstants.PAY_USER_KEY + payTo.getUnionId()));
        //修改订单交易记录
        data.setPayTime(bizContent.getPayTime());
        data.setBankOrderId(bizContent.getBankOrderId());
        data.setThirdTradeNo(bizContent.getOutTradeNo());
        //修改支付和优惠状态
        ObjectResponse<Integer> objectResponse = orderPayService.finishPayAndDiscount(data, payTo.getParkCode(), payTo.getChannelId());
        log.info("[停车费异步通知] 修改订单支付状态为已支付,参数[{}],响应[{}]",data, JsonUtils.toString(objectResponse));
        ObjectResponse.notError(objectResponse, "修改支付和优惠状态失败");
        //为保证本地系统在开闸时验证有无待支付费用，增加了出口支付流水的相关信息
        asyncMethodExecutor.execute(ThreadUtils.wrapTrace(() -> openAndUpdate(bizContent, data)));
        return ObjectResponse.success();
    }

    /**
     * 抬杆放行并修改无牌车记录
     * @param notification4PayRequest 通知参数
     * @param data 支付记录
     */
    private void openAndUpdate(Notification4PayRequest notification4PayRequest, OrderPay data) {
        PayTo payTo = JsonUtils.parseObject(notification4PayRequest.getExtraInfo(), PayTo.class);
        if (payTo == null || StringUtils.isBlank(payTo.getChannelId())) {
            return;
        }
        //断电应急处理
        if (Boolean.TRUE.equals(payTo.getIsOffline())) {
            OfflineExitRequest offlineExitRequest = new OfflineExitRequest();
            offlineExitRequest.setOrderNum(payTo.getOrderNum());
            offlineExitRequest.setPlateNum(payTo.getPlateNum());
            offlineExitRequest.setParkCode(payTo.getParkCode());
            offlineExitRequest.setAisleCode(payTo.getChannelId());
            ObjectResponse<Void> mapObjectResponse = manageService.offLineExit(offlineExitRequest);
            log.info("[登记出场] 参数[{}], 响应[{}]", offlineExitRequest, mapObjectResponse);
        }
        //2.放行
        NoplateExitRequest noplateExitRequest = new NoplateExitRequest();
        noplateExitRequest.setParkCode(payTo.getParkCode());
        noplateExitRequest.setChannelId(payTo.getChannelId());
        noplateExitRequest.setPlateNum(payTo.getPlateNum());
        noplateExitRequest.setExitTime(System.currentTimeMillis() / 1000);
        noplateExitRequest.setOrderNum(payTo.getOrderNum());
        noplateExitRequest.setTopic(AsyncNotifyInterface.getTopic());

        //下发开闸
        SendRequest sendRequest = new SendRequest();
        sendRequest.setParkId(data.getParkId());
        sendRequest.setServiceId(data.getId());
        sendRequest.setServiceType(DownServiceEnum.预缴费.getServiceType());
        sendRequest.setTopic(AsyncNotifyInterface.getTopic());
        ObjectResponse<?> send = downSendService.send(sendRequest);
        //端网云等待支付记录下发响应后，再进行开闸
        if(CodeConstants.ERROR_12002.equals(send.getCode())) {
            String messageId = send.getMsg();
            send = AsyncNotifyInterface.wait(messageId, 4000L,
                    () -> redisUtils.get(AsyncNotifyInterface.getMessageKey(messageId), new TypeReference<ObjectResponse<Void>>() {
                    }));
        }
        log.info("[出口支付结果通知] 下发预缴费,参数[{}],返回[{}]", sendRequest, send);
        ObjectResponse<Void> objectResponse = noplateExitService.noplateExit(noplateExitRequest);
        log.info("[出口支付结果通知] 下发开闸,参数[{}],响应[{}]", JsonUtils.toString(noplateExitRequest),
                JsonUtils.toString(objectResponse));
        if(CodeConstants.ERROR_12002.equals(objectResponse.getCode())) {
            String messageId = objectResponse.getMsg();
            objectResponse = AsyncNotifyInterface.wait(messageId, 4000L,
                    () -> redisUtils.get(AsyncNotifyInterface.getMessageKey(messageId), new TypeReference<ObjectResponse<Void>>() {
                    }));
        }
        //判断结果
        if (ObjectResponse.isSuccess(objectResponse)
                && NoplateRecordType.无牌车.getType().equals(payTo.getType())) {
            NoplateRecord noplateRecord = new NoplateRecord();
            noplateRecord.setUnionId(payTo.getUnionId());
            noplateRecord.setPlateNum(payTo.getPlateNum());
            noplateRecord.setParkCode(payTo.getParkCode());
            noplateRecord.setStatus(NoplateRecordStatus.已入场.getStatus());
            noplateRecord.setType(NoplateRecordType.无牌车.getType());
            NoplateRecord noplateRecordUpdate = new NoplateRecord();
            noplateRecordUpdate.setStatus(NoplateRecordStatus.已出场.getStatus());
            noplateRecordUpdate.setExitTime(new Date());
            noplateRecordUpdate.setExitChannelId(payTo.getChannelId());
            noplateRecordService.update(noplateRecordUpdate);
        }
    }
}
