package com.icetech.park.rpc;

import cn.hutool.core.date.DateUtil;
import com.google.common.collect.Lists;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.cloudcenter.api.NotPayService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.third.SendInfoService;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.notpay.NotPayDeleteDTO;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.Page;
import com.icetech.order.domain.dto.NotPayRecordQueryDTO;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderNotpay;
import com.icetech.order.service.OrderNotpayService;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.domain.vo.ParkRecoveryVo;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.down.pnc.impl.DelNotPayRecordServiceImpl;
import com.icetech.park.service.park.ParkRecoveryService;
import com.icetech.third.anno.DS_SLAVE;
import com.icetech.third.domain.entity.third.SendInfo;
import com.icetech.third.utils.DateRangeUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Description NotPayServiceImpl
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2023/3/23 13:39
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class NotPayServiceImpl implements NotPayService {

    private final OrderNotpayService orderNotpayService;

    private final ParkService parkService;

    private final CacheHandle cacheHandle;

    private final DelNotPayRecordServiceImpl delNotPayRecordService;

    private final SendInfoService sendInfoService;

    private final OrderService orderService;

    private final ParkRecoveryService parkRecoveryService;

    private final ParkDeviceService parkDeviceService;

    @Override
    public ObjectResponse<List<OrderNotpay>> queryNotPayFee(Long parkId, String plateNum, int exType) {
//        List<ParkRecovery> recoverys = parkRecoveryService.getOpenParkRecoveryByParkIds(Collections.singletonList(parkId));
//        if (CollectionUtils.isNotEmpty(recoverys)) {
        ParkRecoveryVo recovery = parkRecoveryService.getParkRecoveryByParkId(parkId);
        if (recovery != null && recovery.isEnableRecovery()) {
            List<Integer> hasMorCameraChannelIds = new ArrayList<>();
            List<OrderNotpay> orderNotpays = orderNotpayService.queryNotPayFee(parkId, plateNum, exType);
            if (CollectionUtils.isNotEmpty(orderNotpays)) {
                List<Long> channelIds = orderNotpays.stream().filter(op -> op.getEvidenceVideo() == null)
                        .map(OrderNotpay::getChannelId).distinct().collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(channelIds)) {
                    ObjectResponse<List<ParkDevice>> devices = parkDeviceService.getDeviceListByChannelIds(channelIds, 9);
                    if (ObjectResponse.isSuccess(devices) && CollectionUtils.isNotEmpty(devices.getData())) {
                        hasMorCameraChannelIds = devices.getData().stream().map(ParkDevice::getChannelId).collect(Collectors.toList());
                    }
                }

                for (OrderNotpay orderNotpay : orderNotpays) {
                    if (orderNotpay.getEvidenceVideo() != null
                            || (orderNotpay.getChannelId() != null && hasMorCameraChannelIds.contains(orderNotpay.getChannelId().intValue()))) {
                        orderNotpay.setHasMorCamera(1);
                    } else {
                        orderNotpay.setHasMorCamera(0);
                    }
                }
            }
            return ObjectResponse.success(orderNotpays);
        } else {
            return ObjectResponse.success();
        }
    }

    @Override
    public ObjectResponse<OrderNotpay> getOrderNotpayById(Long id) {
        return ObjectResponse.success(orderNotpayService.getOrderNotpayById(id));
    }

    @Override
    public ObjectResponse<Void> delNotPayOrder(NotPayDeleteDTO notPayDeleteDTO, String operAccount) {
        ObjectResponse<Park> byParkCode = parkService.findByParkCode(notPayDeleteDTO.getParkCode());
        if (!ObjectResponse.isSuccess(byParkCode)) {
            return ObjectResponse.failed(byParkCode.getCode(), byParkCode.getMsg());
        }
        List<OrderNotpay> orderNotpays = orderNotpayService.listByIds(notPayDeleteDTO.getIds());
        if (CollectionUtils.isEmpty(orderNotpays)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        Park park = byParkCode.getData();
        Long parkId = park.getId();
        QueryOrderFeeResponse channelFee = null;
        if (StringUtils.isNotEmpty(notPayDeleteDTO.getAisleCode())) {
            channelFee = cacheHandle.getChannelFee(notPayDeleteDTO.getParkCode(), notPayDeleteDTO.getAisleCode());
        }
        // 是不是端网云类型车场
        Boolean parkTypeFlag = Boolean.FALSE;
        ObjectResponse<ParkConfig> configObj = parkService.getParkConfig(parkId);
        if (ObjectResponse.isSuccess(configObj)) {
            ParkConfig parkConfig = configObj.getData();
            if (Objects.nonNull(parkConfig) && parkConfig.getDataCollection() == 2) {
                parkTypeFlag = Boolean.TRUE;
            }
        }
        Map<Integer, List<OrderNotpay>> listMap = orderNotpays.stream().collect(Collectors.groupingBy(OrderNotpay::getId));
        QueryOrderFeeResponse finalChannelFee = channelFee;
        notPayDeleteDTO.getIds().forEach(id-> {
            OrderNotpay orderNotpay = new OrderNotpay();
            orderNotpay.setId(id);
            orderNotpay.setParkId(parkId);
            orderNotpay.setStatus(3);
            orderNotpay.setOperReason(notPayDeleteDTO.getReason());
            orderNotpay.setOperTime(new Date());
            orderNotpay.setOperAccount(operAccount);
            orderNotpay.setOrderNum(listMap.get(id).get(0).getOrderNum());
            orderNotpayService.dealNotpayOrder(orderNotpay);
            if(finalChannelFee != null) {
                OrderNotpay orderNotpayById = orderNotpayService.getOrderNotpayById(id.longValue());
                if (Objects.nonNull(orderNotpayById) ) {
                    finalChannelFee.removeNotPayItem(orderNotpayById.getOrderNum());
                }
            }
        });
        // 下发端网云本地
        if (parkTypeFlag) {
            if (CollectionUtils.isNotEmpty(orderNotpays)) {
                List<String> orderNums = orderNotpays.stream()
                        .map(OrderNotpay::getOrderNum)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList());
                List<OrderInfo> orderInfos = orderService.selectListByOrderNums(orderNums);
                if (CollectionUtils.isEmpty(orderInfos) || orderInfos.size() != orderNums.size()) {
                    Date startDate = DateUtil.offsetDay(DateUtil.date(), -90);
                    orderInfos = orderService.findWithHistoryByOrderNums(orderNums, DateRangeUtils.getYearQuarterRangeTableName(startDate));
                }
                if (CollectionUtils.isNotEmpty(orderInfos)) {
                    Map<String, String> orderMap = orderInfos.stream()
                            .collect(Collectors.toMap(OrderInfo::getOrderNum, OrderInfo::getLocalOrderNum, (key1, key2) -> key1));
                    String orderIds = orderNums.stream()
                            .map(a -> {
                                String localOrderNum = orderMap.get(a);
                                if (StringUtils.isEmpty(localOrderNum)) {
                                    return a;
                                }
                                return localOrderNum;
                            }).collect(Collectors.joining(","));
                    // 下发失败写入sendInfo
                    ObjectResponse response = delNotPayRecordService.sendDelMessage(orderIds, parkId);
                    if (!ObjectResponse.isSuccess(response)) {
                        SendInfo sendInfo = new SendInfo();
                        sendInfo.setParkId(parkId);
                        sendInfo.setLevel(2);
                        sendInfo.setServiceType(DownServiceEnum.欠费记录删除.getServiceType());
                        sendInfo.setServiceId(parkId);
                        sendInfo.setInclude(orderIds);
                        sendInfo.setSendNum(0);
                        sendInfo.setNextFireTime(0);
                        sendInfoService.insert(sendInfo);
                    }
                }
            }
        }
        if(channelFee != null) {
            cacheHandle.setChannelFee(notPayDeleteDTO.getParkCode(), notPayDeleteDTO.getAisleCode(), channelFee);
        }
        return ObjectResponse.success();
    }

    @Override
    @DS_SLAVE
    public ObjectResponse<Page<OrderNotpay>> getNotPayRecord(NotPayRecordQueryDTO queryDTO, String parkIds) {
        List<Long> parkIdList = Lists.newArrayList();
        if (StringUtils.isNotEmpty(parkIds)) {
            parkIdList = Arrays.stream(parkIds.split(",")).filter(Objects::nonNull).map(Long::parseLong).collect(Collectors.toList());
        }
        return ObjectResponse.success(orderNotpayService.getNotPayRecord(queryDTO, parkIdList));
    }

    @Override
    public ObjectResponse<OrderNotpay> getOrderNotpayByOrderNum(String orderNum) {
        return ObjectResponse.success(orderNotpayService.getOrderNotpayByOrderNum(orderNum));
    }
}
