package com.icetech.park.rpc;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.csp.sentinel.annotation.SentinelResource;
import com.alibaba.fastjson.TypeReference;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.icetech.third.anno.DS_SLAVE;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.parkvip.IParkVipService;
import com.icetech.cloudcenter.api.parkvip.IParkVipUserService;
import com.icetech.order.dao.OrderPayDao;
import com.icetech.basics.dao.park.ParkDao;
import com.icetech.basics.dao.park.ParkInoutdeviceDao;
import com.icetech.park.dao.parkvip.ParkVipMapper;
import com.icetech.user.dao.SaasUserDao;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.user.domain.entity.user.SaasUser;
import com.icetech.cloudcenter.domain.order.SumOrderMoneyDto;
import com.icetech.cloudcenter.domain.pay.OpenParkVipDto;
import com.icetech.cloudcenter.domain.parkvip.ParkTypeNumDto;
import com.icetech.cloudcenter.domain.parkvip.ParkVip;
import com.icetech.cloudcenter.domain.parkvip.ParkVipDto;
import com.icetech.cloudcenter.domain.parkvip.ParkVipEnum;
import com.icetech.cloudcenter.domain.parkvip.Vip;
import com.icetech.cloudcenter.domain.parkvip.VipBuyerRecord;
import com.icetech.cloudcenter.domain.parkvip.VipProductRule;
import com.icetech.cloudcenter.domain.response.ParkDto;
import com.icetech.basics.sentinel.ExceptionUtils;
import com.icetech.park.service.parkvip.CallPhoneNumService;
import com.icetech.park.service.parkvip.ParkVipService;
import com.icetech.park.service.parkvip.VipBuyerRecordService;
import com.icetech.park.service.parkvip.VipProductRuleService;
import com.icetech.park.service.parkvip.VipService;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.constants.SqlConstant;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.redis.handle.RedisHandle;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service("iParkVipServiceImpl")
@Slf4j
public class IParkVipServiceImpl implements IParkVipService {
    @Autowired
    private ParkVipService parkVipService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private ParkInoutdeviceDao parkInoutdeviceDao;
    @Autowired
    private VipService vipService;
    @Autowired
    private VipProductRuleService vipProductRuleService;
    @Autowired
    private CallPhoneNumService callPhoneNumService;
    @Autowired
    private VipBuyerRecordService vipBuyerRecordService;
    @Autowired
    private ParkVipMapper parkVipMapper;
    @Autowired
    private IParkVipUserService parkVipUserService;
    @Autowired
    private SaasUserDao saasUserDao;
    @Autowired
    private ParkDao parkDao;
    @Autowired
    private OrderPayDao orderPayDao;
    @Autowired
    private RedisUtils redisUtils;

    @Autowired
    private RedisHandle redisHandle;
    private static final String LOCK_KEY = "LOCK:VIP";
    public static final String STATISTICS_TIME = "statistics:time:";


    @Override
    public ObjectResponse<Boolean> judgeParkVip(String parkCode) {
        log.info("[判断车场是否vip]|parkCode[{}]",parkCode);
        ObjectResponse<Park> parkResponse = parkService.findByParkCode(parkCode);
        Park park = parkResponse.getData();
        ParkVip parkVip = parkVipService.getOne(Wrappers.lambdaQuery(ParkVip.class).eq(ParkVip::getParkId, park.getId()).last(SqlConstant.LIMIT_ONE));
        if (Objects.nonNull(parkVip) && parkVip.getStatus() == 1){
            return ObjectResponse.success(Boolean.TRUE);
        }
        return ObjectResponse.success(Boolean.FALSE);
    }

    @Override
    public ObjectResponse<ParkVipDto> getParkVipInfo(String parkCode) {
        if (StrUtil.isEmptyOrUndefined(parkCode)) {
            return ObjectResponse.failed(CodeConstants.ERROR_400, "parkCode不能为空");
        }
        Park park = parkService.findByParkCode(parkCode).getData();
        if (park == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_400, "parkCode不存在");
        }
        List<ParkInoutdevice> parkInoutdevices = parkInoutdeviceDao.selectByParkId(park.getId());
        ParkVipDto parkVipDto = new ParkVipDto();
        parkVipDto.setParkName(park.getParkName());
        parkVipDto.setParkCode(park.getParkCode());
        parkVipDto.setExpirationTime(new Date());
        parkVipDto.setExpirationDayNum(0);
        parkVipDto.setChannelNum(parkInoutdevices.size());
        //固定返回开通
        parkVipDto.setStatus(3);
        parkVipDto.setPhoneNum("136");
        return ObjectResponse.success(parkVipDto);
    }

    @Override
//    @Transactional(rollbackFor = Exception.class)
    @DSTransactional
    public ObjectResponse<Void> openVip(OpenParkVipDto openParkVipDto) {
        try {
            VipBuyerRecord byTradeNo = vipBuyerRecordService.getByTradeNo(openParkVipDto.getTradeNo());
            if (Objects.nonNull(byTradeNo)){
                log.warn("[该笔订单已经开通]|tradeNo[{}]",openParkVipDto.getTradeNo());
                return ObjectResponse.success();
            }
            Park park = parkService.findByParkCode(openParkVipDto.getParkCode()).getData();
            //查询
            ParkVip parkVip = parkVipService.getParkVipByParkId(park.getId());
            Vip vip = vipService.getOne(Wrappers.lambdaQuery(Vip.class).eq(Vip::getTargetType, 1).and(query -> {
                query.eq(Vip::getStatus, 1);
            }).last(SqlConstant.LIMIT_ONE));
            VipProductRule vipProductRule = vipProductRuleService.getById(openParkVipDto.getProductId());
            Integer productType = vipProductRule.getProductType();
            Boolean firstOpen = openParkVipDto.getFirstOpen();
            if (Objects.isNull(parkVip)){
                //没有开通记录则为首次开通
                parkVip = new ParkVip();
                parkVip.setParkId(Math.toIntExact(park.getId()));
                parkVip.setVipId(vip.getId());
                //开通时间为24小时之后
                DateTime startDate = DateUtil.date();
                parkVip.setEffectiveTime(startDate);
                parkVip.setExpirationTime(DateUtil.endOfDay(DateUtil.offsetDay(startDate,ParkVipEnum.getDay(productType))));
                parkVip.setDayNum(ParkVipEnum.getDay(productType));
                //从热线号码池中获取热线号码
                parkVip.setPhoneNum(callPhoneNumService.getPhoneNum());
                parkVip.setStatus(1);
                parkVip.setFirstOpen(firstOpen);
                parkVip.setMaxUserNum(ParkVip.MAX_USER);
                parkVip.setVipRuleId(openParkVipDto.getProductId());
                parkVip.setApplyName(openParkVipDto.getUsername());
                parkVip.setApplyTime(new Date());
                parkVip.setCreateTime(new Date());
                parkVipService.save(parkVip);
                //默认开通当前登录账号
                log.info("[车场发起开通]firstOpen{}parkCode{}",firstOpen,park.getParkCode());
                SaasUser load = saasUserDao.load(openParkVipDto.getUserId());
                ObjectResponse<Void> voidObjectResponse = parkVipUserService.authUserVip(park.getParkCode(), Collections.singletonList(load.getUsername()), null);
                log.info("[车场默认授权账户]userName{}开通结果{}",load.getUsername(),voidObjectResponse);
            }else{
                if (firstOpen){
                    log.warn("[该车场已经试用]|parkCode[{}]",openParkVipDto.getParkCode());
                    return ObjectResponse.success();
                }
                //续费
                parkVip.setParkId(Math.toIntExact(park.getId()));
                parkVip.setVipId(vip.getId());
                //开通时间为24小时之后
                Date startDate;
                if (parkVip.getExpirationTime().getTime()<=DateUtil.date().getTime()){
                    startDate =  DateUtil.date();
                }else {
                    startDate = parkVip.getExpirationTime();
                }
                parkVip.setExpirationTime(DateUtil.endOfDay(DateUtil.offsetDay(startDate,ParkVipEnum.getDay(productType))));
                parkVip.setDayNum(ParkVipEnum.getDay(productType)+parkVip.getDayNum());
                //从热线号码池中获取热线号码
//            parkVip.setPhoneNum(callPhoneNumService.getPhoneNum());
                parkVip.setStatus(1);
                parkVip.setFirstOpen(Boolean.FALSE);
                parkVip.setMaxUserNum(ParkVip.MAX_USER);
                parkVip.setVipRuleId(openParkVipDto.getProductId());
//            parkVip.setApplyName(openParkVipDto.getUsername());
//            parkVip.setApplyTime(new Date());
                parkVip.setUpdateTime(new Date());
                log.info("[车场发起续费]{}",park.getParkCode());
                parkVipService.updateById(parkVip);
            }
            //增加开通记录
            VipBuyerRecord vipBuyerRecord = new VipBuyerRecord();
            vipBuyerRecord.setParkId(Math.toIntExact(park.getId()));
            vipBuyerRecord.setUserId(openParkVipDto.getUserId());
            vipBuyerRecord.setVipId(vip.getId());
            vipBuyerRecord.setTradeNo(openParkVipDto.getTradeNo());
            vipBuyerRecord.setOutTradeNo(openParkVipDto.getOutTradeNo());
            vipBuyerRecord.setRemark(ParkVipEnum.getDesc(vipProductRule.getProductType()));
            if (openParkVipDto.getFirstOpen()){
                //使用期金额为 0;
                vipBuyerRecord.setAmount(BigDecimal.ZERO);
            }else {
                vipBuyerRecord.setAmount(vipProductRule.getDiscountAmount());
                vipBuyerRecord.setPayWay(openParkVipDto.getPayWay());
            }
            vipBuyerRecord.setVipRuleId(vipProductRule.getId());
            vipBuyerRecord.setOrderTime(openParkVipDto.getOrderTime());
            vipBuyerRecord.setExpirationTime(parkVip.getExpirationTime());
            vipBuyerRecord.setCreateTime(new Date());
            vipBuyerRecordService.save(vipBuyerRecord);
            return ObjectResponse.success();

        }catch (Exception e){
            log.error("[会员开通异常] openParkVipDto {}",openParkVipDto,e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR);
        }
    }

    @Override
    public ObjectResponse<ParkTypeNumDto> countPark(Integer userId) {
        ObjectResponse<List<ParkDto>> parkList = parkService.getParkList(userId);
        if (!ObjectResponse.isSuccess(parkList)) {
            return ObjectResponse.failed(parkList.getCode(), "没有车场权限");
        }
        int totalParkNUm = parkList.getData().size();
        List<Long> parkIds = parkList.getData().stream().map(ParkDto::getId).collect(Collectors.toList());
        List<ParkVip> parkVipList = parkVipService.list(Wrappers.lambdaQuery(ParkVip.class)
                .in(ParkVip::getParkId, parkIds)
                .and(query -> {
                    query.eq(ParkVip::getStatus, 1);
                }));
        int vipParkNum = parkVipList.size();
        ParkTypeNumDto parkTypeNumDto = new ParkTypeNumDto();
        parkTypeNumDto.setTotalParkNUm(totalParkNUm);
        parkTypeNumDto.setVipParkNum(vipParkNum);
        parkTypeNumDto.setGeneralParkNum(totalParkNUm - vipParkNum);
        return ObjectResponse.success(parkTypeNumDto);
    }

    @Override
    public ObjectResponse<com.icetech.common.domain.Page<ParkVipDto>> getParkVipList(Integer pageNo, Integer pageSize, Integer userId, String parkCode, Integer status) {
        ObjectResponse<List<ParkDto>> parkList = parkService.getParkList(userId);
        List<Long> parkIds = parkList.getData().stream().map(ParkDto::getId).collect(Collectors.toList());
        Page<ParkVipDto> parkVipDtoPage = PageHelper.startPage(pageNo, pageSize).doSelectPage(() -> {
            parkVipMapper.getParkVipList(parkIds, parkCode, status);
        });
        return ObjectResponse.success(com.icetech.common.domain.Page.instance(parkVipDtoPage.getPages(),parkVipDtoPage.getTotal(),parkVipDtoPage.getResult()));
    }


    @Override
    @DS_SLAVE
    @SentinelResource(value = "ParkVipService.getParkVipListWithMoney", defaultFallback = "defaultFallbackHandle", fallbackClass = {ExceptionUtils.class})
    public ObjectResponse<com.icetech.common.domain.Page<ParkVipDto>> getParkVipListWithMoney(Integer pageNo, Integer pageSize, Integer userId, String parkCode) {
        String redisKey = STATISTICS_TIME + userId + pageNo + pageSize + (StrUtil.isNotEmpty(parkCode) ? parkCode : "");
        com.icetech.common.domain.Page<ParkVipDto> parkVipDtoPage = redisHandle.cacheObject(redisKey, new TypeReference<com.icetech.common.domain.Page<ParkVipDto>>() {
        }, () -> {
            ObjectResponse<List<ParkDto>> parkList = parkService.getParkList(userId);
            List<Long> parkIds = parkList.getData().stream().map(ParkDto::getId).collect(Collectors.toList());
            Page<Park> parkListPage = PageHelper.startPage(pageNo, pageSize).doSelectPage(() -> {
                parkDao.selectByIdsAndParkCode(StrUtil.join(",", parkIds), parkCode);
            });
            List<Long> parkDbIds = parkListPage.getResult().stream().map(Park::getId).collect(Collectors.toList());
            if (CollectionUtil.isEmpty(parkDbIds)){
                return com.icetech.common.domain.Page.instance(parkListPage.getPages(), parkListPage.getTotal(), Lists.newArrayList());
            }
            List<ParkVipDto> parkVipDtos = Lists.newArrayList();
            List<ParkVip> parkVips = parkVipService.list(Wrappers.lambdaQuery(ParkVip.class).in(ParkVip::getParkId, parkDbIds));
            Map<Integer, ParkVip> parkVipMap = Maps.newHashMap();
            if (CollectionUtil.isNotEmpty(parkVips)) {
                parkVipMap.putAll(parkVips.stream().collect(Collectors.toMap(ParkVip::getParkId, Function.identity())));
            }
            List<SumOrderMoneyDto> sumOrderMoneyDtos = orderPayDao.getParkListWithMoney(parkDbIds, userId);
            Map<Integer, SumOrderMoneyDto> orderPayMap = Maps.newHashMap();
            if (CollectionUtil.isNotEmpty(sumOrderMoneyDtos)) {
                orderPayMap.putAll(sumOrderMoneyDtos.stream().collect(Collectors.toMap(SumOrderMoneyDto::getParkId, Function.identity())));
            }

            for (Park park : parkListPage.getResult()) {
                ParkVipDto parkVipDto = new ParkVipDto();
                parkVipDto.setParkCode(park.getParkCode());
                parkVipDto.setParkName(park.getParkName());
                ParkVip parkVip = parkVipMap.get(Math.toIntExact(park.getId()));
                if (Objects.nonNull(parkVip) && parkVip.getStatus() == 1) {
                    parkVipDto.setStatus(1);
                    parkVipDto.setExpirationTime(parkVip.getExpirationTime());
                } else {
                    parkVipDto.setStatus(0);
                }
                SumOrderMoneyDto sumOrderMoneyDto = orderPayMap.get((Math.toIntExact(park.getId())));
                if (Objects.nonNull(sumOrderMoneyDto)) {
                    parkVipDto.setReportMoney(sumOrderMoneyDto.getPaidPrice());
                } else {
                    parkVipDto.setReportMoney(BigDecimal.ZERO);
                }
                parkVipDtos.add(parkVipDto);
            }
            return com.icetech.common.domain.Page.instance(parkListPage.getPages(), parkListPage.getTotal(), parkVipDtos);
        }, 30 * 1000);
        return ObjectResponse.success(parkVipDtoPage);
    }
}
