package com.icetech.park.rpc;

import cn.hutool.core.date.DateUtil;
import com.google.common.collect.Lists;
import com.icetech.cloudcenter.api.park.IParkTrusteeshipService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.park.SaasUserParkService;
import com.icetech.cloudcenter.api.user.UserService;
import com.icetech.cloudcenter.domain.park.query.ParkTrusteeshipParam;
import com.icetech.cloudcenter.domain.park.query.ParkTrusteeshipRecordParam;
import com.icetech.cloudcenter.domain.response.ParkTrusteeshipRecordPage;
import com.icetech.cloudcenter.domain.vo.ParkTrusteeshipVo;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.Page;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.park.domain.dto.ParkTrusteeshipDto;
import com.icetech.park.domain.dto.ParkTrusteeshipRecordDto;
import com.icetech.park.domain.entity.ParkTrusteeship;
import com.icetech.park.domain.entity.ParkTrusteeshipRecord;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.park.ParkTrusteeshipRecordService;
import com.icetech.park.service.park.ParkTrusteeshipService;
import com.icetech.user.domain.entity.user.SaasUser;
import com.icetech.user.domain.entity.user.SaasUserPark;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Description IParkTrusteeshipServiceImpl
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2023/6/28 15:22
 */
@Slf4j
@Service("iParkTrusteeshipServiceImpl")
@RequiredArgsConstructor
public class IParkTrusteeshipServiceImpl implements IParkTrusteeshipService {
    
    private final ParkTrusteeshipService parkTrusteeshipService;
    
    private final ParkTrusteeshipRecordService parkTrusteeshipRecordService;
    
    private final ParkService parkService;
    
    private final SaasUserParkService saasUserParkService;
    
    private final UserService userService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ObjectResponse<Boolean> openParkTrusteeship(ParkTrusteeshipVo parkTrusteeshipVo) {
        // 判断当前车场是否开通
        ParkTrusteeship parkTrusteeshipByParkId = parkTrusteeshipService.getParkTrusteeshipByParkId(parkTrusteeshipVo.getParkId());
        if (Objects.nonNull(parkTrusteeshipByParkId)) {
            return ObjectResponse.failed(CodeConstants.ERROR_400, "该车场托管服务已开通");
        }
        return ObjectResponse.success(payParkTrusteeship(null, parkTrusteeshipVo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ObjectResponse<Boolean> renewParkTrusteeship(ParkTrusteeshipVo parkTrusteeshipVo) {
        ParkTrusteeship parkTrusteeshipByParkId = parkTrusteeshipService.getParkTrusteeshipByParkId(parkTrusteeshipVo.getParkId());
        if (Objects.isNull(parkTrusteeshipByParkId)) {
            return ObjectResponse.failed(CodeConstants.ERROR_400, "车场未开通托管");
        }
        return ObjectResponse.success(payParkTrusteeship(parkTrusteeshipByParkId.getId(), parkTrusteeshipVo));
    }

    @Override
    public ObjectResponse<Page<ParkTrusteeshipDto>> searchParkTrusteeship(ParkTrusteeshipParam parkTrusteeshipParam) {
        // 分页查询数据
        Date currentDate = new Date();
        Page<ParkTrusteeship> parkTrusteeshipPage = parkTrusteeshipService.searchParkTrusteeship(parkTrusteeshipParam, currentDate);
        // 数据组装
        if (CollectionUtils.isNotEmpty(parkTrusteeshipPage.getRows())) {
            List<ParkTrusteeshipDto> parkTrusteeshipDtoList = parkTrusteeshipPage.getRows().parallelStream().map(t-> {
                ParkTrusteeshipDto parkTrusteeshipDto = new ParkTrusteeshipDto();
                BeanUtils.copyProperties(t, parkTrusteeshipDto);
                ObjectResponse<Park> byParkId = parkService.findByParkId(t.getParkId());
                if (Objects.nonNull(byParkId)) {
                    parkTrusteeshipDto.setParkName(byParkId.getData().getParkName());
                    parkTrusteeshipDto.setInstitutionName(byParkId.getData().getInstitutionName());
                    List<SaasUserPark> saasUserParks = saasUserParkService.listByParkId(t.getParkId());
                    if (CollectionUtils.isNotEmpty(saasUserParks)) {
                        List<Long> userIds = saasUserParks.stream().map(u-> u.getUserId().longValue()).collect(Collectors.toList());
                        ObjectResponse<List<SaasUser>> saasUserByIds = userService.getSaasUserByIds(userIds);
                        List<SaasUser> saasUsers = saasUserByIds.getData();
                        if (CollectionUtils.isNotEmpty(saasUsers)) {
                            List<String> seatNumList = saasUsers.stream().map(SaasUser::getSeatNumber).filter(StringUtils::isNotEmpty).collect(Collectors.toList());
                            parkTrusteeshipDto.setSipNum(CollectionUtils.isNotEmpty(seatNumList) ? String.join(",", seatNumList) : null);
                        }
                    }
                }
                parkTrusteeshipDto.setTrusteeshipStatus(DateUtil.compare(currentDate, parkTrusteeshipDto.getStartTime()) < 0 ? 1 : (DateUtil.compare(currentDate,
                        parkTrusteeshipDto.getStartTime()) > 0 && DateUtil.compare(currentDate, parkTrusteeshipDto.getEndTime()) < 0 ? 2 : 3) );
                return parkTrusteeshipDto;
            }).collect(Collectors.toList());
            Page<ParkTrusteeshipDto> result = Page.instance(parkTrusteeshipPage.getTotalPage(), parkTrusteeshipPage.getTotal(), parkTrusteeshipDtoList);
            return ObjectResponse.success(result);
        }
        return ObjectResponse.success(Page.instance(parkTrusteeshipPage.getTotalPage(), parkTrusteeshipPage.getTotal(), Lists.newArrayList()));
    }

    @Override
    public ObjectResponse<ParkTrusteeshipRecordPage<ParkTrusteeshipRecordDto>> searchParkTrusteeshipRecord(ParkTrusteeshipRecordParam parkTrusteeshipRecordParam) {
        // 分页查询数据
        Page<ParkTrusteeshipRecord> parkTrusteeshipRecord = parkTrusteeshipRecordService.searchParkTrusteeshipRecord(parkTrusteeshipRecordParam);
        // 数据组装
        if (CollectionUtils.isNotEmpty(parkTrusteeshipRecord.getRows())) {
            List<ParkTrusteeshipRecordDto> parkTrusteeshipRecordDtoList = parkTrusteeshipRecord.getRows().parallelStream().map(t-> {
                ParkTrusteeshipRecordDto parkTrusteeshipRecordDto = new ParkTrusteeshipRecordDto();
                BeanUtils.copyProperties(t, parkTrusteeshipRecordDto);
                ObjectResponse<Park> byParkId = parkService.findByParkId(t.getParkId());
                if (Objects.nonNull(byParkId)) {
                    parkTrusteeshipRecordDto.setParkName(byParkId.getData().getParkName());
                    parkTrusteeshipRecordDto.setInstitutionName(byParkId.getData().getInstitutionName());
                }
                return parkTrusteeshipRecordDto;
            }).collect(Collectors.toList());
            ParkTrusteeshipRecordPage<ParkTrusteeshipRecordDto> result = ParkTrusteeshipRecordPage.instance(parkTrusteeshipRecord.getTotalPage(),
                    parkTrusteeshipRecord.getTotal(), parkTrusteeshipRecordDtoList, parkTrusteeshipRecordService.totalPayAmount(parkTrusteeshipRecordParam));
            // 查询汇总数据
            return ObjectResponse.success(result);
        }
        return ObjectResponse.success(ParkTrusteeshipRecordPage.instance(parkTrusteeshipRecord.getTotalPage(), parkTrusteeshipRecord.getTotal(), Lists.newArrayList(),
                BigDecimal.ZERO));
    }

    @Override
    public ObjectResponse<ParkTrusteeship> getParkTrusteeshipByParkId(Long parkId) {
        return ObjectResponse.success(parkTrusteeshipService.getParkTrusteeshipByParkId(parkId));
    }

    /**
     * Description: 支付托管费用
     * Version1.0 2023-06-29 by wgy 创建
     * 
     * @param trusteeshipId                     托管 id
     * @param parkTrusteeshipVo      参数
     * @return java.lang.Boolean       
     */
    private Boolean payParkTrusteeship(Integer trusteeshipId, ParkTrusteeshipVo parkTrusteeshipVo) {
        ObjectResponse<Park> byParkId = parkService.findByParkId(parkTrusteeshipVo.getParkId());
        ObjectResponse.notError(byParkId);
        ParkTrusteeship parkTrusteeship = new ParkTrusteeship();
        BeanUtils.copyProperties(parkTrusteeshipVo, parkTrusteeship);
        parkTrusteeship.setInstitutionId(byParkId.getData().getInstitutionId());
        parkTrusteeship.setId(trusteeshipId);
        ParkTrusteeshipRecord parkTrusteeshipRecord = new ParkTrusteeshipRecord();
        BeanUtils.copyProperties(parkTrusteeshipVo, parkTrusteeshipRecord);
        parkTrusteeshipRecord.setInstitutionId(byParkId.getData().getInstitutionId());
        parkTrusteeshipRecord.setPayType(Objects.nonNull(trusteeshipId) ? 2 : 1);
        parkTrusteeshipService.saveParkTrusteeship(parkTrusteeship);
        parkTrusteeshipRecordService.save(parkTrusteeshipRecord);
        return true;
    }
}
