package com.icetech.park.rpc;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.TypeReference;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.icetech.basics.dao.park.ParkInoutdeviceDao;
import com.icetech.cloudcenter.api.ManageService;
import com.icetech.cloudcenter.api.park.IParkInOutDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.park.ParkInOutDeviceDto;
import com.icetech.cloudcenter.domain.response.ParkDto;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.UUIDTools;
import com.icetech.oss.OssService;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.handle.PublicHandle;
import com.icetech.redis.handle.RedisHandle;
import com.icetech.third.anno.DS_SLAVE;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

@RefreshScope
@Service("iParkInOutDeviceServiceImpl")
@Slf4j
public class IParkInOutDeviceServiceImpl implements IParkInOutDeviceService {

    @Autowired
    private ParkInoutdeviceDao parkInoutdeviceDao;

    @Autowired
    private ParkService parkService;
    @Autowired
    private OssService ossService;
    @Autowired
    private ManageService manageService;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private PublicHandle publicHandle;
    @Autowired
    private CacheHandle cacheHandle;

    @Value(value = "${channel.default.image}")
    private String badImagePath;
    @Value(value = "${channel.default.offline.image}")
    private String offlineImagePath;
    @Value(value = "${channel.default.no.device.image}")
    private String noDeviceImagePath;
    public static final String STATISTICS_TIME = "channel:catch:";

    @Autowired
    private RedisHandle redisHandle;

    private static final String SOFT_CHANNEL_PRE = "SOFT:CHANNEL:";
    private static final String LOCK_KEY = "LOCK:CHANNEL:SOFT";

    @Override
    public ObjectResponse<com.icetech.common.domain.Page<ParkInOutDeviceDto>> getChannelList(Integer pageNo, Integer pageSize, Integer userId,String parkCode) {
        ObjectResponse<List<ParkDto>> parkList = parkService.getParkList(userId);
        if (ObjectResponse.isSuccess(parkList)){
            List<ParkDto> listData = parkList.getData();
            if (StringUtils.isNotEmpty(parkCode)){
                listData = parkList.getData().stream().filter(parkDto -> parkDto.getParkCode().equals(parkCode)).collect(Collectors.toList());
            }
            if (CollectionUtil.isEmpty(listData)){
                return ObjectResponse.returnNotFoundIfNull(null);
            }
            List<String> waitChannels = new ArrayList<>();
            com.icetech.common.domain.Page<ParkInOutDeviceDto> parkInOutDeviceDtoPage = getChannelList(listData, userId, pageNo, pageSize);
            parkInOutDeviceDtoPage.getRows().parallelStream().forEach(parkInOutDeviceDto -> {
                //设置在线进行软触发
                if (Objects.nonNull(parkInOutDeviceDto.getDeviceId())
                        && parkInOutDeviceDto.getDelFlag() == 0){
                    if (1== parkInOutDeviceDto.getStatus()) {
                        //保证三分钟每个通道软触发一次
                        if (redisUtils.tryLock(LOCK_KEY+parkInOutDeviceDto.getChannelCode(), UUIDTools.getUuid(),60*5000)) {
                            //调用软触发
                            refreshImage(waitChannels, parkInOutDeviceDto);
                        } else {
                            String triggerNo = redisUtils.get(SOFT_CHANNEL_PRE + parkInOutDeviceDto.getChannelCode(), String.class);
                            if (StringUtils.isNotEmpty(triggerNo)){
                                String softImage = cacheHandle.getSoftImage(triggerNo);
                                if (StringUtils.isNotEmpty(softImage)) {
                                    parkInOutDeviceDto.setImageUrl(ossService.getImageUrl(softImage));
                                } else {
                                    refreshImage(waitChannels, parkInOutDeviceDto);
                                }
                            }
                        }
                    }else {
                        //离线
                        parkInOutDeviceDto.setImageUrl(ossService.getImageUrl(offlineImagePath));
                    }
                }else {
                    //设备未添加
                    parkInOutDeviceDto.setImageUrl(ossService.getImageUrl(noDeviceImagePath));
                }
                log.debug("组装数据 channelName {} parkInOutDeviceDto {}",parkInOutDeviceDto.getChannelName(),parkInOutDeviceDto);
            });

            if (!waitChannels.isEmpty()) {
                ThreadUtil.sleep(2000);
                //延时加载图片
                parkInOutDeviceDtoPage.getRows().forEach(parkInOutDeviceDto->{
                    if (waitChannels.contains(parkInOutDeviceDto.getChannelCode())) {
                        setChannelImage(parkInOutDeviceDto);
                    }
                });
            }
            return ObjectResponse.success(parkInOutDeviceDtoPage);
        }
        return ObjectResponse.returnNotFoundIfNull(null);
    }

    private void setChannelImage(ParkInOutDeviceDto parkInOutDeviceDto) {
        String triggerNo = redisUtils.get(SOFT_CHANNEL_PRE + parkInOutDeviceDto.getChannelCode(), String.class);
        if (StringUtils.isNotEmpty(triggerNo)){
            String softImage = cacheHandle.getSoftImage(triggerNo);
            if (StringUtils.isNotEmpty(softImage)) {
                parkInOutDeviceDto.setImageUrl(ossService.getImageUrl(softImage));
            } else {
                //图片加载失败
                parkInOutDeviceDto.setImageUrl(ossService.getImageUrl(badImagePath));
            }
        }
    }

    private void refreshImage(List<String> waitChannels, ParkInOutDeviceDto parkInOutDeviceDto) {
        //调用软触发
        ObjectResponse<String> result = manageService.getCarInfoFromCameraAsync(parkInOutDeviceDto.getParkCode(),
                parkInOutDeviceDto.getProtocolVer(), parkInOutDeviceDto.getSerialNumber(),
                parkInOutDeviceDto.getChannelCode(), parkInOutDeviceDto.getChannelType());
        if (ObjectResponse.isSuccess(result)) {
            log.debug("软触发完成 channelName {} triggerNo {}", parkInOutDeviceDto.getChannelName(), result.getData());
            redisUtils.set(SOFT_CHANNEL_PRE + parkInOutDeviceDto.getChannelCode(), result.getData(), 5 * 60);
        }
        waitChannels.add(parkInOutDeviceDto.getChannelCode());
    }


    @DS_SLAVE
    private com.icetech.common.domain.Page<ParkInOutDeviceDto> getChannelList(List<ParkDto> parkList,Integer userId,Integer pageNo, Integer pageSize){
        List<Long> parkIds = parkList.stream().map(ParkDto::getId).collect(Collectors.toList());
        Map<Long, ParkDto> parkDtoMap = parkList.stream().collect(Collectors.toMap(ParkDto::getId, Function.identity()));
        String redisKey = STATISTICS_TIME + userId +":" + StrUtil.join(",",parkIds) + ":" + pageNo + ":"+ pageSize;
        com.icetech.common.domain.Page<ParkInOutDeviceDto> result = redisHandle.cacheObject(redisKey, new TypeReference<com.icetech.common.domain.Page<ParkInOutDeviceDto>>() {
        }, () -> {
            Page<ParkInOutDeviceDto> parkInOutDeviceDtoPage = PageHelper.startPage(pageNo, pageSize).doSelectPage(() -> {
                parkInoutdeviceDao.getChannelList(parkIds);
            });
            parkInOutDeviceDtoPage.getResult().forEach(parkInOutDeviceDto -> {
                ParkDto parkDto = parkDtoMap.get(Long.valueOf(parkInOutDeviceDto.getParkId()));
                parkInOutDeviceDto.setParkCode(parkDto.getParkCode());
                parkInOutDeviceDto.setParkName(parkDto.getParkName());
            });
            return com.icetech.common.domain.Page.instance(parkInOutDeviceDtoPage.getPages(), parkInOutDeviceDtoPage.getTotal(), parkInOutDeviceDtoPage.getResult());
        }, 30* 1000);
        return result;
    }
}
