package com.icetech.park.rpc;

import cn.hutool.core.date.DateUtil;
import com.icetech.cloudcenter.api.order.IOrderRefundService;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.dao.OrderPayDao;
import com.icetech.basics.dao.park.ParkDao;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.order.domain.entity.OrderRefundRecord;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.cloudcenter.domain.order.OrderRefundDto;
import com.icetech.order.service.OrderRefundRecordService;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.MoneyTool;
import com.icetech.paycenter.api.IPayCenterService;
import com.icetech.paycenter.domain.normalpay.RefundStatus;
import com.icetech.paycenter.domain.normalpay.request.RefundRequest;
import com.icetech.paycenter.domain.normalpay.response.RefundResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;

@Service("iOrderRefundServiceImpl")
@Slf4j
public class IOrderRefundServiceImpl implements IOrderRefundService {
    @Autowired
    private OrderPayDao orderPayDao;
    @Autowired
    private OrderInfoDao orderInfoDao;
    @Autowired
    private IPayCenterService payCenterService;
    @Autowired
    private ParkDao parkDao;
    @Autowired
    private OrderRefundRecordService orderRefundRecordService;

    @Override
    @Transactional
    public ObjectResponse<Void> doRefund(OrderRefundDto orderRefundDto) {
        //查询交易信息
        OrderPay orderPay = new OrderPay();
        orderPay.setTradeNo(orderRefundDto.getTradeNo());
        orderPay.setParkId(orderRefundDto.getParkId());
        orderPay.setPayStatus(PayStatusConstants.PAID);
        orderPay = orderPayDao.selectLimitOne(orderPay);
        if (orderPay == null){
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404.getCode(),"交易流水号不存在");
        }
        Park park = parkDao.selectParkById(orderPay.getParkId());
        //请求退款接口
        RefundRequest request = new RefundRequest();
        request.setParkCode(park.getParkCode());
        request.setTradeNo(orderPay.getTradeNo());
        request.setOutTradeNo(orderPay.getThirdTradeNo());
        request.setRefundTradeNo("R"+ CodeTools.GenerateTradeNo());
        request.setPrice(MoneyTool.fromYuanToFen(orderPay.getPaidPrice()));
        request.setOrderNote(orderRefundDto.getRefundReason());
        ObjectResponse<RefundResponse> refundResponse = payCenterService.refund(request);
        if (!ObjectResponse.isSuccess(refundResponse)){
            return ObjectResponse.failed(CodeConstantsEnum.ERROR.getCode(),refundResponse.getMsg());
        }
        RefundResponse responseData = refundResponse.getData();
        OrderInfo orderInfo = new OrderInfo();
        orderInfo.setOrderNum(orderPay.getOrderNum());
        orderInfo.setParkId(orderPay.getParkId());
        orderInfo = orderInfoDao.selectLimitOne(orderInfo);
        //添加退款记录
        OrderRefundRecord orderRefundRecord = new OrderRefundRecord();
        orderRefundRecord.setParkId(Math.toIntExact(park.getId()));
        orderRefundRecord.setTradeNo(orderPay.getTradeNo());
        orderRefundRecord.setOutTradeNo(responseData.getOutTradeNo());
        orderRefundRecord.setPlateNo(orderInfo.getPlateNum());
        orderRefundRecord.setAmount(new BigDecimal(orderPay.getPaidPrice()));
        orderRefundRecord.setPayWay(orderPay.getPayWay());
        orderRefundRecord.setPayTime(DateUtil.date(orderPay.getPayTime()*1000));
        orderRefundRecord.setRefundTradeNo(request.getRefundTradeNo());
        orderRefundRecord.setRefundReason(orderRefundDto.getRefundReason());
        orderRefundRecord.setRefundTime(new Date());
        orderRefundRecord.setAttachment(orderRefundDto.getAttachment());
        orderRefundRecord.setOptionName(orderRefundDto.getOperAccount());
        orderRefundRecord.setOptionTime(new Date());
        orderRefundRecord.setCreateTime(new Date());
        if (ObjectResponse.isSuccess(refundResponse)){
            orderPay.setRefundStatus(responseData.getRefundStatus());
            orderRefundRecord.setRefundStatus(responseData.getRefundStatus());
        }else {
            orderPay.setRefundStatus(RefundStatus.退款失败.getStatus());
            orderRefundRecord.setRefundStatus(RefundStatus.退款失败.getStatus());
        }
        orderPayDao.updateByEntity(orderPay);
        orderRefundRecordService.save(orderRefundRecord);
        if (ObjectResponse.isSuccess(refundResponse)
                && responseData.getRefundStatus().equals(RefundStatus.退款成功.getStatus())){
            return ObjectResponse.success();
        }
        return ObjectResponse.failed(CodeConstantsEnum.ERROR.getCode(),"退款失败");
    }
}
