package com.icetech.park.handle;

import com.alibaba.fastjson.TypeReference;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.cloudcenter.domain.response.p2c.RemoteSwitchResponse;
import com.icetech.cloudcenter.domain.vo.p2c.ChannelRecentPlateNumsVo;
import com.icetech.cloudcenter.domain.vo.p2c.ParkConnectedDeviceVo;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.FixSizeLinkedList;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.UUIDTools;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 通道缓存管理
 */
@Component
@Slf4j
public class CacheHandle {

    @Autowired
    private RedisUtils redisUtils;
    /**
     * 间隔查询时间，50ms
     */
    private static final int INTERVAL = 50;

    /**
     * WS连接关闭后，清除缓存，部分车场启用
     * @param sn
     * @return
     */
    public boolean closeForClearCache(String sn){
        TokenDeviceVo tokenInfo = getDeviceInfo(sn);
        if (tokenInfo != null) {
            return closeForClearCache(tokenInfo.getParkCode(), sn, tokenInfo.getInandoutCode());
        } else {
            return false;
        }
    }
    /**
     * WS连接关闭后，清除缓存，定时心跳检测后执行
     * @return
     */
    public boolean closeForClearCache(String parkCode, String serialNumber, String inandoutCode){
        //车场已连接设备列表，存放入和出的相机序列号
        String key1 = RedisConstants.PARK_CONNECTED_LIST_PROFILE + parkCode;
        //车场已连接设备,存放通道信息
        String key5 = RedisConstants.DEVICE_SN_PROFILE + serialNumber;

        redisUtils.remove(key5);
        String key4 = RedisConstants.SERIAL_NUMBER_PROFILE + parkCode + "_" + inandoutCode;
        String cacheSerialNumber = redisUtils.get(key4, String.class);
        if (cacheSerialNumber != null && cacheSerialNumber.equals(serialNumber)) {
            redisUtils.remove(key4);
        }

        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key1, 0, -1, ParkConnectedDeviceVo.class);
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            if (vo.getInandoutCode() == null || vo.getDeviceNo() == null) {
                redisUtils.lRemove(key1, 0, vo);
                iterator.remove();
                continue;
            }
            if (vo.getInandoutCode().equals(inandoutCode)
                    && vo.getDeviceNo().equals(serialNumber)) {
                redisUtils.lRemove(key1, 0, vo);
                iterator.remove();
                continue;
            }
            if (vo.getDeviceNo().equals(serialNumber)) {
                redisUtils.lRemove(key1, 0, vo);
                iterator.remove();
                continue;
            }
            //同一个通道的另一个设备将成为这个通道的主相机
            if (vo.getInandoutCode().equals(inandoutCode)) {
                log.info("设备[{}]已经下线，由该通道的另一设备接替成为主相机，新的主相机为：{}", serialNumber, vo.getDeviceNo());
                redisUtils.set(key4, vo.getDeviceNo());
            }
        }
        if (list.isEmpty()) {
            redisUtils.remove(key1);
        }
        log.info("缓存清除成功，serialNumber：[{}]", serialNumber);
        return true;
    }

    /**
     * 缓存SN对应的信息
     * @param sn
     * @param vo
     */
    public void cacheDeviceInfo(String sn, TokenDeviceVo vo){
        //设备相关信息缓存
        String key = RedisConstants.DEVICE_SN_PROFILE + sn;
        //有效期4分钟(掉线后通过定时任务的3分钟检测离线来删除key，防止定时任务找不到设备key)
        redisUtils.set(key, vo, 240L);
    }

    /**
     * 更新SN对应的信息
     * @param sn
     * @param vo
     */
    public void updateDeviceInfo(String sn, TokenDeviceVo vo){
        //设备相关信息缓存
        String key = RedisConstants.DEVICE_SN_PROFILE + sn;
        //有效期4分钟
        redisUtils.set(key, vo, 240L);
    }

    /**
     * 延长设备在线的缓存
     * @param sn
     */
    public void expireDeviceInfo(String sn){
        //设备相关信息缓存
        String key = RedisConstants.DEVICE_SN_PROFILE + sn;
        //有效期4分钟
        redisUtils.expire(key, 240L);
    }

    /**
     * 设置token对应的设备信息
     * @param sn
     * @param vo
     */
    public void setRobotDeviceInfo(String sn, TokenDeviceVo vo){
        //车场已连接设备,存放通道信息
        String key = RedisConstants.P2R_SN_PROFILE + sn;
        //有效期4分钟
        redisUtils.set(key, vo, 240L);
    }

    /**
     * 延长设备在线的缓存
     * @param sn
     */
    public void expireRobotDeviceInfo(String sn){
        //设备相关信息缓存
        String key = RedisConstants.P2R_SN_PROFILE + sn;
        //有效期4分钟
        redisUtils.expire(key, 240L);
    }

    /**
     * 获取token对应的设备信息
     * @param sn
     */
    public TokenDeviceVo getRobotDeviceInfo(String sn){
        //车场已连接设备,存放通道信息
        String key = RedisConstants.P2R_SN_PROFILE + sn;
        return redisUtils.get(key, TokenDeviceVo.class);
    }

    /**
     * 机器人连接关闭后，清除缓存，定时心跳检测后执行
     * @return
     */
    public boolean closeForRobotClearCache(String serialNumber){
        TokenDeviceVo robotTokenInfo = getRobotDeviceInfo(serialNumber);
        if (robotTokenInfo == null) {
            return false;
        }
        String parkCode = robotTokenInfo.getParkCode();
        String inandoutCode = robotTokenInfo.getInandoutCode();
        //车场已连接设备列表，存放入和出的相机序列号
        String key1 = RedisConstants.PARK_ROBOT_CONNECTED_LIST_PROFILE + parkCode;
        //车场已连接设备,存放通道信息
        String key2 = RedisConstants.P2R_SN_PROFILE + serialNumber;
        redisUtils.remove(key2);
        String key4 = RedisConstants.PARK_CHANNEL_ROBOT + parkCode + "_" + inandoutCode;
        String cacheSerialNumber = redisUtils.get(key4, String.class);
        if (cacheSerialNumber != null && cacheSerialNumber.equals(serialNumber)){
            redisUtils.remove(key4);
        }

        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key1, 0, -1, ParkConnectedDeviceVo.class);
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            if (vo.getDeviceNo().equals(serialNumber)) {
                redisUtils.lRemove(key1, 0, vo);
                iterator.remove();
                //同一个通道的另一个设备将成为这个通道的主相机
            } else if (vo.getInandoutCode().equals(inandoutCode)) {
                log.info("设备[{}]已经下线，由该通道的另一设备接替成为主相机，新的主相机为[{}]", serialNumber, vo.getDeviceNo());
                redisUtils.set(key4, vo.getDeviceNo());
            }
        }
        if (list.isEmpty()) {
            redisUtils.remove(key1);
        }
        log.info("机器人缓存清除成功，serialNumber[{}]", serialNumber);
        return true;
    }

    /**
     * 添加车场的连接设备信息到列表
     * @param parkCode
     * @param parkConnectedDeviceVo
     */
    public void addParkConnectList(String parkCode, ParkConnectedDeviceVo parkConnectedDeviceVo){
        String key = RedisConstants.PARK_CONNECTED_LIST_PROFILE + parkCode;
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key, 0, -1, ParkConnectedDeviceVo.class);
        if (list.isEmpty()) {
            list = new ArrayList<>();
        }
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            // 检查是否有重复，如果有，则先移除旧的
            if (vo.getDeviceNo().equals(parkConnectedDeviceVo.getDeviceNo())) {
                redisUtils.lRemove(key, 0, vo);
                iterator.remove();
                continue;
            }
        }
        list.add(parkConnectedDeviceVo);
        log.info("[注册设备] 车场[{}]当前在线设备[{}]台，分别是[{}]", parkCode, list.size(), list);
        redisUtils.lRightPush(key, parkConnectedDeviceVo);
    }

    /**
     * 获取车场在线的相机列表
     * @param parkCode
     * @return
     */
    public List<ParkConnectedDeviceVo> getParkConnectList(String parkCode){
        String key = RedisConstants.PARK_CONNECTED_LIST_PROFILE + parkCode;
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key, 0, -1, ParkConnectedDeviceVo.class);
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        List<String> snList = new ArrayList<>();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            TokenDeviceVo deviceInfo = getDeviceInfo(vo.getDeviceNo());
            if (snList.contains(vo.getDeviceNo()) || deviceInfo == null || !parkCode.equals(deviceInfo.getParkCode())) {
                redisUtils.lRemove(key, 0, vo);
                iterator.remove();
                log.info("[获取最新在线列表] parkCode[{}], 移除不在线设备[{}]", parkCode, vo);
                continue;
            }
            snList.add(vo.getDeviceNo());
        }
        log.info("[获取最新在线列表] parkCode[{}], 设备列表[{}]", parkCode, list);
        return list;
    }

    /**
     * 添加车场的连接设备信息到列表
     * @param parkCode
     * @param parkConnectedDeviceVo
     */
    public void addParkRobotConnectList(String parkCode, ParkConnectedDeviceVo parkConnectedDeviceVo){
        String key1 = RedisConstants.PARK_ROBOT_CONNECTED_LIST_PROFILE + parkCode;
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key1, 0, -1, ParkConnectedDeviceVo.class);
        if (list.isEmpty()) {
            list = new ArrayList<>();
        }
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            // 检查是否有重复，如果有，则先移除旧的
            if (vo.getDeviceNo().equals(parkConnectedDeviceVo.getDeviceNo())) {
                redisUtils.lRemove(key1, 0, vo);
                iterator.remove();
                continue;
            }
        }
        list.add(parkConnectedDeviceVo);
        log.info("[注册机器人设备] 车场[{}]当前机器人在线设备[{}]台，分别是[{}]", parkCode, list.size(), list);
        redisUtils.lRightPush(key1, parkConnectedDeviceVo);
    }

    /**
     * 获取车场的在线机器人列表
     * @param parkCode
     * @return
     */
    public List<ParkConnectedDeviceVo> getParkRobotConnectList(String parkCode){
        String key = RedisConstants.PARK_ROBOT_CONNECTED_LIST_PROFILE + parkCode;
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key, 0, -1, ParkConnectedDeviceVo.class);
        Iterator<ParkConnectedDeviceVo> iterator = list.iterator();
        while (iterator.hasNext()) {
            ParkConnectedDeviceVo vo = iterator.next();
            if (getDeviceInfo(vo.getDeviceNo()) == null) {
                redisUtils.lRemove(key, 0, vo);
                iterator.remove();
                log.info("[获取车场最新机器人在线列表] parkCode[{}], 移除不在线设备[{}]", parkCode, vo);
            }
        }
        log.info("[获取车场最新机器人在线列表] parkCode[{}] 在线设备列表[{}]", parkCode, list);
        return list;
    }

    /**
     * 获取设备信息
     * @param sn
     * @return
     */
    public TokenDeviceVo getDeviceInfo(String sn){
        String key = RedisConstants.DEVICE_SN_PROFILE + sn;
        return redisUtils.get(key, TokenDeviceVo.class);
    }

    /**
     * 获取当前车场已连接设备列表
     * @param parkCode
     * @return
     */
    public List<ParkConnectedDeviceVo> getConnectedList(String parkCode){
        //车场已连接设备列表，存放入和出的相机序列号
        String key = RedisConstants.PARK_CONNECTED_LIST_PROFILE + parkCode;
        List<ParkConnectedDeviceVo> list = redisUtils.lRange(key, 0, -1, ParkConnectedDeviceVo.class);
        return list;
    }
    /**
     * 获取当前通道已连接设备列表
     * @param parkCode
     * @param inandoutCode 出入口编号
     * @return
     */
    public List<ParkConnectedDeviceVo> getConnectedList(String parkCode, String inandoutCode){
        //车场已连接设备列表，存放入和出的相机序列号
        List<ParkConnectedDeviceVo> connectedList = getParkConnectList(parkCode);
        return connectedList.stream().filter(vo -> inandoutCode.equals(vo.getInandoutCode())).collect(Collectors.toList());
    }
    /**
     * 添加或更新入口通道的缓存
     * @param parkCode
     * @param channelCode
     * @param enterRequest
     * @return
     */
    public boolean setEntrance(String parkCode, String channelCode, CarEnterRequest enterRequest) {
        String key = RedisConstants.CHANNEL_ENTRACE_PROFILE + parkCode + "_" + channelCode;
        try {
            redisUtils.set(key, enterRequest, 10 * 60L);
            log.info("<通道缓存操作> 入口通道缓存保存完成，key：{}，value：{}", key, enterRequest);
            return true;
        } catch (Exception e) {
            log.error("<通道缓存操作> redis写入错误，key：{}，value：{}", key, enterRequest);
            return false;
        }
    }
    /**
     * 添加或更新出口通道的缓存
     * @param parkCode 车场编号
     * @param channelCode 通道编号
     * @param exitRequest 离场参数
     * @return 成功/失败
     */
    public boolean setExit(String parkCode, String channelCode, CarExitRequest exitRequest, QueryOrderFeeResponse queryOrderFeeResponse){
        if (queryOrderFeeResponse != null) {
            //缓存费用
            setChannelFee(parkCode, channelCode, queryOrderFeeResponse);

            exitRequest.setTotalAmount(queryOrderFeeResponse.getTotalAmount());
            exitRequest.setPaidAmount(queryOrderFeeResponse.getPaidAmount());
            exitRequest.setDiscountAmount(queryOrderFeeResponse.getDiscountAmount());
            exitRequest.setUnpayPrice(queryOrderFeeResponse.getUnpayPrice());
            exitRequest.setLastPayTime(queryOrderFeeResponse.getPayTime());
        } else {
            removeChannelFee(parkCode, channelCode);
        }
        return setExit(parkCode, channelCode, exitRequest);
    }

    public boolean setExit(String parkCode, String channelCode, CarExitRequest exitRequest){
        String key = RedisConstants.CHANNEL_EXIT_PROFILE + parkCode + "_" + channelCode;
        String key2 = null;
        if (exitRequest.getOrderNum() != null){
            key2 = RedisConstants.CHANNEL_ORDER_EXIT_PROFILE + parkCode + "_" + exitRequest.getOrderNum();
        }
        try{
            redisUtils.set(key, exitRequest, 10 * 60L + 10);
            if (key2 != null){
                redisUtils.set(key2, exitRequest, 10 * 60L + 10);
            }
            log.info("<通道缓存操作> 出口通道缓存保存完成, key[{}], value[{}]", key, exitRequest);
            return true;
        }catch (Exception e){
            log.error("<通道缓存操作> redis写入错误，key：{}，value：{}", key, exitRequest, e);
            return false;
        }
    }
    /**
     * 添加或更新入口通道的缓存
     * @param parkCode
     * @param channelCode
     * @return
     */
    public CarEnterRequest getEntrance(String parkCode, String channelCode){
        String key = RedisConstants.CHANNEL_ENTRACE_PROFILE + parkCode + "_" + channelCode;
        try{
            CarEnterRequest request = redisUtils.get(key, CarEnterRequest.class);
            if (request == null) {
                log.info("<通道缓存操作> 入口通道缓存未获取到数据，key：{}", key);
            }
            return request;
        }catch (Exception e){
            log.error("<通道缓存操作> redis获取错误，key：{}", key);
        }
        return null;
    }
    /**
     * 清除入口通道的缓存
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean removeEntrace(String parkCode, String channelCode) {
        String key = RedisConstants.CHANNEL_ENTRACE_PROFILE + parkCode + "_" + channelCode;
        try {
            if (Boolean.TRUE.equals(redisUtils.remove(key))) {
                log.info("<通道缓存操作> 入口通道缓存清除完成|{}", key);
            } else {
                log.info("<通道缓存操作> 入口通道缓存没有异常数据|{}", key);
            }
            return true;
        } catch (Exception e) {
            log.error("<通道缓存操作> redis删除错误，key：{}", key);
            return false;
        }
    }

    /**
     * 添加或更新出口通道的缓存
     * @param parkCode
     * @param channelCode
     * @return
     */
    public CarExitRequest getExit(String parkCode, String channelCode) {
        String key = RedisConstants.CHANNEL_EXIT_PROFILE + parkCode + "_" + channelCode;
        try {
            CarExitRequest request = redisUtils.get(key, CarExitRequest.class);
            if (request == null) {
                log.info("<通道缓存操作> 出口通道缓存未获取到数据，key：{}", key);
            }
            return request;
        } catch (Exception e) {
            log.error("<通道缓存操作> redis获取错误，key：{}", key);
        }
        return null;
    }

    /**
     * 添加或更新出口通道的缓存
     * @param parkCode
     * @param orderNum
     * @return
     */
    public CarExitRequest getExitByOrderNum(String parkCode, String orderNum) {
        String key = RedisConstants.CHANNEL_ORDER_EXIT_PROFILE + parkCode + "_" + orderNum;
        try {
            CarExitRequest request = redisUtils.get(key, CarExitRequest.class);
            if (request == null) {
                log.info("<通道缓存操作-订单号查询> 出口通道缓存未获取到数据，key：{}", key);
            }
            return request;
        } catch (Exception e) {
            log.error("<通道缓存操作-订单号查询> redis获取错误，key：{}", key);
        }
        return null;
    }

    /**
     * 清除出口通道的缓存
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean removeExit(String parkCode, String channelCode){
        String key = RedisConstants.CHANNEL_EXIT_PROFILE + parkCode + "_" + channelCode;
        try{
            CarExitRequest carExitRequest = redisUtils.get(key, CarExitRequest.class);
            if (carExitRequest != null) {
                log.info("[通道缓存操作] 准备清除key:{}, {}", key, carExitRequest);
                if (carExitRequest.getOrderNum() != null){
                    String key2 = RedisConstants.CHANNEL_ORDER_EXIT_PROFILE + parkCode + "_" + carExitRequest.getOrderNum();
                    redisUtils.remove(key2);
                    log.info("[通道缓存操作] 出口通道订单缓存清除完成key:{}", key);
                }
                redisUtils.remove(key);
                log.info("[通道缓存操作] 出口通道缓存清除完成key:{}", key);
            }
            String key2 = RedisConstants.QUERY_FEE_FAIL_PROFILE + parkCode + "_" + channelCode;
            redisUtils.remove(key2);
            String key3 = RedisConstants.SWITCH_FAIL_PROFILE + parkCode + "_" + channelCode;
            redisUtils.remove(key3);
            return true;
        }catch (Exception e){
            log.error("[通道缓存操作] redis删除错误key:{}", key);
            return false;
        }
    }
    /**
     * 清除出口通道的缓存
     * @param parkCode
     * @param orderNum
     * @return
     */
    public boolean removeExitOrder(String parkCode, String orderNum) {
        String key = RedisConstants.CHANNEL_ORDER_EXIT_PROFILE + parkCode + "_" + orderNum;
        try {
            if (Boolean.TRUE.equals(redisUtils.remove(key))) {
                log.info("<通道订单缓存操作> 清除完成，key：{}", key);
            }
            return true;
        } catch (Exception e) {
            log.error("<通道订单缓存操作> redis删除错误，key：{}", key);
            return false;
        }
    }

    /**
     * 软触发缓存保存
     * @param triggerNo
     * @param enexRequest
     * @return
     */
    @Deprecated
    public boolean setSoftTrigger(String triggerNo, Object enexRequest){
        String key = RedisConstants.SOFT_TRIGGER_PROFILE + triggerNo;
        try{
            redisUtils.set(key, enexRequest, 5*60);
            log.info("<通道缓存操作> 软触发缓存保存完成，key：{}，value：{}", key, enexRequest);
            return true;
        }catch (Exception e){
            log.error("<通道缓存操作> redis写入错误，key：{}，value：{}", key, enexRequest);
            return false;
        }
    }

    public boolean setSoftImage(String triggerNo, String imagePath){
        String key = RedisConstants.SOFT_IMAGE_PROFILE + triggerNo;
        try{
            redisUtils.set(key, imagePath, 5*60);
            return true;
        }catch (Exception e){
            return false;
        }
    }

    public String getSoftImage(String triggerNo){
        String key = RedisConstants.SOFT_IMAGE_PROFILE + triggerNo;
        try{
            return redisUtils.get(key,String.class);
        }catch (Exception e){
            return "";
        }
    }

    /**
     * 获取软触发结果
     * @param triggerNo
     * @return
     */
    @Deprecated
    public Object getSoftTrigger(String triggerNo, int type){
        String key = RedisConstants.SOFT_TRIGGER_PROFILE + triggerNo;
        if (type == 1){
            CarEnterRequest obj = getDataFromRedis(key, 4000L, CarEnterRequest.class);
            return obj;
        }else{
            CarExitRequest obj = getDataFromRedis(key, 4000L, CarExitRequest.class);
            return obj;
        }
    }

    /**
     * 获取远程开关闸返回结果
     * @param messageId
     * @return
     */
    public RemoteSwitchResponse getRemoteSwitch(String messageId){
        ObjectResponse<String> response = getResponseFromRedis(messageId, 11000L);
        if (response == null) {
            return null;
        }
        if (CodeConstants.ERROR_405.equals(response.getCode())){
            RemoteSwitchResponse remoteSwitchResponse = new RemoteSwitchResponse();
            remoteSwitchResponse.setResult(1);
            remoteSwitchResponse.setExecuteTime(DateTools.unixTimestamp());
            remoteSwitchResponse.setImage(null);
            return remoteSwitchResponse;
        }else if (StringUtils.isNotBlank(response.getData())){
            return JsonUtils.parseObject(response.getData(), RemoteSwitchResponse.class);
        }else{
            return null;
        }
    }

    /**
     * 获取data信息
     * @param messageId
     * @param timeOut
     * @return
     */
    public <T> T getDataFromRedis(String messageId, long timeOut, Class<T> clazz){
        String key = messageId;
        long currentTimeMillis = System.currentTimeMillis();
        long lastTime = currentTimeMillis + timeOut;
        int n = 1;
        //查询redis中是否返回
        while (lastTime > currentTimeMillis) {
            T data = redisUtils.get(key, clazz);
            if (data != null) {
                log.info("第{}次从redis中读取到了key：{}响应的信息：{}", n, key, data);
                return data;
            }
            try {
                Thread.sleep(INTERVAL);
            } catch (InterruptedException e) {
                log.warn(String.valueOf(e.getMessage()), e);
            }
            currentTimeMillis = System.currentTimeMillis();
            n ++;
        }
        log.info("时限内未查询到key：{}响应的信息！", key);
        return null;
    }

    /**
     * 获取data信息
     * @param messageId
     * @param timeOut
     * @return
     */
    public ObjectResponse<String> getResponseFromRedis(String messageId, Long timeOut){
        String key = RedisConstants.RESP_MSG_PROFILE + messageId;
        long currentTimeMillis = System.currentTimeMillis();
        long lastTime = currentTimeMillis + timeOut;
        int n = 1;
        //查询redis中是否返回
        while (lastTime > currentTimeMillis) {
            ObjectResponse<String> data = redisUtils.get(key, new TypeReference<ObjectResponse<String>>(){});
            if (data != null) {
                log.info("第{}次从redis中读取到了key：{}响应的信息：{}", n , key, data);
                return data;
            }
            try {
                Thread.sleep(INTERVAL);
            } catch (InterruptedException e) {
                log.warn(String.valueOf(e.getMessage()), e);
            }
            currentTimeMillis = System.currentTimeMillis();
            n ++;
        }
        log.info("时限内未查询到key：{}响应的信息！", key);
        return null;
    }

    /**
     * 根据车场编号和通道编号获取设备序列号
     * @param parkCode
     * @param channelCode
     * @return
     */
    public String getSerialNumber(String parkCode, String channelCode){
        String deviceNo = redisUtils.get(RedisConstants.SERIAL_NUMBER_PROFILE + parkCode + "_" + channelCode, String.class);
        if (deviceNo == null){
            throw new ResponseBodyException(CodeConstants.ERROR_3003, "相机未连接");
        }
        return deviceNo;
    }
    /**
     * 保存当前通道一个相机的设备序列号
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean setChannelSn(String parkCode, String channelCode, String sn, boolean isMaster){
        String key = RedisConstants.SERIAL_NUMBER_PROFILE + parkCode + "_" + channelCode;
        try{
            if (isMaster){
                redisUtils.set(key, sn, 180L);
            }else{
                String otherDeviceNo = redisUtils.get(key, String.class);
                if (StringUtils.isBlank(otherDeviceNo)) {
                    redisUtils.set(key, sn);
                }
            }
            return true;
        }catch (Exception e){
            log.error("<缓存操作-通道相机序列号> redis写入异常，key：{}，value：{}", key, sn);
            return false;
        }
    }
    /**
     * 延长通道SN的缓存有效期
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean expireChannelSn(String parkCode, String channelCode){
        String key = RedisConstants.SERIAL_NUMBER_PROFILE + parkCode + "_" + channelCode;
        //有效期3分钟
        redisUtils.expire(key, 180L);
        return true;
    }

    /**
     * 缓存通道当前车辆的费用
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean setChannelFee(String parkCode, String channelCode, QueryOrderFeeResponse queryOrderFeeResponse){
        String key = RedisConstants.CHANNEL_FEE_PROFILE + parkCode + "_" + channelCode;
        try{
            redisUtils.set(key, queryOrderFeeResponse, 10 * 60L);
            log.info("<通道缓存操作> 费用缓存保存完成, key[{}], value[{}]", key, queryOrderFeeResponse);
            if (Boolean.TRUE.equals(queryOrderFeeResponse.getIsOffline())) {
                redisUtils.set(RedisConstants.OFF_LINE_FEE_PROFILE + queryOrderFeeResponse.getOrderNum(), 1, 10 * 60L);
            }
            return true;
        }catch (Exception e){
            log.error("<通道缓存操作> redis写入错误，key：{}，value：{}", key, queryOrderFeeResponse);
            return false;
        }
    }
    /**
     * 根据车场编号和通道编号获取通道缴费信息
     * @param parkCode
     * @param channelCode
     * @return
     */
    public QueryOrderFeeResponse getChannelFee(String parkCode, String channelCode){
        String key = RedisConstants.CHANNEL_FEE_PROFILE + parkCode + "_" + channelCode;
        QueryOrderFeeResponse queryOrderFeeResponse = redisUtils.get(key, QueryOrderFeeResponse.class);
        if (queryOrderFeeResponse == null){
            log.info("<通道缓存操作> 费用缓存获取为空，parkCode：{}，channelCode：{}", parkCode, channelCode);
        }
        return queryOrderFeeResponse;
    }

    /**
     * 清除通道费用的缓存
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean removeChannelFee(String parkCode, String channelCode){
        String key = RedisConstants.CHANNEL_FEE_PROFILE + parkCode + "_" + channelCode;
        try {
            redisUtils.remove(key);
            log.info("<通道缓存操作> 通道费用缓存清除完成，key：{}", key);
            return true;
        } catch (Exception e){
            log.error("<通道缓存操作> redis删除错误，key：{}", key);
            return false;
        }
    }
    /**
     * 缓存通道当前无入场记录的车牌
     * @param parkCode
     * @param channelCode
     * @return
     */
    @Deprecated
    public boolean setNoEnterPlate(String parkCode, String channelCode, String plateNum){
        String key = RedisConstants.CHANNEL_NOENTER_PROFILE + parkCode + "_" + channelCode;
        try{
            redisUtils.set(key, plateNum, 10L * 60);
            log.info("<通道缓存操作> 当前通道无入场记录的车牌，key：{}，value：{}", key, plateNum);
            return true;
        }catch (Exception e){
            log.error("<通道缓存操作> redis写入错误，key：{}，value：{}", key, plateNum);
            return false;
        }
    }
    /**
     * 缓存通道当前无入场记录的车牌
     * @param parkCode
     * @param channelCode
     * @return
     */
    @Deprecated
    public String getNoEnterPlate(String parkCode, String channelCode){
        String key = RedisConstants.CHANNEL_NOENTER_PROFILE + parkCode + "_" + channelCode;
        try{
            return redisUtils.get(key, String.class);
        }catch (Exception e){
            log.error("<通道缓存操作> redis获取错误，key：{}", key);
        }
        return null;
    }
    /**
     * 缓存主从相机通道最近处理过的车牌信息
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean addMSPlateNums(String parkCode, String channelCode, ChannelRecentPlateNumsVo channelRecentPlateNumsVo){
        String key = RedisConstants.MASTER_SLAVE_CHANNEL_PROFILE + parkCode + "_" + channelCode;
        try{
            FixSizeLinkedList<ChannelRecentPlateNumsVo> fixSizeLinkedList = redisUtils.get(key, new TypeReference<FixSizeLinkedList<ChannelRecentPlateNumsVo>>(){});
            if (fixSizeLinkedList == null){
                fixSizeLinkedList = new FixSizeLinkedList<>(10);
            }
            fixSizeLinkedList.add(channelRecentPlateNumsVo);
            redisUtils.set(key, fixSizeLinkedList, 60 * 30L);
            log.info("<通道缓存-主从缓存操作> 保存最近处理车牌到主从缓存，key：{}，value：{}", key, fixSizeLinkedList);
            return true;
        }catch (Exception e){
            log.error("<通道缓存-主从缓存操作> redis写入错误，key：{}", key);
            return false;
        }
    }
    /**
     * 获取通道最近上报车牌
     * @param parkCode
     * @param channelCode
     * @return
     */
    public FixSizeLinkedList<ChannelRecentPlateNumsVo> getMSPlateNums(String parkCode, String channelCode){
        String key = RedisConstants.MASTER_SLAVE_CHANNEL_PROFILE + parkCode + "_" + channelCode;
        try{
            return redisUtils.get(key, new TypeReference<FixSizeLinkedList<ChannelRecentPlateNumsVo>>(){});
        }catch (Exception e){
            log.error("<通道缓存-主从缓存操作> redis获取错误，key：{}", key, e);
        }
        return null;
    }
    /**
     * 保存通道的的机器人序列号
     * @param parkCode
     * @param channelCode
     * @return
     */
    public boolean setChannelRobot(String parkCode, String channelCode, String serialNumber){
        String key = RedisConstants.PARK_CHANNEL_ROBOT + parkCode + "_" + channelCode;
        try{
            //有效期3分钟
            return redisUtils.set(key, serialNumber, 180L);
        }catch (Exception e){
            log.error("<通道缓存-保存通道机器人序列号> redis获取错误，key：{}", key, e);
        }
        return false;
    }
    /**
     * 获取通道机器人序列号
     * @param parkCode
     * @param channelCode
     * @return
     */
    public String getChannelRobot(String parkCode, String channelCode){
        String key = RedisConstants.PARK_CHANNEL_ROBOT + parkCode + "_" + channelCode;
        try{
            return redisUtils.get(key, String.class);
        }catch (Exception e){
            log.error("<通道缓存-获取通道机器人序列号> redis获取错误，key：{}", key, e);
        }
        return null;
    }

    /**
     * 最多发送10次，大约需要24小时41分钟
     */
    private static final long EXPIRE_TIME_SEND_DEVICES_ = 24 * 3600 + 41 * 60;
    /**
     * 缓存下发设备集合
     * @param sendRequest
     * @param serialNumberArr
     * @return
     */
    public boolean setSendDevices(SendRequest sendRequest, String serialNumberArr){
        if (sendRequest.getParkId() == null
                || sendRequest.getServiceType() == null
                || (sendRequest.getTaskId() == null
                    && sendRequest.getRecordId() == null)) {
            return false;
        }
        String key;
        if (sendRequest.getTaskId() != null) {
            key = RedisConstants.BIZ_SEND_DEVICE_PROFILE + sendRequest.getParkId() +
                    "_" + sendRequest.getServiceType() +
                    "_" + sendRequest.getTaskId();
        } else {
            key = RedisConstants.BIZ_SEND_DEVICE_RECORD + sendRequest.getParkId() +
                    ":" + sendRequest.getServiceType() +
                    ":" + sendRequest.getRecordId();
        }
        try{
            redisUtils.set(key, serialNumberArr, EXPIRE_TIME_SEND_DEVICES_);
            log.info("<下发设备缓存> 缓存设备集合，key：{}，value：{}", key, serialNumberArr);
            return true;
        }catch (Exception e){
            log.error("<下发设备缓存> redis写入错误，key：{}，value：{}", key, serialNumberArr);
            return false;
        }
    }
    /**
     * 获取下发设备
     * @param sendRequest
     * @return
     */
    public String getSendDevices(SendRequest sendRequest){
        if (sendRequest.getParkId() == null
                || sendRequest.getServiceType() == null
                || (sendRequest.getTaskId() == null
                    && sendRequest.getRecordId() == null)) {
            return null;
        }
        String key = null;
        if (sendRequest.getTaskId() != null) {
            key = RedisConstants.BIZ_SEND_DEVICE_PROFILE + sendRequest.getParkId() +
                    "_" + sendRequest.getServiceType() +
                    "_" + sendRequest.getTaskId();
        } else {
            key = RedisConstants.BIZ_SEND_DEVICE_RECORD + sendRequest.getParkId() +
                    ":" + sendRequest.getServiceType() +
                    ":" + sendRequest.getRecordId();
        }
        try{
            return redisUtils.get(key, String.class);
        }catch (Exception e){
            log.error("<获取下发设备> redis获取错误，key：{}", key);
        }
        return null;
    }
    /**
     * 成功响应后更新设备列表
     * @param sendRequest
     * @param serialNumber
     * @return true代表所有设备都已响应成功，false代表还有设备未响应成功
     */
    public boolean retSendDevice(SendRequest sendRequest, String serialNumber){
        if (sendRequest.getParkId() == null
                || sendRequest.getServiceType() == null
                || (sendRequest.getTaskId() == null
                    && sendRequest.getRecordId() == null)) {
            return true;
        }
        String key;
        if (sendRequest.getTaskId() != null) {
            key = RedisConstants.BIZ_SEND_DEVICE_PROFILE + sendRequest.getParkId() +
                    "_" + sendRequest.getServiceType() +
                    "_" + sendRequest.getTaskId();
        } else {
            key = RedisConstants.BIZ_SEND_DEVICE_RECORD + sendRequest.getParkId() +
                    ":" + sendRequest.getServiceType() +
                    ":" + sendRequest.getRecordId();
        }
        String reqId = UUIDTools.getUuid();
        try{
            if (redisUtils.tryGetDistributedLock(key, reqId)){
                String devices = redisUtils.get(key, String.class);
                if (devices != null) {
                    String newDevices = devices.replace(serialNumber + ";", "");
                    if (newDevices.length() == 0){
                        redisUtils.remove(key);
                        log.info("<下发设备响应成功操作> 所有设备都响应成功，key：{}", key);
                        return true;
                    }else {
                        redisUtils.set(key, newDevices, EXPIRE_TIME_SEND_DEVICES_);
                        log.info("<下发设备响应成功操作> 更新设备列表，newDevices：{}", newDevices);
                    }
                }else{
                    return true;
                }
            }
        }catch (Exception e){
            log.error("<下发设备响应成功操作> redis获取错误，serialNumber：{}", serialNumber);
        }finally {
            redisUtils.releaseDistributedLock(key, reqId);
        }
        return false;
    }

    /**
     * 删除通道机器人序列号
     * @param parkCode
     * @param channelCode
     * @return
     */
    public void removeChannelRobot(String parkCode, String channelCode){
        String key = RedisConstants.PARK_CHANNEL_ROBOT + parkCode + "_" + channelCode;
        try{
            redisUtils.remove(key);
        }catch (Exception e){
            log.error("<通道缓存-删除通道机器人序列号> redis获取错误，key：{}", key, e);
        }
    }
    /**
     * 缓存进场记录
     * @param parkCode 车场编号
     * @param carEnterRequest 入场记录
     * @return 响应
     */
    public void cacheEnterRecords(String parkCode, String plateNum, CarEnterRequest carEnterRequest){
        String key = RedisConstants.ENTER_RECORDS_PROFILE + parkCode + "_" + plateNum;
        try{
            redisUtils.lRightPush(key, carEnterRequest);
            redisUtils.expire(key, 600);
        }catch (Exception e){
            log.error("[脱机记录缓存] redis获取错误，key：{}", key, e);
        }
    }
    /**
     * 缓存出场记录
     * @param parkCode 车场编号
     * @param carExitRequest 出场记录
     * @return 响应
     */
    public void cacheExitRecords(String parkCode, String plateNum, CarExitRequest carExitRequest){
        String key = RedisConstants.EXIT_RECORDS_PROFILE + parkCode + "_" + plateNum;
        try{
            redisUtils.lRightPush(key, carExitRequest);
            redisUtils.expire(key, 600);
        }catch (Exception e){
            log.error("[脱机记录缓存] redis获取错误，key：{}", key, e);
        }
    }
    /**
     * 获取缓存入场记录
     * @param parkCode 车场编号
     * @return 响应
     */
    public List<CarEnterRequest> getEnterRecords(String parkCode, String plateNum){
        String key = RedisConstants.ENTER_RECORDS_PROFILE + parkCode + "_" + plateNum;
        try{
            return redisUtils.lRange(key, 0, -1, CarEnterRequest.class);
        }catch (Exception e){
            log.error("[脱机记录缓存] redis获取错误，key：{}", key, e);
        }
        return null;
    }
    /**
     * 获取缓存出场记录
     * @param parkCode 车场编号
     * @return 响应
     */
    public List<CarExitRequest> getExitRecords(String parkCode, String plateNum){
        String key = RedisConstants.EXIT_RECORDS_PROFILE + parkCode + "_" + plateNum;
        try{
            return redisUtils.lRange(key, 0, -1, CarExitRequest.class);
        }catch (Exception e){
            log.error("[脱机记录缓存] redis获取错误，key：{}", key, e);
        }
        return null;
    }
    /**
     * 移除缓存入场记录
     * @param parkCode 车场编号
     * @return 响应
     */
    public void removeEnterRecord(String parkCode, String plateNum, CarEnterRequest obj){
        String key = RedisConstants.ENTER_RECORDS_PROFILE + parkCode + "_" + plateNum;
        try{
            redisUtils.lRemove(key, 0, obj);
        }catch (Exception e){
            log.error("[脱机记录缓存] redis移除错误，key：{}", key, e);
        }
    }
    /**
     * 移除缓存出场记录
     * @param parkCode 车场编号
     * @return 响应
     */
    public void removeExitRecord(String parkCode, String plateNum, CarExitRequest obj){
        String key = RedisConstants.EXIT_RECORDS_PROFILE + parkCode + "_" + plateNum;
        try{
            redisUtils.lRemove(key, 0, obj);
        }catch (Exception e){
            log.error("[脱机记录缓存] redis移除错误，key：{}", key, e);
        }
    }

    /**
     * 判断相机是否有抓拍图片接口的权限
     * @param sn
     * @return
     */
    public boolean judgeTakePicturePrivilege(String sn) {
        String key = RedisConstants.NO_TAKE_PICTURE_FUNC_KEY;
        try {
            Long lastTime = redisUtils.hGet(key, sn, Long.class);
            //无权限时，超过6小时后再次尝试
            if (lastTime == null || DateTools.unixTimestamp() - lastTime > 60 * 60 * 6) {
                return true;
            }
        } catch (Exception e) {
            log.error("[抓拍图片接口权限] redis错误，key：{}", key, e);
        }
        return false;
    }

}
