package com.icetech.park.config;

import com.icetech.park.domain.enums.OrderEvent;
import com.icetech.park.domain.enums.OrderStatus;

import java.util.HashMap;
import java.util.Map;

/**
 * Description OrderStateMachine
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2023/11/24 15:31
 */
public class OrderStateMachine {

    private static final Map<OrderStatus, Map<OrderEvent, OrderStatus>> TRANSITION_MAP = new HashMap<>();

    static {
        initializeTransitions();
    }

    private static void initializeTransitions() {
        addTransition(OrderStatus.PENDING, OrderEvent.CREATE_ORDER, OrderStatus.PENDING);
        addTransition(OrderStatus.PENDING, OrderEvent.ACCEPT_ORDER, OrderStatus.PROCESSING);
        addTransition(OrderStatus.PENDING, OrderEvent.REJECT_ORDER, OrderStatus.RETURNED);
        addTransition(OrderStatus.PENDING, OrderEvent.CANCEL_ORDER, OrderStatus.CANCELLED);
        addTransition(OrderStatus.RETURNED, OrderEvent.REOPEN_ORDER, OrderStatus.PENDING);

        addTransition(OrderStatus.PROCESSING, OrderEvent.COMPLETE_ORDER, OrderStatus.COMPLETED);
        addTransition(OrderStatus.PROCESSING, OrderEvent.CANCEL_ORDER, OrderStatus.CANCELLED);

        addTransition(OrderStatus.RETURNED, OrderEvent.CANCEL_ORDER, OrderStatus.CANCELLED);
    }

    private static void addTransition(OrderStatus currentState, OrderEvent action, OrderStatus nextState) {
        // 如果状态不存在，则创建一个新的状态映射
        TRANSITION_MAP.computeIfAbsent(currentState, k -> new HashMap<>());
        // 添加状态转移
        TRANSITION_MAP.get(currentState).put(action, nextState);
    }

    public static OrderStatus getNextState(OrderStatus currentState, OrderEvent action) {
        // 检查是否存在当前状态的映射
        if (TRANSITION_MAP.containsKey(currentState)) {
            Map<OrderEvent, OrderStatus> actionMap = TRANSITION_MAP.get(currentState);
            // 检查是否存在触发行为的映射
            if (actionMap.containsKey(action)) {
                return actionMap.get(action); // 返回下一个状态
            }
        }
        // 如果映射不存在，返回null表示没有有效的状态转移
        return null;
    }

    public static void main(String[] args) {
        // 使用订单状态机进行状态转移
        OrderStatus currentState = OrderStatus.PENDING;
        OrderEvent action = OrderEvent.ACCEPT_ORDER;
        OrderStatus nextState = OrderStateMachine.getNextState(currentState, action);

        // 打印结果
        if (nextState != null) {
            System.out.println("从状态 " + currentState.getDescription() + " 执行行为 " + action.getDescription() + " 到达状态 " + nextState.getDescription());
        } else {
            System.out.println("无效的状态转移");
        }
    }
}
