package com.icetech.order.domain.entity;

import com.baomidou.mybatisplus.annotation.FieldFill;
import com.baomidou.mybatisplus.annotation.IdType;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.annotation.TableId;
import com.baomidou.mybatisplus.annotation.TableName;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 支付优惠关联表
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author fangchangtong
 * @version 1.0 @date 2024-06-03
 */
@Data
@EqualsAndHashCode(callSuper = false)
@NoArgsConstructor
@TableName("ice_order_pay_discount")
public class OrderPayDiscount implements Serializable {

    private static final long serialVersionUID=1L;

    /**
     * 主键自增长id
     */
    @TableId(value = "id", type = IdType.AUTO)
    private Long id;

    /**
     * 车场id
     */
    private Long parkId;

    /**
     * 订单号
     */
    private String orderNum;

    /**
     * 交易流水号
     */
    private String tradeNo;

    /**
     * 车牌号
     */
    private String plateNum;

    /**
     * 车辆类型(1临时车2月卡车3特殊车辆，默认1)
     */
    private Integer type;

    /**
     * 车型(2大型车、1小型车，默认1)
     */
    private Integer carType;

    /**
     * 进场时间
     */
    private Long enterTime;

    /**
     * 离场时间
     */
    private Long exitTime;
    /**
     * 上次支付时间
     */
    private Long lastPayTime;

    /**
     * 支付时间
     */
    private Date payDate;
    /**
     * 缴费渠道1本地出口支付,2 H5扫码支付,3自助缴费机,4中央收费站,5第三方,6运维管理APP,7免密支付,8停车机器人,9公众号缴费,10遥控器 11支付宝停车缴费
     */
    private Integer payChannel;
    /**
     * 缴费终端（当缴费渠道是1时，此字段是出口名称； 当缴费渠道是3时，为自助缴费机编号;缴费渠道是4时,为收费站编号;缴费渠道是5时,为第三方名称）
     */
    private String payTerminal;

    /**
     * 下单时间
     */
    private Long orderTime;

    /**
     * 出口通道编号
     */
    private String exitChannelId;

    /**
     * 应收金额
     */
    private BigDecimal totalPrice;

    /**
     * 实收金额
     */
    private BigDecimal paidPrice;

    /**
     * 优惠总金额
     */
    private BigDecimal discountPrice;

    /**
     * 缴费方式1现金，2微信，3支付宝，4银联 10优惠抵扣
     */
    private Integer payWay;

    /**
     * 商户端优惠金额
     */
    private BigDecimal merchantDiscountPrice;

    /**
     * 第三方优惠金额
     */
    private BigDecimal thirdDiscountPrice;

    /**
     * vip车辆优惠金额
     */
    private BigDecimal vipDiscountPrice;

    /**
     * 收费员免费放行优惠金额
     */
    private BigDecimal tollDiscountPrice;

    /**
     * 优惠来源（1商户端优惠、2第三方优惠、3vip车辆抵扣优惠、4收费员免费放行优惠），多个之间用逗号隔开
     */
    private String discountSource;

    /**
     * 商户端优惠信息json，[{"merchantId":1, "discountType":2,"discountName":"abc"}],merchantId:商户端优惠，关联商户ID,
     * discountType: 商户端优惠券类型，优惠类型1全免2减免时间3减免金额4折扣5按天优免, discountName 商户端优惠券名称
     */
    private String merchantCouponsInfo;

    /**
     * 第三方优惠信息json，[{"thirdId":1, "discountType":2,"discountName":"abc"}],thirdId:第三方ID,
     * discountType: 商户端优惠券类型，优惠类型1全免2减免时间3减免金额4折扣5按天优免, discountName 商户端优惠券名称
     */
    private String thirdCouponsInfo;

    /**
     * VIP优惠信息json，[{"typeId":"", "discountType":2}],typeId: VIP车辆类型ID,
     * discountType: 商户端优惠券类型，优惠类型1全免2减免时间3减免金额4折扣
     */
    private String vipCouponsInfo;

    /**
     * 免费放行优惠信息json，[{"exitOperaAccount":"", "exitTerminal":""}],exitOperaAccount:免费放行的操作人账号，
     * exitTerminal: 免费放行操作的终端，1 云坐席、2 车场帮APP 、3 云岗亭、4岗亭计费软件、5 车场监控
     */
    private String tollCouponsInfo;

    /**
     * 创建时间
     */
    @TableField(fill = FieldFill.INSERT)
    private Date createTime;

    public void addDiscountSource(Integer discountSource) {
        if (this.discountSource == null) {
            this.discountSource = discountSource.toString();
        } else {
            this.discountSource += ("," + discountSource.toString());
        }
    }

    public void setMerchantDiscountPrice(BigDecimal merchantDiscountPrice) {
        if (merchantDiscountPrice != null) {
            this.merchantDiscountPrice = merchantDiscountPrice.setScale(2, BigDecimal.ROUND_HALF_UP);
        } else {
            this.merchantDiscountPrice = merchantDiscountPrice;
        }
    }

    public void setThirdDiscountPrice(BigDecimal thirdDiscountPrice) {
        if (thirdDiscountPrice != null) {
            this.thirdDiscountPrice = thirdDiscountPrice.setScale(2, BigDecimal.ROUND_HALF_UP);
        } else {
            this.thirdDiscountPrice = thirdDiscountPrice;
        }
    }

    public void setVipDiscountPrice(BigDecimal vipDiscountPrice) {
        if (vipDiscountPrice != null) {
            this.vipDiscountPrice = vipDiscountPrice.setScale(2, BigDecimal.ROUND_HALF_UP);
        } else {
            this.vipDiscountPrice = vipDiscountPrice;
        }
    }

    public void setTollDiscountPrice(BigDecimal tollDiscountPrice) {
        if (tollDiscountPrice != null) {
            this.tollDiscountPrice = tollDiscountPrice.setScale(2, BigDecimal.ROUND_HALF_UP);
        } else {
            this.tollDiscountPrice = tollDiscountPrice;
        }
    }

    @Getter
    @Setter
    @ToString
    public static class MerchantCouponsInfo {
        /**
         * 商户端优惠，关联商户ID，多个之间用逗号隔开
         */
        private String merchantId;
        /**
         * 商户端优惠券类型，优惠类型1全免2减免时间3减免金额4折扣5按天优免，多个之间用逗号隔开
         */
        private String discountType;
        /**
         * 商户端优惠券名称
         */
        private String discountName;
        private BigDecimal discountPrice;
        public void addMerchantId(Long merchantId) {
            if (this.merchantId == null) {
                this.merchantId = merchantId.toString();
            } else {
                if (!Arrays.asList(this.merchantId.split(",")).contains(merchantId.toString())) {
                    this.merchantId += ("," + merchantId.toString());
                }
            }
        }
        public void addDiscountType(Integer discountType) {
            if (this.discountType == null) {
                this.discountType = discountType.toString();
            } else {
                if (!this.discountType.contains(discountType.toString())) {
                    this.discountType += ("," + discountType.toString());
                }
            }
        }
        public void addDiscountName(String discountName) {
            if (this.discountName == null) {
                this.discountName = discountName;
            } else {
                if (!Arrays.asList(this.discountName.split(",")).contains(discountName)) {
                    this.discountName += ("," + discountName);
                }
            }
        }
        public void setDiscountPrice(BigDecimal discountPrice) {
            if (discountPrice != null) {
                this.discountPrice = discountPrice.setScale(2, BigDecimal.ROUND_HALF_UP);
            } else {
                this.discountPrice = discountPrice;
            }
        }
    }
    @Getter
    @Setter
    @ToString
    public static class ThirdCouponsInfo {
        /**
         * 商户端优惠，关联商户ID，多个之间用逗号隔开
         */
        private String thirdId;
        /**
         * 商户端优惠券类型，优惠类型1全免2减免时间3减免金额4折扣5按天优免，多个之间用逗号隔开
         */
        private String discountType;
        /**
         * 商户端优惠券名称
         */
        private String discountName;
        private BigDecimal discountPrice;

        public void addThirdId(Long thirdId) {
            if (thirdId == null) {
                return;
            }
            if (this.thirdId == null) {
                this.thirdId = thirdId.toString();
            } else {
                if (!Arrays.asList(this.thirdId.split(",")).contains(thirdId.toString())) {
                    this.thirdId += ("," + thirdId);
                }
            }
        }
        public void addDiscountType(Integer discountType) {
            if (this.discountType == null) {
                this.discountType = discountType.toString();
            } else {
                if (!this.discountType.contains(discountType.toString())) {
                    this.discountType += ("," + discountType.toString());
                }
            }
        }
        public void addDiscountName(String discountName) {
            if (this.discountName == null) {
                this.discountName = discountName;
            } else {
                if (!Arrays.asList(this.discountName.split(",")).contains(discountName)) {
                    this.discountName += ("," + discountName);
                }
            }
        }

        public void setDiscountPrice(BigDecimal discountPrice) {
            if (discountPrice != null) {
                this.discountPrice = discountPrice.setScale(2, BigDecimal.ROUND_HALF_UP);
            } else {
                this.discountPrice = discountPrice;
            }
        }
    }
    @Getter
    @Setter
    @ToString
    public static class VipCouponsInfo {
        /**
         * VIP类型ID
         */
        private String typeId;
        /**
         * 商户端优惠券类型，优惠类型1全免2减免时间3减免金额4折扣，多个之间用逗号隔开
         */
        private String discountType;
        private BigDecimal discountPrice;

        public void addTypeId(Integer typeId) {
            if (this.typeId == null) {
                this.typeId = typeId.toString();
            } else {
                if (!Arrays.asList(this.typeId.split(",")).contains(typeId.toString())) {
                    this.typeId += ("," + typeId.toString());
                }
            }
        }

        public void setDiscountPrice(BigDecimal discountPrice) {
            if (discountPrice != null) {
                this.discountPrice = discountPrice.setScale(2, BigDecimal.ROUND_HALF_UP);
            } else {
                this.discountPrice = discountPrice;
            }
        }

        public void addDiscountPrice(BigDecimal discountPrice) {
            if (this.discountPrice == null) {
                this.discountPrice = discountPrice;
            } else {
                this.discountPrice = this.discountPrice.add(discountPrice)
                        .setScale(2, BigDecimal.ROUND_HALF_UP);
            }
        }
        public void addDiscountType(Integer discountType) {
            if (this.discountType == null) {
                this.discountType = discountType.toString();
            } else {
                if (!Arrays.asList(this.discountType.split(",")).contains(discountType.toString())) {
                    this.discountType += ("," + discountType.toString());
                }
            }
        }

    }
    @Getter
    @Setter
    @ToString
    public static class TollCouponsInfo {
        /**
         * 免费放行的操作人账号
         */
        private String exitOperaAccount;
        /**
         * 免费放行操作的终端，1 云坐席、2 车场帮APP 、3 云岗亭、4岗亭计费软件、5 车场监控
         */
        private Integer exitTerminal;

        private BigDecimal discountPrice;
    }

    public enum DiscountSourceEnum {
        /**
         * 商户端优惠券
         */
        MERCHANT_COUPONS("1"),
        /**
         * 第三方优惠券
         */
        THIRD_COUPONS("2"),
        /**
         * VIP优惠券
         */
        VIP_COUPONS("3"),
        /**
         * 收费员免费放行优惠券
         */
        TOLL_COUPONS("4");

        private final String desc;

        DiscountSourceEnum(String desc) {
            this.desc = desc;
        }

        public String getDesc() {
            return desc;
        }
    }


    public static OrderPayDiscount.MerchantCouponsInfo getMerchantCouponsInfo(List<OrderDiscount> usedDiscountList, BigDecimal discountPrice) {
        List<OrderDiscount> orderDiscounts = usedDiscountList.stream()
                .filter(orderDiscount -> Integer.valueOf(1).equals(orderDiscount.getFrom()))
                .collect(Collectors.toList());
        if (orderDiscounts.size() == 0) {
            return null;
        }
        OrderPayDiscount.MerchantCouponsInfo merchantCouponsInfo = new OrderPayDiscount.MerchantCouponsInfo();
        for (OrderDiscount orderDiscount : orderDiscounts) {
            merchantCouponsInfo.addMerchantId(orderDiscount.getMerchantId());
            merchantCouponsInfo.addDiscountName(orderDiscount.getDiscountName());
            merchantCouponsInfo.addDiscountType(orderDiscount.getType());
        }
        merchantCouponsInfo.setDiscountPrice(discountPrice);
        return merchantCouponsInfo;
    }

    public static OrderPayDiscount.ThirdCouponsInfo getThirdCouponsInfo(List<OrderDiscount> usedDiscountList, BigDecimal discountPrice) {
        List<OrderDiscount> orderDiscounts = usedDiscountList.stream()
                .filter(orderDiscount -> Integer.valueOf(2).equals(orderDiscount.getFrom()))
                .collect(Collectors.toList());
        if (orderDiscounts.size() == 0) {
            return null;
        }
        OrderPayDiscount.ThirdCouponsInfo thirdCouponsInfo = new OrderPayDiscount.ThirdCouponsInfo();
        for (OrderDiscount orderDiscount : orderDiscounts) {
            thirdCouponsInfo.addThirdId(orderDiscount.getThirdId());
            thirdCouponsInfo.addDiscountName(orderDiscount.getDiscountName());
            thirdCouponsInfo.addDiscountType(orderDiscount.getType());
        }
        thirdCouponsInfo.setDiscountPrice(discountPrice);
        return thirdCouponsInfo;
    }

    public Long getParkTime() {
        Long parkTime = null;
        if (this.payDate == null) {
            return null;
        }
        if (this.lastPayTime != null) {
            parkTime = this.payDate.getTime() / 1000 - this.lastPayTime;
        } else {
            if (this.enterTime != null) {
                parkTime = this.payDate.getTime() / 1000 - this.enterTime;
            }
        }
        return parkTime;
    }

}
