package com.icetech.fee.service.merchant.impl;

import com.icetech.cloudcenter.api.discount.MerchantUserService;
import com.icetech.fee.dao.merchant.MerchantRechargeDao;
import com.icetech.fee.dao.merchant.MerchantUserDao;
import com.icetech.fee.dao.merchant.ParkMerchantDao;
import com.icetech.basics.dao.park.ParkDao;
import com.icetech.cloudcenter.domain.pay.MerchantRechargeDto;
import com.icetech.fee.domain.entity.merchant.MerchantRecharge;
import com.icetech.fee.domain.entity.merchant.MerchantUser;
import com.icetech.fee.domain.entity.merchant.ParkMerchant;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.cloudcenter.domain.response.MerchantDto;
import com.icetech.cloudcenter.domain.response.MerchantUserDto;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.math.BigDecimal;
import java.util.Base64;
import java.util.Date;
import java.util.Objects;

/**
 * @author wangzw
 */
@Slf4j
@Service("merchantUserService")
public class MerchantUserServiceImpl implements MerchantUserService {

    @Autowired
    private MerchantUserDao merchantUserDao;
    @Autowired
    private ParkMerchantDao parkMerchantDao;
    @Autowired
    private ParkDao parkDao;
    @Autowired
    private MerchantRechargeDao merchantRechargeDao;

    // 必须是 16 字节
    public static final String SECRET_KEY = "ZocNQD0mdfcmMKEWmtISpQ==";

    // AES CBC 模式需要 16 字节的 IV
    public static final String INIT_VECTOR = "Aocade0xxermMKEW";

    /**
     * 正则
     */
    public static final String REGEX = "^(?=.*[a-zA-Z])(?=.*\\d).{8,16}$";

    @Override
    public ObjectResponse<MerchantUserDto> login(String loginName, String password, String newPwd){
        try {
            MerchantUser merchantUser = merchantUserDao.selectByLoginName(loginName);
            //资源不存在
            if (Objects.isNull(merchantUser)) {
                return ObjectResponse.failed(CodeConstantsEnum.ERROR_404.getCode(),"用户不存在");
            }
            //校验通过返回登陆信息
            MerchantUserDto merchantUserDto = new MerchantUserDto();

            if (StringUtils.isNotBlank(newPwd)) {
                String decryptPwd = decrypt(newPwd);
                if (StringUtils.isEmpty(decryptPwd)) {
                    return ObjectResponse.failed("1000","密码错误");
                }
                //用户名或密码错误
                if (!DigestUtils.md5Hex(decryptPwd).equalsIgnoreCase(merchantUser.getPassword())){
                    return ObjectResponse.failed("1000","用户名或密码错误");
                }
                if (decryptPwd.length() < 8 || decryptPwd.length() > 16 || !decryptPwd.matches(REGEX)) {
                    merchantUserDto.setUpdateFlag(1);
                }
                // 查询商户信息返回支持是否优惠
                ParkMerchant parkMerchant = new ParkMerchant();
                parkMerchant.setId(merchantUser.getMerchantId());
                ParkMerchant parkMerchantRes = parkMerchantDao.selectById(parkMerchant);
                if (parkMerchantRes != null) {
                    merchantUserDto.setVisitPrivilege(parkMerchantRes.getVisitPrivilege());
                }
            }else {
                //用户名或密码错误
                if (!DigestUtils.md5Hex(password).equalsIgnoreCase(merchantUser.getPassword())){
                    return ObjectResponse.failed("1000","用户名或密码错误");
                }
            }
            BeanUtils.copyProperties(merchantUser, merchantUserDto);
            return ObjectResponse.success(merchantUserDto);
        }catch (Exception e){
            log.error("[登录]异常: {}. loginName[{}]", e.getMessage(), loginName, e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR.getCode(),CodeConstantsEnum.ERROR.getDesc());
        }

    }

    @Override
    public ObjectResponse<MerchantDto> findMerchant(Integer merchantId) {
        //查询商户信息
        ParkMerchant parkMerchant = new ParkMerchant();
        parkMerchant.setId(merchantId);
        ParkMerchant parkMerchantRes = parkMerchantDao.selectById(parkMerchant);

        if (parkMerchantRes == null){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }else{
            MerchantDto merchantDto = new MerchantDto();
            BeanUtils.copyProperties(parkMerchantRes, merchantDto);
            Park park = parkDao.selectParkById(Long.valueOf(parkMerchantRes.getParkId()));
            merchantDto.setParkName(park.getParkName());
            return ObjectResponse.success(merchantDto);
        }
    }

    @Override
    public ObjectResponse<MerchantDto> findMerchantByCode(String merchantCode) {
        //查询商户信息
        ParkMerchant parkMerchantRes = parkMerchantDao.selectByCode(merchantCode);

        if (parkMerchantRes == null){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }else{
            MerchantDto merchantDto = new MerchantDto();
            BeanUtils.copyProperties(parkMerchantRes, merchantDto);
            Park park = parkDao.selectParkById(Long.valueOf(parkMerchantRes.getParkId()));
            merchantDto.setParkName(park.getParkName());
            return ObjectResponse.success(merchantDto);
        }
    }

    @Override
    public ObjectResponse modifyPwd(Integer userId, String oldPwd, String newPwd) {
        String decryptPwd = decrypt(oldPwd);
        String decryptNewPwd = decrypt(newPwd);
        if (decryptNewPwd.length() < 8 || decryptNewPwd.length() > 16) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_401.getCode(),"请输入8-16位密码");
        }
        boolean flag = decryptNewPwd.matches(REGEX);
        if (!flag) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_401.getCode(),"密码需包含字母和数字，长度在8-16之间");
        }
        int n = merchantUserDao.updatePwd(userId, DigestUtils.md5Hex(decryptPwd), DigestUtils.md5Hex(decryptNewPwd));
        //n 受影响的行数
        if (n == 0){
            return ObjectResponse.failed(CodeConstants.ERROR_402,"原密码错误");
        }else{
            return ObjectResponse.success();
        }
    }

    @Override
    public ObjectResponse modifyPhoto(Integer userId, String photoPath) {
        MerchantUser merchantUser = new MerchantUser();
        merchantUser.setId(userId);
        merchantUser.setImgUrl(photoPath);
        Integer n = merchantUserDao.update(merchantUser);
        if (n == 0){
            return ObjectResponse.failed(CodeConstants.ERROR_402);
        }else{
            return ObjectResponse.success();
        }
    }

    @Override
    public ObjectResponse<Void> subMerchantMoney(Integer merchantId, BigDecimal bigDecimal) {
        if (bigDecimal.doubleValue() == 0){
            log.info("[商户优惠余额操作] 商户ID:{}的优惠余额不需要扣除(因为扣除金额参数是0)", merchantId);
            return ObjectResponse.success();
        }
        ParkMerchant parkMerchant = new ParkMerchant();
        parkMerchant.setId(merchantId);
        ParkMerchant ret = parkMerchantDao.selectById(parkMerchant);
        if (Objects.isNull(ret)){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        ret.setBalance(ret.getBalance().subtract(bigDecimal));
        log.info("[商户优惠余额操作] 商户ID:{}的优惠余额扣减{}元", merchantId, bigDecimal);
        parkMerchantDao.update(ret);
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Void> addMerchantMoney(Integer merchantId, BigDecimal bigDecimal) {
        if (bigDecimal.doubleValue() == 0){
            log.info("[商户优惠余额操作] 商户ID:{}的优惠余额不需要退费(因为退费金额参数是0)", merchantId);
            return ObjectResponse.success();
        }
        ParkMerchant parkMerchant = new ParkMerchant();
        parkMerchant.setId(merchantId);
        ParkMerchant ret = parkMerchantDao.selectById(parkMerchant);

        if (Objects.isNull(ret) || ret.getStatus() == 1){
            log.info("[商户优惠余额操作] 商户ID:{}商户不存在或者已经删除", merchantId);
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        ret.setBalance(ret.getBalance().add(bigDecimal));
        log.info("[商户优惠余额操作] 商户ID:{}的优惠余额新增{}元", merchantId, bigDecimal);
        parkMerchantDao.update(ret);
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Void> finishRecharge(MerchantRechargeDto dto) {
        MerchantRecharge query = new MerchantRecharge();
        query.setTradeNo(dto.getTradeNo());
        MerchantRecharge merchantRechargeOld = merchantRechargeDao.selectLimitOne(query);
        if (Objects.nonNull(merchantRechargeOld)){
            return ObjectResponse.success();
        }
        //添加商户充值记录
        BigDecimal amount = new BigDecimal(dto.getAmount());
        ParkMerchant parkMerchant = parkMerchantDao.selectByCode(dto.getMerchantCode());
        BigDecimal balance = parkMerchant.getBalance();
        parkMerchant.setBalance(balance.add(amount));
        MerchantRecharge merchantRecharge = new MerchantRecharge();
        merchantRecharge.setParkId(Math.toIntExact(dto.getParkId()));
        merchantRecharge.setMerchantId(parkMerchant.getId());
        merchantRecharge.setRechargePrice(amount);
        merchantRecharge.setBalance(parkMerchant.getBalance());
        merchantRecharge.setRechargeTime(new Date());
        merchantRecharge.setTradeNo(dto.getTradeNo());
        merchantRecharge.setRechargeType(dto.getRechargeType());
        merchantRecharge.setOperator(dto.getOperator());
        merchantRecharge.setCreateTime(new Date());
        merchantRecharge.setPayMoney(amount);
        merchantRecharge.setPayWay(dto.getPayWay());
        merchantRechargeDao.insert(merchantRecharge);
        parkMerchantDao.update(parkMerchant);
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Void> orderResult(String tradeNo) {
        MerchantRecharge query = new MerchantRecharge();
        query.setTradeNo(tradeNo);
        MerchantRecharge merchantRecharge = merchantRechargeDao.selectLimitOne(query);
        return Objects.nonNull(merchantRecharge)?ObjectResponse.success():ObjectResponse.failed();
    }

    public static String decrypt(String encrypted) {
        try {
            // 创建 AES 密钥和 IV
            SecretKey key = new SecretKeySpec(SECRET_KEY.getBytes("UTF-8"), "AES");
            IvParameterSpec iv = new IvParameterSpec(INIT_VECTOR.getBytes("UTF-8"));
            // 创建解密器
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5PADDING");
            cipher.init(Cipher.DECRYPT_MODE, key, iv);
            // 解密
            byte[] original = cipher.doFinal(Base64.getDecoder().decode(encrypted));
            // 返回解密后的明文
            return new String(original);
        }catch (Exception e) {
            log.error("[登录]解密失败,错误信息{}", e.getMessage());
        }
        return "";
    }
}
