package com.icetech.basics.service.blacklist.impl;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.basics.api.BlacklistService;
import com.icetech.basics.dao.blacklist.BlacklistDao;
import com.icetech.basics.domain.entity.Blacklist;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.request.BlacklistBatchDeleteParam;
import com.icetech.cloudcenter.api.IBatchDownSendService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.third.SendInfoService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.enumeration.PncVersionEnum;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.third.domain.entity.third.SendInfo;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 月卡服务类
 * @author wangzw
 */
@Slf4j
@Service("blacklistService")
public class BlacklistServiceImpl extends BaseServiceImpl<BlacklistDao, Blacklist> implements BlacklistService {

    @Autowired
    private BlacklistDao blacklistDao;
    
    @Autowired
    private SendInfoService sendInfoService;

    @Autowired
    private IBatchDownSendService batchDownSendService;
    
    @Autowired
    private ParkDeviceService parkDeviceService;
    
    @Autowired
    private ParkService parkService;
    @Autowired
    private RedisUtils redisUtils;

    @Override
    public ObjectResponse<Blacklist> getById(Long id) {
        Blacklist blacklist = blacklistDao.load(id);
        if (blacklist != null){
            return ObjectResponse.success(blacklist);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<Blacklist> getValidByPlate(Long parkId, String plateNum) {
        Blacklist blacklist = blacklistDao.selectValidByPlate(parkId, plateNum);
        if (blacklist != null){
            return ObjectResponse.success(blacklist);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<List<Blacklist>> getValidByParkId(Long parkId) {
        List<Blacklist> blacklists = blacklistDao.selectValidByParkId(parkId);
        if (blacklists != null && blacklists.size() > 0){
            return ObjectResponse.success(blacklists);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<List<Blacklist>> getValidByParkId(Long parkId, Date createTime) {
        List<Blacklist> blacklists = blacklistDao.selectValidByParkIdAndCreateTime(parkId, createTime);
        if (blacklists != null && blacklists.size() > 0){
            return ObjectResponse.success(blacklists);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<List<Blacklist>> getValidByIds(List<String> idList) {
        List<Blacklist> blacklists = blacklistDao.selectValidByIds(idList);
        if (blacklists != null && blacklists.size() > 0){
            return ObjectResponse.success(blacklists);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<List<Blacklist>> getByIds(List<Long> idList) {
        List<Blacklist> blacklists = blacklistDao.selectByIds(idList);
        if (blacklists != null && blacklists.size() > 0){
            return ObjectResponse.success(blacklists);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<Integer> addBlackList(Blacklist blacklist) {
        return ObjectResponse.success(blacklistDao.insert(blacklist));
    }

    @Override
    public ObjectResponse<Blacklist> selectByParkIdAndThirdId(Long parkId, Long thirdBlackId) {
        return ObjectResponse.returnNotFoundIfNull(blacklistDao.selectByParkIdAndThirdId(parkId, thirdBlackId));
    }

    @Override
    public ObjectResponse<Void> deleteBlackList(List<Long> parkIds, BlacklistBatchDeleteParam blacklistBatchDeleteParam) {
        if (Objects.isNull(blacklistBatchDeleteParam) || CollectionUtils.isEmpty(blacklistBatchDeleteParam.getBlacklistIds())) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        // 校验车场相机是否都在线
        ObjectResponse<ParkConfig> parkConfigResp = parkService.getParkConfig(blacklistBatchDeleteParam.getParkCode());
        ParkConfig parkConfig = parkConfigResp.getData();
        if (parkConfig.getDataCollection() == 2) {
            String version = redisUtils.hGet(RedisConstants.PNC_VERSION, String.valueOf(parkConfig.getParkId()), String.class);
            if (version != null && PncVersionEnum.getIndex(version) < PncVersionEnum.版本2540.getIndex()) {
                return ObjectResponse.failed(CodeConstants.ERROR, "计费系统版本不支持");
            }
        }
        ObjectResponse<Boolean> allOnline = parkDeviceService.isAllOnline(parkConfig.getParkId(), 1, parkConfig.getDataCollection());
        if (!ObjectResponse.isSuccess(allOnline) || !allOnline.getData()) {
            return ObjectResponse.failed(CodeConstants.ERROR, "相机不在线");
        }
        ObjectResponse<List<Blacklist>> blacklistResponse = getByIds(blacklistBatchDeleteParam.getBlacklistIds());
        if (!ObjectResponse.isSuccess(blacklistResponse)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        List<Long> blacklistIds =
                blacklistResponse.getData().stream().filter(t -> t.getStatus() == 0).filter(t -> parkIds.contains(-1L) || parkIds.contains(t.getParkId())).map(t -> t.getId().longValue()).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(blacklistIds)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        // 批量更新黑名单数据
        LambdaUpdateWrapper<Blacklist> wrapper = Wrappers.lambdaUpdate(Blacklist.class).eq(Blacklist::getStatus, 0).in(Blacklist::getId, blacklistIds);
        Blacklist update = new Blacklist();
        update.setStatus(1);
        blacklistDao.update(update, wrapper);
        // 批量增加 sendInfo 数据
        List<SendInfo> sendInfoList =
                blacklistResponse.getData().stream().filter(t-> blacklistIds.contains(t.getId().longValue()))
                        .map(t -> new SendInfo(t.getParkId(), t.getId().longValue(), DownServiceEnum.黑名单下发.getServiceType())).collect(Collectors.toList());
        sendInfoService.batchInsert(sendInfoList);
        // 异步发送相机
        List<SendRequest> sendRequestList = sendInfoList.stream().map(t-> {
            SendRequest sendRequest = new SendRequest();
            sendRequest.setParkId(t.getParkId());
            sendRequest.setServiceId(t.getServiceId());
            sendRequest.setServiceType(t.getServiceType());
            sendRequest.setTaskId(t.getId());
            return sendRequest;
        }).collect(Collectors.toList());
        batchDownSendService.execute(sendRequestList);
        return ObjectResponse.success();
    }

    @Override
    public ObjectResponse<Blacklist> addBlacklist(Blacklist blacklist) {
        save(blacklist);
        return ObjectResponse.success(blacklist);
    }

    @Override
    public ObjectResponse<Boolean> updateBlacklist(Blacklist blacklist) {
        boolean result = updateById(blacklist);
        return ObjectResponse.success(result);
    }

    @Override
    public ObjectResponse<Boolean> deleteBlacklistById(Long id) {
        boolean result = update(Wrappers.lambdaUpdate(getEntityClass())
                .set(Blacklist::getStatus, 1)
                .eq(Blacklist::getId, id));
        return ObjectResponse.success(result);
    }

    @Override
    public ObjectResponse<List<Blacklist>> selectListByParkIdAndPlateNums(Long parkId, List<String> carNumberList) {
        List<Blacklist> blacklists = blacklistDao.selectList(Wrappers.lambdaQuery(Blacklist.class)
                .eq(Blacklist::getParkId, parkId)
                .eq(Blacklist::getStatus, 0)
                .in(Blacklist::getPlate, carNumberList));
        if (CollectionUtils.isEmpty(blacklists)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        return ObjectResponse.success(blacklists);
    }
}
