package com.icetech.basics.service.other;

import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.icetech.basics.dao.NotWorkDayMapper;
import com.icetech.basics.domain.entity.NotWorkDay;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 非工作日表 服务实现类
 * <p>
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author fangct
 * @version 1.0 @date 2022-12-02
 */
@Slf4j
@Service
public class NotWorkDayServiceImpl extends BaseServiceImpl<NotWorkDayMapper, NotWorkDay> implements NotWorkDayService {
    private static final String url = "https://apis.tianapi.com/jiejiari/index";
    private static final String key = "a784907f951f238bbb427f3d13568d6a";

    @Override
    @Transactional
    public List<String> queryNotDays(Integer year) {
        List<NotWorkDay> notWorkDays = lambdaQuery().likeRight(NotWorkDay::getYmd, year).list();
        if (CollectionUtils.isEmpty(notWorkDays)) {
            //2年后的数据，不调用接口
            if (year - LocalDate.now().getYear() > 1) {
                throw new ResponseBodyException(CodeConstants.ERROR_404, "未发布节假日信息");
            }
            //1年后的数据，当年12月前不调用接口
            if (year - LocalDate.now().getYear() == 1 && LocalDate.now().getMonthValue() < 12) {
                throw new ResponseBodyException(CodeConstants.ERROR_404, "未发布节假日信息");
            }
            List<DayInfo> onlineNotWorkDay = getOnlineNotWorkDay(year);
            if (CollectionUtils.isEmpty(onlineNotWorkDay)) {
                throw new ResponseBodyException(CodeConstants.ERROR_404, "未获取到非工作日数据");
            }
            //节假日没公布时，不保存
            if (onlineNotWorkDay.stream().noneMatch(info -> info.getDate().contains(year + "-10-01"))) {
                throw new ResponseBodyException(CodeConstants.ERROR_404, "未发布节假日信息");
            }
            notWorkDays = onlineNotWorkDay.stream().map(info -> {
                NotWorkDay notWorkDay = new NotWorkDay();
                notWorkDay.setYmd(LocalDate.parse(info.getDate(), DateUtils.DEFAULT_DATE_FORMAT));
                notWorkDay.setDateType(convertDateType(info.getDaycode()));
                return notWorkDay;
            }).collect(Collectors.toList());
            saveBatch(notWorkDays);
        }
        return notWorkDays.stream().map(notWorkDay -> notWorkDay.getYmd().format(DateUtils.DEFAULT_DATE_FORMAT)).collect(Collectors.toList());
    }

    /**
     * 在线获取某年的非工作日
     * @param year 年
     * @return
     */
    public static List<DayInfo> getOnlineNotWorkDay(int year) {
        List<DayInfo> notWorkDayList = new ArrayList<>();
        for (int i = 0; i < 12; i++) {
            List<DayInfo> noWorkDay = getOnlineNotWorkDay(year, i + 1);
            notWorkDayList.addAll(noWorkDay);
        }
        return notWorkDayList;
    }

    public static List<DayInfo> getOnlineNotWorkDay(int year, int month) {
        LocalDate start = LocalDate.of(year, month, 1);
        // 获取当月最后一天
        LocalDate lastDayOfMonth = start.withDayOfMonth(start.lengthOfMonth());

        String fullUrl = url + "?key=" + key +
                "&type=3" +
                "&date=" + DateUtils.DEFAULT_DATE_FORMAT.format(start) + "~" + DateUtils.DEFAULT_DATE_FORMAT.format(lastDayOfMonth);
        String response = HttpUtil.get(fullUrl);
        log.info("请求开放接口参数[{}], 响应[{}]", fullUrl, response);
        if (!StringUtils.startsWith(response, "{")) {
            log.warn("[监控埋点] 请求开放接口获取非工作日失败, response[{}]", response);
            return Collections.emptyList();
        }
        JSONObject map = JSON.parseObject(response);
        if (map.getIntValue("code", 0) != 200) {
            log.warn("[监控埋点] 请求开放接口获取非工作日失败, response[{}]", response);
            return Collections.emptyList();
        }
        return map.getJSONObject("result").getJSONArray("list").toList(DayInfo.class).stream()
                .filter(info -> DayInfo.DAY_CODE_HOLIDAY.equals(info.getDaycode()) || DayInfo.DAY_CODE_WEEKEND.equals(info.getDaycode()))
                .collect(Collectors.toList());
    }

    @Getter
    @Setter
    @ToString
    @NoArgsConstructor
    private static class DayInfo {
        protected String date;
        protected Integer daycode;
        protected Integer weekday;
        protected String cnweekday;
        protected String lunaryear;
        protected String lunarmonth;
        protected String lunarday;
        protected String info;
        protected Integer start;
        protected Integer now;
        protected Integer end;
        protected String holiday;
        protected String name;
        protected String enname;
        protected Integer isnotwork;
        protected String[] vacation;
        protected String remark;
        protected Integer wage;
        protected String tip;
        protected String rest;

        /** 日期类型-工作日 */
        public static final Integer DAY_CODE_WORKDAY = 0;
        /** 日期类型-节假日 */
        public static final Integer DAY_CODE_HOLIDAY = 1;
        /** 日期类型-周末 */
        public static final Integer DAY_CODE_WEEKEND = 2;
        /** 日期类型-调休 */
        public static final Integer DAY_CODE_SWAPDAY = 3;
    }

    private static int convertDateType(Integer dayCode) {
        if (DayInfo.DAY_CODE_HOLIDAY.equals(dayCode)) return NotWorkDay.DATE_TYPE_HOLIDAY;
        if (DayInfo.DAY_CODE_WEEKEND.equals(dayCode)) return NotWorkDay.DATE_TYPE_WEEKEND;
        if (DayInfo.DAY_CODE_SWAPDAY.equals(dayCode)) return NotWorkDay.DATE_TYPE_SWAPDAY;
        return NotWorkDay.DATE_TYPE_WORKDAY;
    }
}
