package com.icetech.web.aop;

import com.icetech.common.constants.LogWarnTypeEnum;
import com.icetech.web.aop.anno.TimeoutWarning;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;

@Slf4j
@Aspect
@Component
public class TimeoutWarningAop {
    @Pointcut("@annotation(com.icetech.web.aop.anno.TimeoutWarning)")
    public void pointcut() {
    }

    @Around("pointcut()")
    public Object process(ProceedingJoinPoint joinPoint) throws Throwable {
        long startTime = System.currentTimeMillis();
        try {
            return joinPoint.proceed(joinPoint.getArgs());
        } finally {
            loggingWarn(joinPoint, System.currentTimeMillis() - startTime);
        }
    }

    private void loggingWarn(ProceedingJoinPoint joinPoint, long elapsedTime) {
        try {
            long timeout = TimeoutWarning.DEFAULT_TIMEOUT;
            LogWarnTypeEnum warnType = TimeoutWarning.DEFAULT_WARN_TYPE;
            String message = null;
            String content = null;
            Object[] contentArgs = null;

            Signature signature = joinPoint.getSignature();
            if (signature instanceof MethodSignature) {
                MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
                TimeoutWarning warning = methodSignature.getMethod().getAnnotation(TimeoutWarning.class);
                if (warning != null) {
                    warnType = warning.warnType();
                    message = warning.message();
                    content = warning.content();
                    contentArgs = warning.contentArgs();
                }
            }

            if (elapsedTime > timeout) {
                if (StringUtils.isBlank(content)) {
                    content = "[监控埋点] alarmType[{}] {}: {}";
                    contentArgs = new Object[]{warnType, message, elapsedTime};
                }

                log.warn(content, contentArgs);
            }
        } catch (Exception e) {
            log.error("超时监控埋点日志输出失败: {}", e.getMessage(), e);
        }
    }
}
