package com.icetech.web.aop;

import com.google.common.collect.Lists;
import com.icetech.common.utils.JsonUtils;
import com.icetech.mq.constants.Constant;
import com.icetech.mq.sender.RabbitSender;
import com.icetech.token.model.LoginUser;
import com.icetech.token.utils.LoginHelper;
import com.icetech.web.aop.anno.Log;
import com.icetech.web.bean.OperatorLogMsg;
import com.icetech.web.utils.ServletUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.Expression;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.http.HttpMethod;
import org.springframework.validation.BindingResult;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.HandlerMapping;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Objects;

/**
 * Description 审计日志切面
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2021/6/7 7:27 下午
 */
@Slf4j
@Aspect
@ConditionalOnClass({HttpServletRequest.class, RequestContextHolder.class, LoginHelper.class, RabbitSender.class})
public class AuditLogAspect {

    @Resource
    private RabbitSender rabbitSender;

    /**
     * 用于SpEL表达式解析.
     */
    private final SpelExpressionParser spelExpressionParser = new SpelExpressionParser();
    /**
     * 用于获取方法参数定义名字.
     */
    private final DefaultParameterNameDiscoverer nameDiscoverer = new DefaultParameterNameDiscoverer();

    @AfterReturning(pointcut = "@annotation(auditLog)", returning = "jsonResult")
    public void beforeMethod(JoinPoint joinPoint, Log auditLog, Object jsonResult) {
        handleLog(joinPoint, auditLog, null, jsonResult);
    }

    /**
     * 拦截异常操作
     *
     * @param joinPoint 切点
     * @param e         异常
     */
    @AfterThrowing(value = "@annotation(auditLog)", throwing = "e")
    public void doAfterThrowing(JoinPoint joinPoint, Log auditLog, Exception e) {
        handleLog(joinPoint, auditLog, e, null);
    }

    protected void handleLog(final JoinPoint joinPoint, Log auditLog, final Exception e, Object jsonResult) {
        try {
            // 获取当前的用户
            LoginUser loginUser = LoginHelper.getLoginUser();
            OperatorLogMsg operatorLogMsg = new OperatorLogMsg();
            operatorLogMsg.setStatus(1);
            // 请求的地址
            String ip = ServletUtils.getClientIP();
            operatorLogMsg.setIp(ip);
            operatorLogMsg.setUrl(Objects.requireNonNull(ServletUtils.getRequest()).getRequestURI());
            if (loginUser != null) {
                operatorLogMsg.setPlatformType(loginUser.getPlatformType());
                operatorLogMsg.setUserId(loginUser.getUserId());
                operatorLogMsg.setUsername(loginUser.getUsername());
            }
            if (e != null) {
                operatorLogMsg.setStatus(0);
                operatorLogMsg.setErrorMsg(StringUtils.substring(e.getMessage(), 0, 2000));
            }
            // 设置方法名称
            String className = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            operatorLogMsg.setMethodName(className + "." + methodName + "()");
            // 设置请求方式
            operatorLogMsg.setRequestMethod(ServletUtils.getRequest().getMethod());
            // 处理设置注解上的参数
            getControllerMethodDescription(joinPoint, auditLog, operatorLogMsg, jsonResult);

            String operation = auditLog.operatorMsg();
            if (operation.contains("#")) {
                MethodSignature methodSignature = (MethodSignature)joinPoint.getSignature();
                //获取方法参数值
                Object[] args = joinPoint.getArgs();
                operation = getValBySpEL(operation, methodSignature, args);
            }
            operatorLogMsg.setOperationMsg(operation);
            operatorLogMsg.setOperationTime(new Date());
            rabbitSender.sendMessage("", Constant.COMMON_AUDIT_LOG_QUEUE, operatorLogMsg, 0L, Lists.newArrayList(Constant.commonMq));
        } catch (Exception exp) {
            // 记录本地异常日志
            log.error("==前置通知异常==");
            log.error("异常信息:{}", exp.getMessage());
            exp.printStackTrace();
        }
    }

    /**
     * 解析spEL表达式
     */
    private String getValBySpEL(String spEL, MethodSignature methodSignature, Object[] args) {
        //获取方法形参名数组
        String[] paramNames = nameDiscoverer.getParameterNames(methodSignature.getMethod());
        if (paramNames != null && paramNames.length > 0) {
            Expression expression = spelExpressionParser.parseExpression(spEL);
            // spring的表达式上下文对象
            EvaluationContext context = new StandardEvaluationContext();
            // 给上下文赋值
            for(int i = 0; i < args.length; i++) {
                context.setVariable(paramNames[i], args[i]);
            }
            return expression.getValue(context).toString();
        }
        return null;
    }


    /**
     * 获取注解中对方法的描述信息 用于Controller层注解
     *
     * @param log     日志
     * @param operatorLogMsg 操作日志
     * @throws Exception
     */
    public void getControllerMethodDescription(JoinPoint joinPoint, Log log, OperatorLogMsg operatorLogMsg, Object jsonResult) throws Exception {
        // 设置action动作
        operatorLogMsg.setBusinessType(log.businessType().ordinal());
        // 设置标题
        operatorLogMsg.setTitle(log.title());
        // 设置操作人类别
        operatorLogMsg.setOperatorType(log.operatorType().ordinal());
        // 是否需要保存request，参数和值
        if (log.isSaveRequestData()) {
            // 获取参数的信息，传入到数据库中。
            setRequestValue(joinPoint, operatorLogMsg);
        }
        // 是否需要保存response，参数和值
        if (log.isSaveResponseData() && Objects.nonNull(jsonResult)) {
            operatorLogMsg.setJsonResult(StringUtils.substring(JsonUtils.toJsonString(jsonResult), 0, 2000));
        }
    }

    /**
     * 获取请求的参数，放到log中
     *
     * @param operatorLogMsg 操作日志
     * @throws Exception 异常
     */
    private void setRequestValue(JoinPoint joinPoint, OperatorLogMsg operatorLogMsg) throws Exception {
        String requestMethod = operatorLogMsg.getRequestMethod();
        if (HttpMethod.PUT.name().equals(requestMethod) || HttpMethod.POST.name().equals(requestMethod)) {
            String params = argsArrayToString(joinPoint.getArgs());
            operatorLogMsg.setParam(StringUtils.substring(params, 0, 2000));
        } else {
            Map<?, ?> paramsMap = (Map<?, ?>) ServletUtils.getRequest().getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
            operatorLogMsg.setParam(StringUtils.substring(paramsMap.toString(), 0, 2000));
        }
    }

    /**
     * 参数拼装
     */
    private String argsArrayToString(Object[] paramsArray) {
        StringBuilder params = new StringBuilder();
        if (paramsArray != null && paramsArray.length > 0) {
            for (Object o : paramsArray) {
                if (Objects.nonNull(o) && !isFilterObject(o)) {
                    try {
                        params.append(JsonUtils.toJsonString(o)).append(" ");
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return params.toString().trim();
    }

    /**
     * 判断是否需要过滤的对象。
     *
     * @param o 对象信息。
     * @return 如果是需要过滤的对象，则返回true；否则返回false。
     */
    @SuppressWarnings("rawtypes")
    public boolean isFilterObject(final Object o) {
        Class<?> clazz = o.getClass();
        if (clazz.isArray()) {
            return clazz.getComponentType().isAssignableFrom(MultipartFile.class);
        } else if (Collection.class.isAssignableFrom(clazz)) {
            Collection collection = (Collection) o;
            for (Object value : collection) {
                return value instanceof MultipartFile;
            }
        } else if (Map.class.isAssignableFrom(clazz)) {
            Map map = (Map) o;
            for (Object value : map.entrySet()) {
                Map.Entry entry = (Map.Entry) value;
                return entry.getValue() instanceof MultipartFile;
            }
        }
        return o instanceof MultipartFile || o instanceof HttpServletRequest || o instanceof HttpServletResponse
                || o instanceof BindingResult;
    }
}
