package com.icetech.web.configuration;

import com.icetech.web.aop.ControllerLogAspect;
import com.icetech.web.exception.GlobalExceptionHandler;
import com.icetech.web.exception.ICustomizeExceptionAdvice;
import com.icetech.web.filter.LoggingHttpFilter;
import com.icetech.web.filter.SaTokenFilter;
import com.icetech.web.filter.TraceFilter;
import com.icetech.web.filter.WrapperServletRequestFilter;
import com.icetech.web.message.I18nLocaleResolver;
import com.icetech.web.route.RequestMappingScan;
import com.icetech.web.swagger.SwaggerDocRun;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.web.servlet.LocaleResolver;
import springfox.documentation.spring.web.DocumentationCache;

import javax.servlet.DispatcherType;
import javax.servlet.ServletException;
import java.util.Collections;
import java.util.EnumSet;

/**
 * Description IceWebConfiguration
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2021/5/10 10:05 上午
 */
@Configuration
public class IceWebConfiguration {
    @ConditionalOnMissingBean({WrapperServletRequestFilter.class})
    @Bean
    public FilterRegistrationBean<WrapperServletRequestFilter> apiFilterRegistrationBean() {
        FilterRegistrationBean<WrapperServletRequestFilter> bean = new FilterRegistrationBean<>();
        bean.setFilter(new WrapperServletRequestFilter());
        bean.addUrlPatterns("/*");
        bean.setOrder(0);
        return bean;
    }

    /**
     * Description: 异常配置
     * Version1.0 2021-05-10 by wgy 创建
     *
     * @return com.icetech.web.exception.GlobalExceptionHandler
     */
    @ConditionalOnMissingBean({GlobalExceptionHandler.class, ICustomizeExceptionAdvice.class})
    @ConditionalOnClass({ServletException.class})
    @Bean
    public GlobalExceptionHandler globalExceptionHandler() {
        return new GlobalExceptionHandler();
    }

    @Bean
    @ConditionalOnProperty(prefix = "ice.web.aop.controller-log", name = "enable", havingValue = "true", matchIfMissing = false)
    public ControllerLogAspect controllerLogAspect() {
        return new ControllerLogAspect();
    }

    @Bean
    @ConditionalOnProperty(prefix = "ice.web.filter.logging-trace", name = "enable", havingValue = "true", matchIfMissing = true)
    public FilterRegistrationBean<TraceFilter> traceFilterBean() {
        TraceFilter filter = new TraceFilter();
        FilterRegistrationBean<TraceFilter> bean = new FilterRegistrationBean<>(filter);
        bean.setOrder(Ordered.HIGHEST_PRECEDENCE);
        bean.setUrlPatterns(Collections.singleton("/*"));
        bean.setDispatcherTypes(EnumSet.allOf(DispatcherType.class));
        return bean;
    }

    @Bean
    @ConditionalOnProperty(prefix = "ice.web.filter.logging-http", name = "enable", havingValue = "true", matchIfMissing = true)
    public FilterRegistrationBean<LoggingHttpFilter> loggingHttpFilterBean() {
        // 直接使用new实例化对象无法给LoggingHttpFilter中的@Value属性赋值, 故此处额外声明LoggingHttpFilter对象
        LoggingHttpFilter filter = loggingHttpFilter();
        FilterRegistrationBean<LoggingHttpFilter> bean = new FilterRegistrationBean<>(filter);
        bean.setOrder(apiFilterRegistrationBean().getOrder() + 1);
        bean.setUrlPatterns(Collections.singleton("/*"));
        // 默认为 REQUEST
        bean.setDispatcherTypes(EnumSet.allOf(DispatcherType.class));
        return bean;
    }

    @Bean
    @ConditionalOnProperty(prefix = "ice.web.filter.logging-http", name = "enable", havingValue = "true", matchIfMissing = true)
    public LoggingHttpFilter loggingHttpFilter() {
        return new LoggingHttpFilter();
    } 
    
    @Bean
    @ConditionalOnClass(name = "com.icetech.token.utils.LoginHelper")
    public SaTokenFilter saTokenFilter() {
        return new SaTokenFilter();
    }

    /**
     * 自定义注解扫描
     */
    @Bean
    @ConditionalOnClass(name = "com.icetech.mq.sender.RabbitSender")
    @ConditionalOnProperty(DefaultMvcConfiguration.ICE_ROUTE_SCAN)
    public RequestMappingScan resourceAnnotationScan() {
        return new RequestMappingScan();
    }

    /**
     * 扫描文档
     */
    @Bean
    @ConditionalOnClass(name = "com.icetech.mq.sender.RabbitSender")
    @ConditionalOnProperty(DefaultMvcConfiguration.ICE_ROUTE_SCAN)
    @ConditionalOnBean(DocumentationCache.class)
    public SwaggerDocRun swaggerDocRun() {
        return new SwaggerDocRun();
    }

    @Bean
    public LocaleResolver localeResolver() {
        return new I18nLocaleResolver();
    }

}
