package com.icetech.web.aop;

import java.lang.reflect.Method;
import java.util.Arrays;

import javax.servlet.http.HttpServletRequest;

import com.alibaba.fastjson.JSON;
import com.icetech.common.annotation.LogIgnore;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * <p>Spring控制器方法AOP日志</p>
 * <p>
 * Description ControllerLogAspect
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0.5
 * @since Date 2021/6/7 7:27 下午
 */
@Slf4j
@Aspect
public class ControllerLogAspect {
    @Value("${ice.web.aop.controller-log.trunc-length:512}")
    private int truncLength;

    @Pointcut("execution( * com.icetech..controller..*(..)) && !@annotation(com.icetech.common.annotation.LogIgnore)")
    public void logPointCut() {
    }

    @Around("logPointCut()")
    public Object doAround(ProceedingJoinPoint pjp) throws Throwable {
        Signature signature = pjp.getSignature();
        if (!(signature instanceof MethodSignature)) {
            return pjp.proceed(pjp.getArgs());
        }

        MethodSignature methodSignature = (MethodSignature) signature;
        Method targetMethod = methodSignature.getMethod();
        LogIgnore ignore = targetMethod.getDeclaringClass().getAnnotation(LogIgnore.class);
        if (ignore != null) {
            return pjp.proceed(pjp.getArgs());
        }

        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (!(requestAttributes instanceof ServletRequestAttributes)) {
            return pjp.proceed(pjp.getArgs());
        }

        long startTime = System.currentTimeMillis();
        // 接收到请求，记录请求内容
        ServletRequestAttributes attributes = (ServletRequestAttributes) requestAttributes;
        HttpServletRequest request = attributes.getRequest();
        String url = request.getRequestURL().toString();
        String prefix = "[" + url + "][" + request.getMethod() + "][" + signature.getDeclaringTypeName() + "." + signature.getName() + "]";
        Object result = null;
        try {
            log.info(prefix + "[receive:{}]", Arrays.toString(pjp.getArgs()));
            result = pjp.proceed();
            String content = JSON.toJSONString(String.valueOf(result));
            log.info(prefix + "[T:{}][return:{}]", System.currentTimeMillis() - startTime,
                    content.length() > truncLength ? content.substring(0, truncLength) + " ......" : content);
            return result;
        } catch (Exception e) {
            log.error(prefix + "[T:{}]执行异常", System.currentTimeMillis() - startTime, e);
            throw e;
        }
    }
}
