package com.icetech.token.config;

import com.icetech.redis.config.RedissonAutoConfigurationCustomizer;
import com.icetech.redis.config.RedissonProperties;
import com.icetech.token.core.dao.PlusSaTokenAloneRedisDao;
import org.redisson.Redisson;
import org.redisson.codec.JsonJacksonCodec;
import org.redisson.config.Config;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.ApplicationContext;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.core.io.Resource;
import org.springframework.util.ReflectionUtils;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 为 SaTokenDao 单独设置Redis连接信息
 * @author kong
 */
@Configuration
public class SaAloneRedisInject implements EnvironmentAware{

	/**
	 * 配置信息的前缀
	 */
	public static final String ALONE_PREFIX = "sa-token.alone-redis";

	private static final String REDIS_PROTOCOL_PREFIX = "redis://";
	private static final String REDISS_PROTOCOL_PREFIX = "rediss://";

	/**
	 * Sa-Token 持久层接口
	 */
	@Autowired(required = false)
	public PlusSaTokenAloneRedisDao saTokenDao;
	@Autowired
	private RedissonProperties redissonProperties;
	@Autowired
	private ApplicationContext ctx;
	@Autowired(required = false)
	private List<RedissonAutoConfigurationCustomizer> redissonAutoConfigurationCustomizers;

	/**
	 * 开始注入
	 */
	@Override
	public void setEnvironment(Environment environment) {
		try {
			// 如果为空或者默认实现，则不进行任何操作
			if(saTokenDao == null) {
				return;
			}
			// 如果配置文件不包含相关配置，则不进行任何操作
			if(environment.getProperty(ALONE_PREFIX + ".host") == null) {
				return;
			}

			// ------------------- 开始注入

			// 获取cfg对象
			RedisProperties redisProperties = Binder.get(environment).bind(ALONE_PREFIX, RedisProperties.class).get();
			Config config = null;
			Method clusterMethod = ReflectionUtils.findMethod(RedisProperties.class, "getCluster");
			Method timeoutMethod = ReflectionUtils.findMethod(RedisProperties.class, "getTimeout");
			Object timeoutValue = ReflectionUtils.invokeMethod(timeoutMethod, redisProperties);
			int timeout;
			if(null == timeoutValue){
				timeout = 10000;
			}else if (!(timeoutValue instanceof Integer)) {
				Method millisMethod = ReflectionUtils.findMethod(timeoutValue.getClass(), "toMillis");
				timeout = ((Long) ReflectionUtils.invokeMethod(millisMethod, timeoutValue)).intValue();
			} else {
				timeout = (Integer)timeoutValue;
			}

			if (redissonProperties.getConfig() != null) {
				try {
					config = Config.fromYAML(redissonProperties.getConfig());
				} catch (IOException e) {
					try {
						config = Config.fromJSON(redissonProperties.getConfig());
					} catch (IOException e1) {
						throw new IllegalArgumentException("Can't parse config", e1);
					}
				}
			} else if (redissonProperties.getFile() != null) {
				try {
					InputStream is = getConfigStream();
					config = Config.fromYAML(is);
				} catch (IOException e) {
					// trying next format
					try {
						InputStream is = getConfigStream();
						config = Config.fromJSON(is);
					} catch (IOException e1) {
						throw new IllegalArgumentException("Can't parse config", e1);
					}
				}
			} else if (redisProperties.getSentinel() != null) {
				Method nodesMethod = ReflectionUtils.findMethod(RedisProperties.Sentinel.class, "getNodes");
				Object nodesValue = ReflectionUtils.invokeMethod(nodesMethod, redisProperties.getSentinel());

				String[] nodes;
				if (nodesValue instanceof String) {
					nodes = convert(Arrays.asList(((String)nodesValue).split(",")));
				} else {
					nodes = convert((List<String>)nodesValue);
				}

				config = new Config();
				config.useSentinelServers()
						.setMasterName(redisProperties.getSentinel().getMaster())
						.addSentinelAddress(nodes)
						.setDatabase(redisProperties.getDatabase())
						.setConnectTimeout(timeout)
						.setPassword(redisProperties.getPassword());
			} else if (clusterMethod != null && ReflectionUtils.invokeMethod(clusterMethod, redisProperties) != null) {
				Object clusterObject = ReflectionUtils.invokeMethod(clusterMethod, redisProperties);
				Method nodesMethod = ReflectionUtils.findMethod(clusterObject.getClass(), "getNodes");
				List<String> nodesObject = (List) ReflectionUtils.invokeMethod(nodesMethod, clusterObject);

				String[] nodes = convert(nodesObject);

				config = new Config();
				config.useClusterServers()
						.addNodeAddress(nodes)
						.setConnectTimeout(timeout)
						.setPassword(redisProperties.getPassword());
			} else {
				config = new Config();
				String prefix = REDIS_PROTOCOL_PREFIX;
				Method method = ReflectionUtils.findMethod(RedisProperties.class, "isSsl");
				if (method != null && (Boolean)ReflectionUtils.invokeMethod(method, redisProperties)) {
					prefix = REDISS_PROTOCOL_PREFIX;
				}

				config.useSingleServer()
						.setAddress(prefix + redisProperties.getHost() + ":" + redisProperties.getPort())
						.setConnectTimeout(timeout)
						.setDatabase(redisProperties.getDatabase())
						.setPassword(redisProperties.getPassword());
			}
			if (redissonAutoConfigurationCustomizers != null) {
				for (RedissonAutoConfigurationCustomizer customizer : redissonAutoConfigurationCustomizers) {
					customizer.customize(config);
				}
			}
			config.setCodec(JsonJacksonCodec.INSTANCE);
			saTokenDao.init(Redisson.create(config));
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * 骗过编辑器，增加配置文件代码提示
	 * @return 配置对象
	 */
	@ConfigurationProperties(prefix = ALONE_PREFIX)
	public RedisProperties getSaAloneRedisConfig() {
		return new RedisProperties();
	}


	private InputStream getConfigStream() throws IOException {
		Resource resource = ctx.getResource(redissonProperties.getFile());
		InputStream is = resource.getInputStream();
		return is;
	}


	private String[] convert(List<String> nodesObject) {
		List<String> nodes = new ArrayList<String>(nodesObject.size());
		for (String node : nodesObject) {
			if (!node.startsWith(REDIS_PROTOCOL_PREFIX) && !node.startsWith(REDISS_PROTOCOL_PREFIX)) {
				nodes.add(REDIS_PROTOCOL_PREFIX + node);
			} else {
				nodes.add(node);
			}
		}
		return nodes.toArray(new String[nodes.size()]);
	}
}
