package com.icetech.rpc.trace;

import cn.hutool.core.lang.Snowflake;
import com.alibaba.fastjson.JSON;
import com.icetech.common.annotation.LogIgnore;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.rpc.*;
import org.slf4j.MDC;
import org.springframework.context.i18n.LocaleContextHolder;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Locale;
import java.util.Objects;

/**
 * @author wanggy 
 */

@Slf4j
@Activate(group = {CommonConstants.PROVIDER, CommonConstants.CONSUMER}, order = -3000, value = "traceIdFilter")
public class TraceIdFilter implements Filter {

    Snowflake snowflake = new Snowflake(1, 0);

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {

        RpcContext rpcContext = RpcContext.getContext();
        String traceId;
        if (rpcContext.isProviderSide()) {
            traceId = rpcContext.getAttachment(com.icetech.common.constants.CommonConstants.TRACE_ID_KEY);
            if (traceId == null) {
                traceId = snowflake.nextIdStr();
            }
            MDC.put(com.icetech.common.constants.CommonConstants.TRACE_ID_KEY, traceId);
            // 获取国际化值
            Locale locale = (Locale) rpcContext.getObjectAttachment(com.icetech.common.constants.CommonConstants.LOCALE_KEY);
            if (Objects.nonNull(locale)) {
                LocaleContextHolder.setLocale(locale);
            }
        } else {
            traceId = MDC.get(com.icetech.common.constants.CommonConstants.TRACE_ID_KEY);
            if (traceId == null) {
                traceId = snowflake.nextIdStr();
            }
            rpcContext.setAttachment(com.icetech.common.constants.CommonConstants.TRACE_ID_KEY, traceId);
            rpcContext.setObjectAttachment(com.icetech.common.constants.CommonConstants.LOCALE_KEY, LocaleContextHolder.getLocale());
        }
        Method invokeMethod;
        try {
            invokeMethod = invoker.getInterface().getMethod(invocation.getMethodName(), invocation.getParameterTypes());
        } catch (Exception e) {
            log.error("[dubbo过滤器]找不到对应接口方法: {}.{}({})", invoker.getInterface().getName(), invocation.getMethodName(), Arrays.toString(invocation.getParameterTypes()), e);
            throw new RpcException(e);
        }
        Annotation annotation = invokeMethod.getAnnotation(LogIgnore.class);
        URL targetUrl = invoker.getUrl();
        String address = targetUrl.getAddress();
        String[] interfaces = invoker.getUrl().getParameter("interface").split("\\.");
        String interfaceName = (interfaces.length > 0 ? interfaces[interfaces.length - 1] : "") + "." + invocation.getMethodName();
        String contextType = rpcContext.isConsumerSide() ? "client" : "server";
        if (Objects.isNull(annotation)) {
            String param = JSON.toJSONString(invocation.getArguments());
            log.info("[{}][{}|{}]receive:{}", contextType, address, interfaceName, StringUtils.isNotEmpty(param) && param.length() > 512 ? param.substring(0, 512) + "......" : param);
        }
        try {
            //开始计时
            long beginTime = System.currentTimeMillis();
            Result result = invoker.invoke(invocation);
            // 判断是否忽略打印日志
            if (Objects.nonNull(annotation)) {
                return result;
            }
            //结束计时
            long endTime = System.currentTimeMillis();
            if (result.hasException()) {
                Throwable throwable = result.getException();
                // 非ObjectResponse
                if (invokeMethod.getReturnType() != ObjectResponse.class) {
                    throw new RpcException(throwable);
                }
                ObjectResponse<?> response = new ObjectResponse<>();
                if (throwable instanceof ResponseBodyException) {
                    ResponseBodyException exception = (ResponseBodyException) throwable;
                    log.error("[dubbo过滤器]调用[{}|{}.{}({})]出错: {}.{}. Arguments[{}]", address, invoker.getInterface().getName(), invocation.getMethodName(), Arrays.toString(invocation.getParameterTypes()), exception.getErrCode(), exception.getMessage(), Arrays.toString(invocation.getArguments()), exception);
                    response.setCode(exception.getErrCode());
                    response.setMsg(exception.getMessage());
                } else {
                    log.error("[dubbo过滤器]调用[{}|{}.{}({})]出错: {}. Arguments[{}]", address, invoker.getInterface().getName(), invocation.getMethodName(), Arrays.toString(invocation.getParameterTypes()), throwable.getMessage(), Arrays.toString(invocation.getArguments()), throwable);
                    response.setCode(CodeConstants.ERROR);
                    response.setMsg(CodeConstants.getName(CodeConstants.ERROR));
                }
                result.setAttachments(result.getAttachments());
                result.setValue(response);
                result.setException(null);
            }
            String value = "";
            Object o = JSON.toJSON(result.getValue());
            if (o != null) {
                value = o.toString();
            }
            log.info("[{}][{}|{}](T:{})return:{}", contextType, address, interfaceName, endTime - beginTime, value.length() > 512 ? value.substring(0, 512) + "......" : value);
            return result;
        } catch (RpcException e) {
            log.error("[dubbo过滤器]调用[{}|{}.{}({})]出错: {}. Arguments[{}]", address, invoker.getInterface().getName(), invocation.getMethodName(), Arrays.toString(invocation.getParameterTypes()), e.getMessage(), Arrays.toString(invocation.getArguments()), e);
            throw e;
        } finally {
            if (rpcContext.isProviderSide()) {
                MDC.remove(com.icetech.common.constants.CommonConstants.TRACE_ID_KEY);
                LocaleContextHolder.resetLocaleContext();
            }
        }
    }
}
