package com.icetech.redis.lock;

import cn.hutool.core.date.DateUtil;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;

import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * Description redisson 分布式锁，自持全局和局部
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2021/5/7 7:27 下午
 */
@Slf4j
public class RedissonDistributedLock {


    private RedissonClient redissonClient;

    /**
     * 锁等待时间
     */
    private int defaultLockWaitTime = 5;

    /**
     * 锁默认失效时间
     */
    private int defaultLockLeaseTime = 10;

    public RedissonClient getRedisson() {
        return redissonClient;
    }

    public RedissonDistributedLock(RedissonClient redissonClient){
        this.redissonClient =redissonClient;
    }
    /**
     * 获取锁对象
     * @param name
     * @return
     */
    public RLock getLock(String name) {
        return getRedisson().getLock(name);
    }

    /**
     * Description: 加锁方法
     *
     * @param <T>      the type parameter
     * @param supplier 方法
     * @param key      锁Key
     * @return the t
     * @throws Exception the exception
     */
    public <T> T locked(Supplier<T> supplier, String key) throws Exception {
        return locked(supplier, key, defaultLockWaitTime, defaultLockLeaseTime);
    }

    /**
     * Description: 加锁方法
     *
     * @param <T>           the type parameter
     * @param supplier      方法
     * @param key           锁Key
     * @param lockWaitTime  等待时间
     * @param lockLeaseTime 默认失效时间
     * @return the t
     */
    public <T> T locked(Supplier<T> supplier, String key, int lockWaitTime, int lockLeaseTime) {
        return locked(supplier, key, lockWaitTime, lockLeaseTime, null);
    }

    /**
     * Description: 加锁方法
     *
     * @param <T>           the type parameter
     * @param supplier      方法
     * @param key           锁Key
     * @param errorSupplier 取锁失败时取指定返回值
     * @return the t
     */
    public <T> T locked(Supplier<T> supplier, String key, Supplier<T> errorSupplier) {
        return doLocked(supplier, key, defaultLockWaitTime, defaultLockLeaseTime, errorSupplier);
    }

    /**
     * Description: 加锁方法
     *
     * @param <T>           the type parameter
     * @param supplier      方法
     * @param key           锁Key
     * @param lockWaitTime  等待时间
     * @param lockLeaseTime 默认失效时间
     * @param errorSupplier 取锁失败时取指定返回值
     * @return the t
     */
    public <T> T locked(Supplier<T> supplier, String key, int lockWaitTime, int lockLeaseTime, Supplier<T> errorSupplier) {
        return doLocked(supplier, key, lockWaitTime, lockLeaseTime, errorSupplier);
    }

    /**
     * Description: 加锁方法
     *
     * @param <T>           the type parameter
     * @param supplier      方法
     * @param key           锁Key
     * @param lockWaitTime  等待时间
     * @param lockLeaseTime 默认失效时间
     * @param errorSupplier 取锁失败时取指定返回值
     * @return the t
     * @throws Exception the exception
     */
    @SneakyThrows
    public <T> T doLocked(Supplier<T> supplier, String key, int lockWaitTime, int lockLeaseTime, Supplier<T> errorSupplier) {
        lockWaitTime = lockWaitTime < 0 ? defaultLockWaitTime : lockWaitTime;
        lockLeaseTime = lockLeaseTime < 0 ? defaultLockLeaseTime : lockLeaseTime;

        RedissonClient redisson = this.getRedisson();
        RLock lock = redisson.getLock(key);
        boolean locked = false;
        try {
            locked = lock.tryLock(lockWaitTime, lockLeaseTime, TimeUnit.SECONDS);
            if (!locked) {
                log.info("分布式锁等待超时! 等待时长:{},当前时间为:{}", defaultLockWaitTime, DateUtil.date());
                if (Objects.isNull(errorSupplier)) {
                    throw new RedissonLockException("获取分布式锁失败");
                } else {
                    return errorSupplier.get();
                }
            }
            return supplier.get();
        } finally {
            if (locked) {
                lock.unlock();
            }
        }
    }


}
