package com.icetech.redis.lock;

import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.StringUtils;
import com.icetech.redis.lock.anno.RedisLock;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.Expression;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.util.Objects;

/**
 * Description RedisLockAspect
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2022/4/19 11:40
 */
@Slf4j
@Aspect
public class RedisLockAspect {
    
    @Resource
    private RedissonDistributedLock redissonDistributedLock;

    /**
     * 用于SpEL表达式解析.
     */
    private final SpelExpressionParser spelExpressionParser = new SpelExpressionParser();
    /**
     * 用于获取方法参数定义名字.
     */
    private final DefaultParameterNameDiscoverer nameDiscoverer = new DefaultParameterNameDiscoverer();

    @Pointcut("@annotation(com.icetech.redis.lock.anno.RedisLock)")
    public void redisLockAspect() {
    }

    @Around("redisLockAspect()")
    public Object lockAroundAction(ProceedingJoinPoint joinPoint) throws Throwable {
        Signature signature = joinPoint.getSignature();
        MethodSignature methodSignature = (MethodSignature)signature;
        Method targetMethod = methodSignature.getMethod();
        RedisLock annotation = targetMethod.getAnnotation(RedisLock.class);
        if (Objects.isNull(annotation)) {
            return joinPoint.proceed(joinPoint.getArgs());
        }
        String lockKey = annotation.lockKey();
        if (StringUtils.isEmpty(lockKey)) {
            throw new RuntimeException("分布式锁key 未设置");
        }
        if (lockKey.contains("#")) {
            //获取方法参数值
            Object[] args = joinPoint.getArgs();
            lockKey = getValBySpEL(annotation.lockKey(), methodSignature, args);
        }
        return redissonDistributedLock.locked(() -> {
            try {
                return joinPoint.proceed(joinPoint.getArgs());
            } catch (Throwable e) {
                if (e instanceof ResponseBodyException) {
                    throw (ResponseBodyException) e;
                }
                throw new RuntimeException(e);
            }
        }, lockKey, annotation.lockWaitTime(), annotation.lockLeaseTime());
    }

    /**
     * 解析spEL表达式
     */
    private String getValBySpEL(String spEL, MethodSignature methodSignature, Object[] args) {
        //获取方法形参名数组
        String[] paramNames = nameDiscoverer.getParameterNames(methodSignature.getMethod());
        if (paramNames != null && paramNames.length > 0) {
            Expression expression = spelExpressionParser.parseExpression(spEL);
            // spring的表达式上下文对象
            EvaluationContext context = new StandardEvaluationContext();
            // 给上下文赋值
            for(int i = 0; i < args.length; i++) {
                context.setVariable(paramNames[i], args[i]);
            }
            return expression.getValue(context).toString();
        }
        return null;
    }
}
