package com.icetech.mq.config;

import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;
import org.springframework.amqp.rabbit.connection.ConnectionNameStrategy;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.MessageConverter;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.amqp.RabbitAutoConfiguration;
import org.springframework.boot.autoconfigure.amqp.RabbitRetryTemplateCustomizer;
import org.springframework.boot.autoconfigure.amqp.RabbitTemplateConfigurer;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Map;


/**
 * RabbitMQ 多mq 配置类
 *
 * @author wgy
 */
@Slf4j
@Configuration
@EnableConfigurationProperties({RabbitMultipleProperties.class})
public class RabbitMultipleAutoConfiguration extends RabbitAutoConfiguration {


    /**
     * 模版存储
     */
    private final Map<String, RabbitTemplate> rabbitTemplateMap = Maps.newConcurrentMap();

    public RabbitMultipleAutoConfiguration(RabbitMultipleProperties rabbitMultipleProperties, ObjectProvider<ConnectionNameStrategy> connectionNameStrategy,
                                           ObjectProvider<MessageConverter> messageConverter, ObjectProvider<RabbitRetryTemplateCustomizer> retryTemplateCustomizers)
        throws Exception {
        List<RabbitMultipleProperties.RabbitPropertiesExtend> mqs = rabbitMultipleProperties.getMqs();
        if (CollectionUtils.isEmpty(mqs)) {
            return;
        }
        for (RabbitMultipleProperties.RabbitPropertiesExtend rabbitProperties : mqs) {
            if (StringUtils.isEmpty(rabbitProperties.getMqName())) {
                log.error("mq name is null");
                throw new Exception("mq name is null");
            }
            try {
                RabbitConnectionFactoryCreatorChildren rabbitConnectionFactoryCreatorChildren = new RabbitConnectionFactoryCreatorChildren();
                CachingConnectionFactory cachingConnectionFactory = rabbitConnectionFactoryCreatorChildren.rabbitConnectionFactory(rabbitProperties,
                    connectionNameStrategy);
                RabbitTemplateConfiguration rabbitTemplateConfiguration = new RabbitTemplateConfigurationChildren();
                RabbitTemplateConfigurer rabbitTemplateConfigurer = rabbitTemplateConfiguration.rabbitTemplateConfigurer(rabbitProperties, messageConverter,
                        retryTemplateCustomizers);
                RabbitTemplate rabbitTemplate = rabbitTemplateConfiguration.rabbitTemplate(rabbitTemplateConfigurer, cachingConnectionFactory);
                rabbitTemplate.setUsePublisherConnection(rabbitProperties.isUsePublisherConnection());
                rabbitTemplateMap.put(rabbitProperties.getMqName(), rabbitTemplate);
            } catch (Exception e) {
                log.error("multiple mq config exception", e);
                throw new Exception("multiple mq config exception");
            }
        }
    }

    /**
     * 通过mq name 查询 rabbitTemplate
     *
     * @param mqName mq 名字
     */
    public RabbitTemplate getRabbitTemplate(String mqName) throws Exception {
        RabbitTemplate rabbitTemplate = rabbitTemplateMap.get(mqName);
        if (rabbitTemplate == null) {
            throw new Exception("mq 信息 未配置");
        }
        return rabbitTemplate;
    }

    /**
     * 获取所有发送模板
     *
     */
    public Map<String, RabbitTemplate> getRabbitTemplateMap() {
        return this.rabbitTemplateMap;
    }

    /**
     * 构造子类
     */
    private static class RabbitConnectionFactoryCreatorChildren extends RabbitConnectionFactoryCreator {

    }

    /**
     * 构造子类
     */
    private static class RabbitTemplateConfigurationChildren extends RabbitTemplateConfiguration {

    }

}
