package com.icetech.mq.listener;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.icetech.common.constants.CommonConstants;
import com.icetech.common.utils.StringUtils;
import com.icetech.mq.IMQExceptionData;
import com.icetech.mq.constants.Constant;
import com.icetech.mq.sender.RabbitSender;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ResolvableType;
import org.springframework.messaging.handler.annotation.Payload;
import org.springframework.util.Assert;

import java.lang.reflect.Type;

/**
 * @author wanggy
 * @Description: 消费者抽象类
 * @date 2021/4/29 18:32
 */
@Slf4j
public abstract class MessageListenerAbstract<T> {

    /**
     * 消费重试次数
     */
    protected int tryCount = 0;

    /**
     *
     * 异常处理 的接口， 子类可以自定义实现类。 在子类中添加如下方法方法
     *
     */
    protected IMQExceptionData iMQExceptionData;

    @Autowired
    private RabbitSender rabbitSender;

    /**
     * 默认mq 名字, 取virtual-host
     */
    @Value("${spring.rabbitmq.mqName:}")
    private String mqName;

    /**
     * 监听队列消息
     *
     * @param msg     消息内容
     * @param channel 渠道
     * @param message 消息体
     * @throws Exception
     */
    @RabbitHandler(isDefault = true)
    public void onMessage(@Payload T msg, Channel channel, Message message) throws Exception {
        MDC.put(CommonConstants.TRACE_ID_KEY, message.getMessageProperties().getCorrelationId());
        String consumerQueue = message.getMessageProperties().getConsumerQueue();
        try {
            // 参数校验
            Assert.notNull(msg, "sendMessage 消息体不能为NULL");
            // 处理消息
            process(msg);
            // 确认消息已经消费成功
            channel.basicAck(message.getMessageProperties().getDeliveryTag(), false);
        } catch (Throwable e) { // java.lang.NoClassDefFoundError: com/icetech/cloudcenter/domain/enumeration/ReportCmdEnum Consumer thread error, thread abort
            try {
                int count = StringUtils.getInt(message.getMessageProperties().getHeaders().get(Constant.tryCountKey));
                log.error("MQ消息处理异常，消息体:{}, 消息内容:{}， 失败第 {} 次 ", message.getMessageProperties().getCorrelationId(), JSON.toJSONString(msg), count + 1, e);
                if (count < tryCount) {
                    // 消息重新入队列
                    message.getMessageProperties().getHeaders().put(Constant.tryCountKey, ++count);
                    channel.basicAck(message.getMessageProperties().getDeliveryTag(), false);
                    rabbitSender.sendMessage("", consumerQueue, message);
                    return;
                }
                String className = (String) message.getMessageProperties().getHeaders().get(Constant.className);
                ErrorMsgBean errorMsgBean = new ErrorMsgBean(message.getMessageProperties().getCorrelationId(), mqName, consumerQueue, msg, !StringUtils.isEmpty
                        (className) ? className : msg.getClass().getName(), 2, e.getMessage());
                if (iMQExceptionData != null) {
                    // 个性化处理
                    iMQExceptionData.processException(consumerQueue, JSON.toJSONString(errorMsgBean));
                } else {
                    // 统一处理 异常信息发送到队列
                    rabbitSender.sendMessage("", Constant.COMMON_MSG_ERROR_QUEUE, errorMsgBean, 0L, Lists.newArrayList(Constant.commonMq));
                }
                // 确认消息已经消费成功
                channel.basicAck(message.getMessageProperties().getDeliveryTag(), false);
            } catch (Throwable dbe) {
                log.error("保存异常MQ消息到数据库异常，放到死性队列，消息体：{}", JSON.toJSONString(msg), dbe);
                // 确认消息将消息放到死信队列
                channel.basicNack(message.getMessageProperties().getDeliveryTag(), false, false);
            }
        } finally {
            MDC.remove(CommonConstants.TRACE_ID_KEY);
        }
    }

    /**
     * Description: 解析 map 对象
     *
     * @param msg
     * @param channel
     * @param message
     * @author wgy 2019-08-02 12:32:12
     * @return void
     */
    @RabbitHandler
    public void onMessageMsg(@Payload Object msg, Channel channel, Message message) throws Exception {
        T t;
        try {
            ResolvableType resolvableType;
            Class clazz = this.getClass();
            while (true) {
                if (clazz.getSuperclass().getName().equals(MessageListenerAbstract.class.getName())) {
                    resolvableType = ResolvableType.forClass(clazz);
                    break;
                }
                clazz = clazz.getSuperclass();
            }
            Class<?> resolve = resolvableType.getSuperType().getGeneric(0).resolve();
            if (resolve.getName().equals(msg.getClass().getName())) {
                t = (T) msg;
            } else if (msg instanceof String) {
                t = JSON.parseObject((String) msg, (Type) resolve);
            } else {
                t = JSON.parseObject(JSON.toJSONString(msg), (Type) resolve);
            }
        } catch (Exception e) {
            log.error("JSON 解析异常,{}", JSON.toJSON(msg), e);
            channel.basicAck(message.getMessageProperties().getDeliveryTag(), false);
            return;
        }
        onMessage(t, channel, message);
    }

    /**
     * 处理消息
     *
     * @param msg
     */
    protected abstract void process(T msg);

}
